//    
/*++
Cat::EuclideanDomain -- the category of euclidean domains

An Cat::EuclideanDomain is an principal ideal domain with an 'Euclidean degree'
function and operations "quo" and "rem" computing the Euclidean quotient
and the Euclidean reminder resp.

Methods:
euclideanDegree(x) - returns the Euclidean degree of x, a non-negative integer
quo(x,y)	   - returns the Euclidean quotient of x and y
rem(x,y)	   - returns the Euclidean reminder of x and y
divide(x,y)	   - returns the sequence quo(x,y),rem(x,y)
gcdex(x,y)	   - returns (g, v, w) such that g = gcd(x,y) = x*v + y*w

The euclideanDegree returns non-negative integers such that for each x, y <> 0

	x = quo(x,y) * y + rem(x,y)

and euclideanDegree(rem(x,y)) < euclideanDegree(y).

The gcd is computed by the Euclidean algorithm. The gcdex is computed by the
extended Euclidean algorithm.
++*/

category Cat::EuclideanDomain
    category Cat::PrincipalIdealDomain;

    euclideanDegree; divide;

    quo := proc(x,y) begin dom::divide(x,y)[1] end_proc;

    rem := proc(x,y) begin dom::divide(x,y)[2] end_proc;

    _divide := proc(x,y) begin
	x:= dom::divide(x,y);
	if dom::iszero(x[2]) then x[1] else FAIL end_if;
    end_proc;

    gcd := proc(x, y)
	local l, i, g;
    begin
	case args(0)
	of 2 do
	    x:= dom::unitNormal(x);
	    y:= dom::unitNormal(y);

	    while not dom::iszero(y) do
		dom::rem(x, y); x:= y; y:= %2
	    end_while;

	    dom::unitNormal(x);
	    break;
	
	of 1 do
	    dom::unitNormal(x);
	    break;
	    
	of 0 do 
	    dom::zero;
	    break;
	    
	otherwise
	    l:= sort([ args() ],
		     proc() begin 
			 bool(dom::euclideanDegree(args(1)) <
			      dom::euclideanDegree(args(2)))
		      end_proc );
	    g:= dom::gcd(l[1], l[2]);
	    for i from 3 to args(0) do
		g:= dom::gcd(g, l[i])
	    end_for;
	    g;
	end_case
    end_proc;

    gcdex := proc(xx, yy)
	local x, y, x1, x2, y1, y2, q, uc, ux, uy;
    begin
	x:= dom::unitNormalRep(xx); ux:= x[2]; x:= x[1];
	y:= dom::unitNormalRep(yy); uy:= y[2]; y:= y[1];

	x1:= dom::one; x2:= dom::zero;
	y1:= dom::zero; y2:= dom::one;

	while not dom::iszero(y) do
	    q:= dom::_negate(dom::quo(x, y));

	    dom::_plus(x, dom::_mult(q, y));
	    x:= y; y:= %2;
	    dom::_plus(x1, dom::_mult(q, y1));
	    x1:= y1; y1:= %2;
	    dom::_plus(x2, dom::_mult(q, y2));
	    x2:= y2; y2:= %2;
	end_while;

	x:= dom::unitNormalRep(x); uc:= x[2];

	( x[1], dom::_mult(x1, uc, ux), dom::_mult(x2, uc, uy) )
    end_proc;
    
    idealGenerator := dom::gcd;
    
begin


end_category:
