//    
/*++
Cat::IntegralDomain -- the category of integral domains

An Cat::IntegralDomain is a Cat::CommutativeRing and an Cat::EntireRing such that
x = _divide(a,b) and x <> FAIL implies that x is unique:

	i)  b <> 0 and _divide(a,b) <> FAIL ==> b * _divide(a,b) = a
	ii) a = b * c ==> _divide(a,b) <> FAIL

Use the axiom Ax::canonicalUnitNormal to state in addition that unitNormal(x)
returns a unique associate among the associate class of x, ie. that
	associates(x,y) implies unitNormal(x) = unitNormal(y)

Use the axiom Ax::closedUnitNormals in addition to state that unitNormals's are
closed under multiplication, ie. that
	x = unitNormal(a) * unitNormal(b) implies unitNormal(x) = x

These two axioms are not implizitly valid for an Cat::IntegralDomain!

Methods:-
_divide(x,y)     - returns a divisor d such that x = d*y if y <> 0 or FAIL;
	           if the result isn't FAIL then d is unique
divides(x,y)     - returns TRUE if x divides y
associates(x,y)  - returns TRUE iff x and y are associates
isUnit(x)	 - returns TRUE iff x is a unit
unitNormalRep(x) - returns a list [n,u,v] where n is an associate of x, u is a
		   unit and v is the inverse of u such that u*x = n and
		   u*v = 1; note that the associate returned is only unique
		   (in the associate class) if the axiom Ax::canonicalUnitNormal
		   states it
unitNormal(x)    - returns unitNormalRep(x)[1], ie. an associate of x
++*/

category Cat::IntegralDomain
    category Cat::EntireRing, Cat::CommutativeRing, Cat::Algebra(dom) ;

    _divide; isUnit;

    divides := proc(x,y) begin bool(dom::_divide(y,x) <> FAIL) end_proc;

    associates := proc(x,y) begin
	bool(dom::divides(x,y) and dom::divides(y,x))
    end_proc;

    unitNormal :=  if not dom::hasProp(Ax::canonicalUnitNormal) then
	id
    else
	toBeDefined
    end_if ;

    unitNormalRep := if not dom::hasProp(Ax::canonicalUnitNormal) then
	proc(x) begin [x,dom::one,dom::one] end_proc
    else
	proc(x) local n, u; begin
	    if dom::iszero(x) then
	        [ dom::zero, dom::one, dom::one ]
	    else
		n:= dom::unitNormal(x);
		u:= dom::_divide(n,x);
		[ n, u, dom::_divide(dom::one,u) ]
	    end_if
	end_proc
    end_if ;
    
begin

end_category:

