/*++ ---------------- OrePolyCat.mu ---------------------

Description:
This is the category for rings of univariate skew polynomials the so-called
Ore-polynomials. The coefficients of Ore polynomials may be in an
arbitrary commutative or non-commutative ring or field, while the 
multiplication of Ore-polynomials is restricted that the degree of a product
is equal to the sum of the degrees of the factors.

Call:

 Cat::UnivariateSkewPolynomial( Ring ) 
 where
    Ring  => arbitrary ring of the Domains-package


Methods:

 - used Parameter:
    ++ a,b   : this Category
    ++ a,b   = a, b are Ore-polynomials of this Category

 - _mult(a,b,...)
    ++ multiplies a,b,... by the non-commutative definition for multiplication
    ++ of Ore-polynomials.
 - _power(a, n)
    ++ computes the n-th power of a by the previous defined operation _mult.
 - leftDivide(a, b)
    ++ computes q, r, such that a=b*q+r with deg(r) < deg(b) and 
    ++ returns a table of quotient=q, remainder=r.
 - leftExactQuotient(a, b)
    ++ computes q, such that a=b*q if it exists else returns FAIL.
 - leftExtendedEuclid(a, b)
    ++ computes [[r0, s0, t0], [s1, t1]] such that
    ++ leftGcd(a,b) = r0 = a*s0+b*t0 and rightLcm(a,b) = -a*s1 = b*t1. 
 - leftExtendedGcd(a, b)
    ++ computes [r0, s0, t0] such that leftGcd(a,b) = r0 = a*s0+b*t0.
 - leftGcd(a,b)
    ++ computes the value g of highest degree such that a=g*a0 and b=g*b0
    ++ for some values a0 and b0.
 - leftLcm(a,b)
    ++ computes the value s of lowest degree such that s=a0*a=b0*b
    ++ for some values a0 and b0.
 - leftQuotient(a, b)
    ++ computes q such that a=b*q+r with deg(r)=0 or deg(r) < deg(b).
 - leftRemainder(a, b)
    ++ computes r such that a=b*q+r with deg(r)=0 or deg(r) < deg(b).
 - rightDivide(a, b)
    ++ computes q, r, such that a=q*b+r with deg(r) < deg(b) and 
    ++ returns a table of quotient=q, remainder=r.
 - rightExactQuotient(a, b)
    ++ computes q, such that a=q*b if it exists else returns FAIL.
 - rightExtendedEuclid(a, b)
    ++ computes [[r0, s0, t0], [s1, t1]] such that
    ++ rightGcd(a,b) = r0 = s0*a+t0*b and leftLcm(a,b) = -s1*a = t1*b. 
 - rightExtendedGcd(a, b)
    ++ computes [r0, s0, t0] such that rightGcd(a,b) = r0 = s0*a+t0*b.
 - rightGcd(a,b)
    ++ computes the value g of highest degree such that a=a0*g and b=b0*g
    ++ for some values a0 and b0.
 - rightLcm(a,b)
    ++ computes the value s of lowest degree such that s=a*a0=b*b0
    ++ for some values a0 and b0.
 - rightQuotient(a, b)
    ++ computes q such that a=q*b+r with deg(r)=0 or deg(r) < deg(b).
 - rightRemainder(a, b)
    ++ computes r such that a=q*b+r with deg(r)=0 or deg(r) < deg(b).
 

See: 
 -  M. Bronstein & M. Petkovsek (1993). On Ore Rings, Linear Operators 
    and Factorisation. Bericht Nr. 200  ETH Zuerich 
 -  M. Bronstein & M. Petkovsek (1996). An introduction to pseudo-linear
    algebra. Theoretical Computer Science 157 No. 1, 3-33.
 -  O. Ore (1933). Theory of non-commutative polynomials. Annals of
    Mathematics 34, 480-508.

Example:
>> EF:=Dom::ExpressionField(normal);

                Dom::ExpressionField(normal, iszero@normal)

>> OC:=Cat::UnivariateSkewPolynomial(EF);

Cat::UnivariateSkewPolynomial(Dom::ExpressionField(normal, iszero@normal))


++*/


/* Dependencies:

 */

category Cat::UnivariateSkewPolynomial( R = Dom::ExpressionField(normal) )

    local hasField ;

    category Cat::Ring, Cat::LeftModule(R), Cat::RightModule(R) ;

    axiom    Ax::normalRep, 
             if R::hasProp(Ax::canonicalRep) then Ax::canonicalRep end_if;

  // entries: 

  _plus; ore_mult; _mult; _power; degree; lcoeff; reductum; 
  coeff; zero;
  //one; // temporarily disabled to avoid overwriting the one from
         // Cat::AlgebraWithBasis
   iszero; indets; mainvar;

  coeffRing := R;

  characteristic := R::characteristic();

//  --- the following functions needs to be implemented in a domain 
//      (It is necessary that 'info' will work correct)             

  leftDivide :=
   if hasField then
      proc() begin  error("not implemented")  end_proc  end_if;

  rightDivide :=
   if hasField then
      proc() begin  error("not implemented")  end_proc  end_if;

//  --- 

  leftQuotient := 
   if hasField then
      proc() 
        begin dom::leftDivide(args(1),args(2))[hold(quotient)]  
        end_proc  
   end_if;

  leftRemainder := 
   if hasField then
      proc() 
        begin  dom::leftDivide(args(1),args(2))[hold(remainder)]
        end_proc 
   end_if;

  leftExactQuotient :=
   if hasField then
     proc() 
       begin 
         dom::leftDivide(args(1), args(2));
         if iszero(%[hold(remainder)]) then %[hold(quotient)] else FAIL end_if 
      end_proc 
   end_if;

  leftGcd := 
   if hasField then
      proc(a:dom, b:dom) local r;
        begin
          if dom::iszero(a) then return(b) end_if;
          if dom::iszero(b) then return(a) end_if;
          while degree(b) > 0 do
            r := dom::leftRemainder(a, b);
            a := b;
            b := r
          end_while;
          if dom::iszero(b) then a else b end_if
        end_proc
      end_if ;

  leftExtendedEuclid :=        // lgcd(a,b) = r0 = a*s0+b*t0  
   if hasField then               // lcrm(a,b) = -a*s1 = b*t1    
      proc(a:dom, b:dom) 
        local s0, s1, t0, t1, r0, r1, q, h;
        begin
          r0 := a;         r1 := b;
          s0 := dom::one;  s1 := dom::zero;
          t0 := dom::zero; t1 := dom::one;
          while not iszero(r1) do
            q  := dom::leftDivide(r0, r1);
            r0 := r1; r1 := q[hold(remainder)]; 
            h  := s0; s0 := s1; s1 := h - dom::ore_mult(s1, q[hold(quotient)]);
            h  := t0; t0 := t1; t1 := h - dom::ore_mult(t1, q[hold(quotient)])

          end_while;
        [[r0, s0, t0], [s1, t1]]
      end_proc
    end_if ;

  leftExtendedGcd := 
    if hasField then 
      proc() begin dom::leftExtendedEuclid(args())[1] end_proc
    end_if;

  leftLcm := 
   if hasField then
      proc(a, b)
        begin
         dom::mult( dom::rightExtendedEuclid(a, b)[2][2], b) 
        end_proc end_if; 

  rightQuotient := 
   if hasField then
      proc() 
        begin dom::rightDivide(args(1), args(2))[hold(quotient)]
        end_proc  
   end_if;

  rightRemainder := 
   if hasField then
      proc() 
        begin dom::rightDivide(args(1),args(2))[hold(remainder)]
        end_proc  
   end_if;

  rightExactQuotient :=
   if hasField then
     proc() 
       begin
         dom::rightDivide(args(1), args(2));
         if iszero(%[hold(remainder)]) then %[hold(quotient)] else FAIL end_if 
       end_proc 
   end_if;

  rightGcd := 
   if hasField then
      proc(a:dom, b:dom) local r;
        begin
          if dom::iszero(a) then return(b) end_if;
          if dom::iszero(b) then return(a) end_if;
          while degree(b) > 0 do
            r := dom::rightRemainder(a, b);
            a := b;
            b := r
          end_while;
          if dom::iszero(b) then a else b end_if
        end_proc
      end_if ;

  rightExtendedEuclid :=       // rgcd(a,b) = r0 = s0*a+t0*b  
   if hasField then               // lclm(a,b) = -s1*a = t1*b    
      proc(a:dom, b:dom) 
        local s0, s1, t0, t1, r0, r1, q, h;
        begin
          r0 := a;         r1 := b;
          s0 := dom::one;  s1 := dom::zero;
          t0 := dom::zero; t1 := dom::one;
          while not iszero(r1) do
            q  := dom::rightDivide(r0, r1);
            r0 := r1; r1 := q[hold(remainder)]; 
            h  := s0; s0 := s1; s1 := h - dom::ore_mult(q[hold(quotient)], s1);
            h  := t0; t0 := t1; t1 := h - dom::ore_mult(q[hold(quotient)], t1)
          end_while;
        [[r0, s0, t0], [s1, t1]]
      end_proc
    end_if;

  rightExtendedGcd :=
    if hasField then 
       proc() begin dom::rightExtendedEuclid(args())[1] end_proc
    end_if;

  rightLcm := 
   if hasField then
      proc(a, b)
        begin
         dom::mult(b, dom::leftExtendedEuclid(a, b)[2][2]) 
        end_proc end_if;

begin
  // initialization: 
    if args(0) <> 1 then 
        error("wrong no of args")
    end_if;
    hasField := bool(R::hasProp(Cat::Field)); 
end_category:



