//   

/*++


Cat::Set - the category of sets


Sets have the following set-theoretic operations:

union, intersection, difference. They may also have algebraic
operations (_plus, _mult).


The associative operators are, by default, realized as follows:

first, the arguments are sorted by their type. Types which do not
belong to Cat::Set are handled separately; concerning the others,
all arguments from the same domain are passed to the method homog.<operator>
of the domain. The results are then processed in a ''round-robin'' manner:
starting from the leftmost, each one is paired with all sets on the right of
it until a simplification occurs. If this happens, both sets that could be
combined are deleted from the list of operands, and the result is stored
in a separate list. Finally, operands that were obtained by simplification
are then paired again: first to each other, then to the ''old'' operands;
this continues until no new simplifications occur. In the end,
hold(<operator>)(<all remaining operands>) is returned.

Pairing of two operands of different types is done as follows: the entry
inhomog.<operator> is searched in the domain of the first, then in the
domain of the second operand. (That entry must contain a table of binary
operators, indexed by the type of the other argument.)
If this fails, it is tested (using arg::whichEntry(<operator>))  whether
one of the domains of the operands defines the operator itself; in this case,
that method is used. If no method is found at all, FAIL is returned.

Binary operators are realized, by default, as follows: if both
operands are of the same type, slot "bin".operatorname is used;
otherwise slot "inhomogleft".operatorname of the left operand; if this
also fails, "inhomogright".operatorname of the right operand. The order
of operands in never changed.

++*/


category Cat::Set

  category Cat::BaseCategory;

    // no axioms

    // necessary entries


    /*

       commassop(operatorname) - generate a method
       that can be used as slot(dom, operatorname)

       operatorname must be a string such that text2expr
       is an associative and commutative operator

    */


  commassop:=
    proc(operatorname)
      option escape;
      local Operator;
    begin
      Operator:=text2expr(operatorname);

      // return value:
      proc()
        local catset, others, i,j, results, res, types, operands, a, b, sl;
      begin
        if args(0)=1 then
          return(args(1))
        end_if;
        catset:=split([args()], x -> x::dom::hasProp(Cat::Set) <> TRUE);
        others:=op(catset,1);
        catset:=op(catset,2);

        // handle arguments of other types separately

        case nops(others)
          of 0 do
          of 1 do
            break;
          otherwise
            others:=eval(Operator)(op(others));
            if type(others)=operatorname then
              others:= [op(others)]
            else
              others:=[others]
            end_if;
        end_case;

        // sort arguments by type
        catset:=sort(catset, (x,y) -> sysorder(type(x), type(y)));
        types:= map(catset, type);

        // try to combine operands of the same type first

        results:=[];

        for i in {op(types)} do
          operands:=select(catset, x -> type(x)=i);
          if nops(operands)=1 then
            results:=append(results, op(operands,1))
          else
            sl:= slot(i, "homog".operatorname)(op(operands));
            if sl = FAIL then
              results:=results.operands
            elif type(sl) = operatorname then
              results:=append(results, op(sl))
            else
              results:=append(results, sl)
            end_if
          end_if
        end_for;

        userinfo(20, "Results of homogenous operation: ".expr2text(results));

        operands:= results.others;
        results:=[];

        // round - robin

        i:=1;
        while i<nops(operands) do
          j:=i+1;
          while j<=nops(operands) do
            a:=operands[i];
            b:=operands[j];
            userinfo(20, "Trying binary operation for operands ".
                     expr2text(a)." and ". expr2text(b));
            if type(a) = type(b) then
              // go to next iteration, since operands
              // of identical types have already been handled
              res:=slot(a::dom, "bin".operatorname);
              if res<>FAIL then
                res:=res(a,b)
              else
                userinfo(30, "Slot bin".operatorname." does not exist in ".
                         expr2text(a::dom))
              end_if
            else
              // entry exists ?
              res:=slot(a::dom,"inhomog".operatorname);
              // entry knows how to simplify the expression ?
              if res<>FAIL then
                sl:= res[b::dom];
                if type(sl)<>"_index" then
                  res:=sl(a,b)
                else
                  sl:= res[expr2text(b::dom)];
                  if type(sl)<>"_index" then
                    res:=sl(a,b)
                  else
                    sl:= res[b::dom::constructor];
                    if type(sl)<>"_index" then
                      res:=sl(a,b)
                    else
                      res:=FAIL
                    end_if
                  end_if;
                end_if
              else
                userinfo(30, "Slot inhomog".operatorname.
                         " does not exist in ".expr2text(a::dom))
              end_if;
              // if this failed for some reason, try
              // the right operand
              if res=FAIL then
                res:=slot(b::dom,"inhomog".operatorname);
                if res<>FAIL then
                  sl:= res[domtype(a)];
                  if type(sl)<>"_index" then
                    res:=sl(b,a)
                  else
                    sl:= res[expr2text(domtype(a))];
                    if type(sl)<>"_index" then
                      res:=sl(b,a)
                    else
                      sl:= res[a::dom::constructor];
                      if type(sl)<>"_index" then
                        res:=sl(b,a)
                      else
                        res:=FAIL
                      end_if
                    end_if;
                  end_if;
                else
                  userinfo(30, "Slot inhomog".operatorname.
                         " does not exist in ".expr2text(b::dom))
                end_if;
              end_if;
            end_if; // type(a)=type(b)
            if res=FAIL then
              userinfo(10, "Simplification failed for ".
                expr2text(a)." and ".expr2text(b));
              j:=j+1
            else
              if type(res)=operatorname then
                case {a, b} minus {op(res)}
                  of {} do
                    // no simplification
                    userinfo(10, "Simplification failed for ".
                    expr2text(a)." and ".expr2text(b));
                    j:=j+1;
                    break
                  of {b} do
                    // only the second operand has vanished
                    // we keep the first
                    // j does not need to be increased as
                    // the former operands[j+1] now becomes operands[j]
                    delete operands[j];
                    results:=append(results, op({op(res)} minus {a}));
                    break
                  of {a} do
                    // only the first operand has vanished
                    // we keep the second
                    // operands[i+1] becomes operands[i]
                    // we start the next iteration of the outer while-loop
                    delete operands[i];
                    j:= i+1;
                    results:=append(results, op({op(res)} minus {b}));
                    break
                  of {a, b} do
                    // both old operands have vanished and
                    // have to be deleted
                    results:=append(results, op(res));
                    // since j>i, the first delete does not affect
                    // the second
                    delete operands[j];
                    delete operands[i];
                    j:=i+1;
                    break
                  otherwise
                    // cannot happen
                    assert(FALSE)
                end_case;
              else
                results:=append(results, res);
                delete operands[j];
                delete operands[i];
                /* by delete, the former operands[i+1]
                   has become operands[i] now, so the next
                   iteration of the outer while-loop is started
                   implicitly */
                j:=i+1
              end_if;
            end_if;
          end_while; /* inner while loop over j */
          i:=i+1
        end_while; /* outer while loop */

        // try to combine operands until no new results appear
        if results=[] then
          results:=operands
        else
          // simplification occurred: recursive call
          results:=eval(Operator)(op(results), op(operands));
          return(results)
        end_if;

        if nops(results)=0 then
          Operator()
        elif nops(results)=1 then
          op(results, 1)
        else
          // force sorting by sysorder
          subsop(hold(id)(op(sort(results))), 0 = Operator, Unsimplified)
        end_if
      end_proc

      /* end of return value */

    end_proc;


    /* binop - generate a binary operator that can be used as
               slot(dom, operatorname)
     */

    binop:=
    proc(operatorname)
      option escape;
      local Operator;

    begin
      Operator:= text2expr(operatorname);

      /* return value */

      proc(a, b)
        local res, sl;

      begin
        if args(0) <> 2 then
          error("Wrong number of arguments")
        end_if;
        if type(a)=type(b) then
          userinfo(20, "Using bin".operatorname);
          res:=slot(a::dom, "bin".operatorname)(a,b);
          if res=FAIL then
            userinfo(30, "Binary operator failed");
            subsop(hold(id)(a, b), 0 = Operator, Unsimplified)
          else
            res
          end_if;
        else
          res:=slot(a::dom,"inhomogleft".operatorname);
          if res<>FAIL then
            sl:= res[b::dom];
            if type(sl)<>"_index" then
              sl:=sl(a,b)
            else
              sl:=FAIL
            end_if;
            if sl=FAIL then
              sl:= res[b::dom::constructor];
              if type(sl)<>"_index" then
                sl:=sl(a,b)
              else
                sl:=FAIL
              end_if;
              if sl<> FAIL then
                return(sl)
              end_if
            else
              return(sl)
            end_if;
          end_if;
          userinfo(30, "Left overloading failed");
          res:=slot(b::dom,"inhomogright".operatorname);
          if res<>FAIL then
            sl:= res[a::dom];
            if type(sl)<>"_index" then
              sl:=sl(a,b)
            else
              sl:=FAIL
            end_if;
            if sl=FAIL then
              sl:= res[a::dom::constructor];
              if type(sl)<>"_index" then
                sl:=sl(a,b)
              else
                sl:=FAIL
              end_if;
              if sl<> FAIL then
                return(sl)
              end_if
            else
              return(sl)
            end_if;
          end_if;
          userinfo(30, "All methods ".operatorname." failed for ".expr2text(a).
                   " and ".expr2text(b));
          subsop(hold(id)(a, b), 0 = Operator, Unsimplified)
        end_if;
      end_proc  // return value
    end_proc;

    commbinop:=
    proc(operatorname)
      option escape;
      local Operator;

    begin
      Operator:= text2expr(operatorname);

      /* return value */

      proc(a, b)
        local res, sl;

      begin
        if args(0) <> 2 then
          error("Wrong number of arguments")
        end_if;
        if type(a)=type(b) then
          userinfo(20, "Using bin".operatorname);
          res:=slot(a::dom, "bin".operatorname)(a,b);
          if res=FAIL then
            userinfo(30, "Binary operator failed");
            subsop(hold(id)(a, b), 0 = Operator, Unsimplified)
          else
            res
          end_if;
        else
          res:=slot(a::dom,"inhomog".operatorname);
          if res<>FAIL then
            sl:= res[b::dom];
            if type(sl)<>"_index" then
              sl:=sl(a,b)
            else
              sl:=FAIL
            end_if;
            if sl=FAIL then
              sl:= res[expr2text(b::dom)];
              if type(sl)<>"_index" then
                sl:=sl(a,b)
              else
                sl:=FAIL
              end_if;
              if sl=FAIL then
                sl:= res[b::dom::constructor];
                if type(sl)<>"_index" then
                  sl:=sl(a,b)
                else
                  sl:=FAIL
                end_if;
              end_if;
              if sl<> FAIL then
                return(sl)
              end_if
            else
              return(sl)
            end_if;
          end_if;
          userinfo(30, "Left overloading failed");
          res:=slot(b::dom,"inhomog".operatorname);
          if res<>FAIL then
            sl:= res[a::dom];
            if type(sl)<>"_index" then
              sl:=sl(a,b)
            else
              sl:=FAIL
            end_if;
            if sl=FAIL then
              sl:= res[expr2text(a::dom)];
              if type(sl)<>"_index" then
                sl:=sl(a,b)
              else
                sl:= res[a::dom::constructor];
                if type(sl)<>"_index" then
                  sl:=sl(a,b)
                else
                  sl:=FAIL
                end_if;
              end_if;
              if sl<> FAIL then
                return(sl)
              end_if
            else
              return(sl)
            end_if;
          end_if;
          userinfo(30, "All methods ".operatorname." failed for ".expr2text(a).
                   " and ".expr2text(b));
          subsop(hold(id)(a, b), 0 = Operator, Unsimplified)
        end_if;
      end_proc  // return value
    end_proc;





    // homogassop - generate homogeneous associative operator
    /* works for domains where corresponding binary operator is defined */
    // for satisfactory performance, it is necessary that the operation
    // always returns a single object of type dom

    homogassop:=
    proc(operatorname, zeroelement)
      option escape;
      local binoperator, Operator, f, d, X;

    begin
      d:=dom; // necessary for access in local methods
      Operator := text2expr(operatorname);

      if (binoperator:=slot(dom, "bin".operatorname)) <> FAIL then

        /* return value */
        // must have a name in order to allow recursive calls

        f:=
        proc()

        begin
          case args(0)
            of 0 do
              return(zeroelement)
            of 1 do
              return(args(1))
            of 2 do
              return(binoperator(args()))
            otherwise
              X:= [f(args(1..args(0) div 2)),
                   f(args((args(0) div 2) +1..args(0)))];
              if map(X, type)= [d, d] then
                binoperator(op(X))
              else
                Operator(op(X))
              end_if
          end_case
        end_proc

      else
        FAIL
      end_if
    end_proc;


    /* n-ary operators */


    _union := dom::commassop("_union");

    _intersect := dom::commassop("_intersect");

    _plus:= dom::commassop("_plus");

    _mult:= dom::commassop("_mult");


    /* binary operators */

    _minus:= dom::binop("_minus");

    bin_minus:= (x,y) -> if x=y then {} else FAIL end_if;

    _power:= dom::binop("_power");


    _equal:= dom::commbinop("_equal");

    equal:=
    proc(S, T)
      local result;
    begin
      if S = T then
        TRUE
      end_if;
      result:= S::dom::_equal(S, T);
      if result = FAIL then
        result:= T::dom::_equal(T, S)
      end_if;
      if result = TRUE or result = FALSE then
        return(result)
      else
        return(UNKNOWN)
      end_if
    end_proc;


    /* unary operators */


    // default map

    map:=
    proc(S, f)
      local newvar;
    begin
      newvar:=genident();
      Dom::ImageSet(f(%), %, S)
    end_proc;

    _negate:= if dom::map<>FAIL then S -> dom::map(S, _negate) end_if;
    _invert:= if dom::map<>FAIL then S -> dom::map(S, _invert) end_if;
    sin:= if dom::map<>FAIL then S -> dom::map(S, sin) end_if;
    cos:= if dom::map<>FAIL then S -> dom::map(S, cos) end_if;
    tan:= if dom::map<>FAIL then S -> dom::map(S, tan) end_if;
    cot:= if dom::map<>FAIL then S -> dom::map(S, cot) end_if;
    sinh:= if dom::map<>FAIL then S -> dom::map(S, sinh) end_if;
    cosh:= if dom::map<>FAIL then S -> dom::map(S, cosh) end_if;
    tanh:= if dom::map<>FAIL then S -> dom::map(S, tanh) end_if;
    coth:= if dom::map<>FAIL then S -> dom::map(S, coth) end_if;
    sec:= if dom::map<>FAIL then S -> dom::map(S, sec) end_if;
    sech:= if dom::map<>FAIL then S -> dom::map(S, sech) end_if;
    csc:=  if dom::map<>FAIL then S -> dom::map(S, csc) end_if;
    csch:= if dom::map<>FAIL then S -> dom::map(S, csch) end_if;

    exp:= if dom::map<>FAIL then S -> dom::map(S, exp) end_if;
    ln := if dom::map<>FAIL then S -> dom::map(S, ln) end_if;

    arcsin:= if dom::map<>FAIL then S -> dom::map(S, arcsin) end_if;
    arccos:= if dom::map<>FAIL then S -> dom::map(S, arccos) end_if;
    arctan:= if dom::map<>FAIL then S -> dom::map(S, arctan) end_if;
    arccot:= if dom::map<>FAIL then S -> dom::map(S, arccot) end_if;
    arcsinh:= if dom::map<>FAIL then S -> dom::map(S, arcsinh) end_if;
    arccosh:= if dom::map<>FAIL then S -> dom::map(S, arccosh) end_if;
    arctanh:= if dom::map<>FAIL then S -> dom::map(S, arctanh) end_if;
    arccoth:= if dom::map<>FAIL then S -> dom::map(S, arccoth) end_if;
    arcsec:= if dom::map<>FAIL then S -> dom::map(S, arcsec) end_if;
    arcsech:= if dom::map<>FAIL then S -> dom::map(S, arcsech) end_if;
    arccsc:= if dom::map<>FAIL then S -> dom::map(S, arccsc) end_if;
    arccsch:= if dom::map<>FAIL then S -> dom::map(S, arccsch) end_if;

    sign:= if dom::map<>FAIL then S -> dom::map(S, sign) end_if;
    abs:= if dom::map<>FAIL then S -> dom::map(S, abs) end_if;
    floor:= if dom::map<>FAIL then S -> dom::map(S, floor) end_if;
    round:= if dom::map<>FAIL then S -> dom::map(S, round, args(2..args(0))) end_if;
    ceil:= if dom::map<>FAIL then S -> dom::map(S, ceil) end_if;
    trunc:= if dom::map<>FAIL then S -> dom::map(S, trunc) end_if;

    // testtype
    testtype:=
    proc(e, T)
    begin
      // domtype(e) = T is checked by the kernel
      if T = Type::Set then 
        TRUE 
      else
        if domtype(T)= DomainConstructor then
          return(bool(e::dom::hasProp(T) = TRUE))
        end_if;

        // an object is regarded as an element of a domain T with system
        // representation iff it can be converted
        if domtype(T) = DOM_DOMAIN and T::hasProp(Ax::systemRep) = TRUE then
          return(bool(coerce(e, T) <> FAIL))
        end_if;
    
        FAIL

      end_if
    end_proc;

end_category():
