/*   
*/
/*++
repeatedSquaring -- create procedure implementing 'repeatedSquaring'

repeatedSquaring(D, o [,u [, inv]])

D   - domain
o   - name of basis operation
u   - name of neutral element according to o (if exists)
inv - name of inverting opertaion

repeatedSquaring returns a procedure p(x,i) which calculates 'x o x o
... o x (i times)' for domain elements 'x' and positive integers 'i'
where 'x' is in the domain 'D' and 'o' is an operation in 'D'. If 'D'
has an neutral element according to 'o', its name may be given by 'u'.
In this case the "power" 'i' may be zero, p(x,0) returns 'u'.
If D additionally has an inversion 'inv' for 'o' the exponents 'i' may
be negative; for negative exponents the inverse of 'x' is squared.

The operation 'o' of 'D' must be associative.

If, for example, o is "_mult" and u is "one" then repeatedSquaring
returns a procedure p to compute x^i for integers i >= 0.  p calculates
the result by repeated squaring.
++*/

domains::repeatedSquaring:=
proc(DOM: DOM_DOMAIN, oper: DOM_STRING,
     unit="": DOM_STRING, inv_mthd="": DOM_STRING)
  option escape;
  
begin
    if args(0) < 2 or args(0) > 4 then error("wrong no of args") end_if;
    // dont test oper, unit and inv_mthd to prevent creation of entries 
    
    case args(0)
    of 4 do
	// create 'repeated squaring' using neutral element and inversion 

	/* proc(x,n) -- returns 'x oper x .... x (n times)' if n > 0,
	  'unit' if n = 0 and 'y oper y .... y (-n times)' if n < 0
	  and y = inv_mthd(x) */
	proc(xx, nn: DOM_INT)
	    local x, n, p;
	begin
	    x:=xx; // to avoid a warning 
	    n:=nn;
	    if n < 0 then
		x:= slot(DOM, inv_mthd)(x);
		if x = FAIL then return(FAIL) end_if;
		n:= -n;
	    end_if;

	    // repeated squaring 
	    p:= slot(DOM, unit);
	    while TRUE do
		if n mod 2 = 1 then
		    p:= slot(DOM, oper)(p, x)
		end_if;
		if n < 2 then break end_if;
		n:= n div 2;
		x:= slot(DOM, oper)(x, x)
	    end_while;
	    return(p)
	end_proc;
	break;

    of 3 do
	// create 'repeated squaring' using neutral element 

	/* proc(x,n) -- returns 'x oper x .... x (n times)' if n > 0
	  and 'unit' if n = 0 */
	proc(xx, nn: DOM_INT)
	    local x, n, p;
	begin
	    x:=xx;
	    n:=nn;
	    if n < 0 then error("negative multiple") end_if;
	    // repeated squaring 
	    p:= slot(DOM, unit);
	    while TRUE do
		if n mod 2 = 1 then
		    p:= slot(DOM, oper)(p, x)
		end_if;
		if n < 2 then break end_if;
		n:= n div 2;
		x:= slot(DOM, oper)(x, x)
	    end_while;
	    return(p)
	end_proc;
	break;

    otherwise
	// create 'repeated squaring' without neutral element 

	// proc(x,n) -- returns 'x oper x .... x (n times)' for n > 0 
	proc(xx,nn)
	    local p, x, n;
	begin
	    x:=xx;
            n:=nn;
	    if n <= 0 then error("not a positive multiple") end_if;
	    if n mod 2 = 1 then
		p:= x;
		n:= n-1
	    else
		p:= slot(DOM, oper)(x, x);
		n:= n-2
	    end_if;
	    // repeated squaring 
	    while TRUE do
		if n mod 2 = 1 then
		    p:= slot(DOM, oper)(p, x)
		end_if;
		if n < 2 then break end_if;
		n:= n div 2;
		x:= slot(DOM, oper)(x, x)
	    end_while;
	    return(p)
	end_proc;
    end_case
end_proc:

// end of file 
