//    
/*++
Category -- the domain of categories

A category is represented as element of the domain 'Category' with
operands:

 0 - the domain 'Category'
 1 - the category constructor, an element of the domain
     'CategoryConstructor'
 2 - the closure of the category

The implementation of the domain 'CategoryConstructor' strongly depends 
on the representation of categories defined here (because of performance
reasons). Do not change!

A category has no entries which may be used as domain entries
directly.  Such entries are created on the fly by the category
constructor.  A category merely stores the closure of the catgory.

None of the methods defined here are normally called directly by the
category implementor or user. They are called implizitly.

Only the method 'slot' may be used within the definition of a
category or domain constructor in order to obtain certain entries of
categories directly. This method may not be used in any other context!
++*/

Category:= newDomain("Category"):
Category::info:= "Domain 'Category'":
Category::interface:= {}:
Category::create_dom:= hold(Category) :
Category::subs:= proc(x) begin x end_proc :

/*++
constructor -- return constructor of Category

constructor(C)

C - category
++*/

Category::constructor:= proc(C: Category): CategoryConstructor
    option noDebug;
begin
    extop(C, 1)
end_proc:

/*++
slot -- get entry of category, changing it into domain entry

slot(C, e)

C - category
e - name of entry to get

'slot' is used to get the entry 'e' of category 'C' directly. The
entry is implicitly changed into a domain entry (by 'getDirectEntry').

'slot' may only be used within the definition of category contructors
with 'CategoryConstructor::new'. It is not allowed in any other
context!
++*/

Category::slot:= proc(C: Category, e: DOM_STRING)
    option noDebug;
begin
    if args(0) = 3 then error("can't assign to Category slot") end_if;
    CategoryConstructor::getNamedEntry(C, e)
end_proc:

/*++
new -- create new category using category constructor

new(CC,c)

CC - category constructor
c  - closure (a procedure environment)
++*/

Category::new:= proc(CC: CategoryConstructor, clos: DOM_PROC_ENV): Category
    option noDebug;
begin
    new(Category, CC, clos)
end_proc:

/*++
print -- print category

print(C)

C - category
++*/

Category::print:= proc(Ca: Category)
    option noDebug;
    local p, nam, C;
begin
    C:= extop(Ca, 2);
    p:= op(C, 1);
    nam:= op(p, 6);
    if op(p,1) = NIL then
	nam
    else
	p:= op(C,[0,2]);
	subsop(hold(sin)(op(C, p..(p+op(C,[0,3])-1))), 0=nam)
    end
end_proc:

Category::create_dom_elem:= Category::print:

/*++
TeX -- return TeX formatted string for category

TeX(C)

C - category
++*/

Category::TeX:= proc(C: Category): DOM_STRING 
    option noDebug;
begin 
    generate::TeX(Category::print(C))
end_proc:

/*++
testtype -- test type of category

testtype(e, T)

e - expression
T - type

Returns TRUE if e is an Category and T is Category and FAIL otherwise.
++*/

Category::testtype:=
  proc(e, T)
  name Category::testtype;
  option noDebug;
begin
  if T = Category and domtype(e) = Category then
    return(TRUE)
  else
    if domtype(T) = Category and domtype(e) = DOM_DOMAIN and
                      e::categories <> FAIL and e::hasProp <> FAIL and e::hasProp(T) then
      return(TRUE)
    end_if
  end_if;
  FAIL;
end_proc:

           /*
Category::testtype:= proc(e, T)
    option noDebug;
begin
    if T = Category then
        if domtype(e) = Category then
            return(TRUE)
        end_if
    end_if;
    FAIL
end_proc:
*/

/*++
_index -- get definition of entry from constructor

_index(C, e)

C - category
e - entry name (string)

_index returns the unevaluated definition of the entry e of C's constructor
or FAIL if e is not directly defined for the constructor of C.
++*/

Category::_index:= proc(C: Category, e: DOM_STRING)
    option noDebug;
begin
    CategoryConstructor::_index(extop(C,1), e)
end_proc:

/*++
printInfo -- print information about category

printInfo(C)

C - category
++*/

Category::printInfo:= proc(C: Category): DOM_NULL
    option noDebug;
    save   PRETTYPRINT;
begin
    PRETTYPRINT:= FALSE;
    print(Unquoted, expr2text(C)." -- category constructed by ".
    		    expr2text(extop(C,1)))
end_proc:

// end of file 
