//    

/*++
Dom::AlgebraicExtension -- domain constructor for simple algebraic extension

Dom::AlgebraicExtension(F, p [,a])

F - field
p - irreducible polynomial or expression defining adjoined element
a - name of the adjoined element (identifier)

Dom::AlgebraicExtension(F,p,a) creates a simple algebraic extension F(a) where a
are the roots of an irreducible polynomial defined by p. p must either be an
equation, a polynomial expression or a DOM_POLY.

- If p is a polynomial expression or a DOM_POLY, it is converted into a
  univariate polynomial over F in the variable a. If a is not given then p
  must have exactly one unknown which then takes the role of a.

- If p is an equation x=y, it is changed into the expression x-y. x-y is then
  used as defining polynomial, such that the same rules as above apply.

It is not tested whether p is irreducible or not!

Methods:-
conjNorm(x) - returns the norm of x, ie. the product of all conjugates of x
conjTrace(x) - returns the trace of x, i.e. the sum of all conjugates of x

++*/

domain Dom::AlgebraicExtension(F , PExpr, Alias)
  local P;
  inherits Dom::BaseDomain;
  category Cat::Field, Cat::Algebra(F), Cat::VectorSpace(F),
    (if F::hasProp(Cat::DifferentialRing) then
       Cat::DifferentialRing
     end_if),
    (if F::hasProp(Cat::PartialDifferentialRing) then
       Cat::PartialDifferentialRing
     end_if);
  axiom (if F::hasProp(Ax::canonicalRep)
           then Ax::canonicalRep else Ax::normalRep end_if),
        (if F::hasProp(Ax::efficientOperation("_invert"))
	   then Ax::efficientOperation("_invert") else null() end_if);

    groundField:= F;

    minpoly:= P;

    deg:= degree(P);

    variable:= Alias;

    characteristic:= F::characteristic;

    sign:=e->hold(sign)(e);

    degreeOverPrimeField:=
    (if F::hasProp(Dom::AlgebraicExtension) or
       F::hasProp(Dom::GaloisField) then
       if F::degreeOverPrimeField=UNKNOWN then
         UNKNOWN
       else
         degree(P)*F::degreeOverPrimeField
       end_if
     elif F::hasProp(Dom::IntegerMod) or F::hasProp(Dom::Rational) then
       degree(P)
     else
       UNKNOWN;
     end_if );

    
    zero:= new(dom, P-P);

    one:= new(dom, P^0);

    _plus:=
    proc()
      local l, i;
    begin
      l:= map([args()],dom::coerce);
      if (i:= contains(l, FAIL)) = 0 then
        return(new(dom, _plus(op(map(l, extop, 1)))))
      end_if;
      l:= map([args()], (args(i))::dom::coerce);
      if contains(l, FAIL) = 0 then
        _plus(op(l))
      else
        FAIL
      end_if
    end_proc;

    _negate:= (()->(new(dom, -extop(args(1),1))));

    _subtract:= (()->((map([args()],dom::coerce);
                       new(dom, _subtract(op(map(%,extop,1)))) )));

    iszero:= (()->(iszero(extop(args(1), 1))));

    _mult:=
    if F::hasProp(Ax::systemRep) then
      proc(a, b)
        local c;
      begin
        if args(0) = 2 then
          if domtype(b) <> dom then
            if domtype(b) = DOM_INT then
              dom::intmult(args())
            elif (c:= F::convert(b)) <> FAIL then
              new(dom, multcoeffs(extop(a,1), c))
            else
              b::dom::_mult(args())
            end_if
          elif domtype(a) <> dom then
            if domtype(a) = DOM_INT then
              dom::intmult(b, a)
            elif (c:= F::convert(a)) <> FAIL then
              new(dom, multcoeffs(extop(b,1), c))
            else
              FAIL
            end_if
          else
            new(dom, divide(extop(a,1) * extop(b,1), P, Rem))
          end_if
        elif args(0) = 1 then
          a
        else
          _mult(args(1..(args(0) div 2)));
          _mult(args(((args(0) div 2) + 1)..args(0)));
          if (%2)::dom::_mult <> FAIL then
            (%2)::dom::_mult(%2, %1)
          elif (%1)::dom::_mult <> FAIL then
            (%1)::dom::_mult(%2, %1)
          else
            _mult(%2, %1)
          end_if
        end_if
      end_proc
    else
      proc(a, b)
        local c;
      begin
        if args(0) = 2 then
          if domtype(b) <> dom then
            case domtype(b)
              of DOM_INT do
                dom::intmult(args()); break;
              of F do
                new(dom, multcoeffs(extop(a,1), b)); break;
              otherwise
                if (c:= F::convert(b)) <> FAIL then
                  new(dom, multcoeffs(extop(a,1), c))
                else  
                  (b)::dom::_mult(args())
                end_if
            end_case
          elif domtype(a) <> dom then
            case domtype(a)
              of DOM_INT do
                dom::intmult(b, a); break;
              of F do
                new(dom, multcoeffs(extop(b,1), a)); break;
              otherwise
                if (c:= F::convert(a)) <> FAIL then
                  new(dom, multcoeffs(extop(b,1), c))
                else
                  FAIL
                end_if
            end_case
          else
            new(dom, divide(extop(a,1) * extop(b,1), P, Rem))
          end_if
        elif args(0) = 1 then
          a
        else
          _mult(args(1..(args(0) div 2)));
          _mult(args(((args(0) div 2) + 1)..args(0)));
          if (%2)::dom::_mult <> FAIL then
            (%2)::dom::_mult(%2, %1)
          elif (%1)::dom::_mult <> FAIL then
            (%1)::dom::_mult(%2, %1)
          else
            _mult(%2, %1)
          end_if
        end_if
      end_proc
    end_if;
    
    intmult:= (x,y)->new(dom, multcoeffs(extop(x,1), y));

    _invert:=
    if F= Dom::Rational then
      // there is a special method in gcdex availabel for this
      proc(cc)
        local c, h, s, t;
      begin
        c:= extop(cc, 1);
        // find s, t with s*c + t*P = 1; we do not need t
        [h, s, t]:= [gcdex(c, P)];
        assert(expr(h) = 1);
        return(new(dom, s))
      end_proc
    else
      proc(cc)
        local c, q,c1,c2,d,d1,d2,r,r1,r2,l;
      begin
        c:= extop(cc,1);
        if degree(c) = 0 then
          return(new(dom, mapcoeffs(c, 1/id)))
        end_if;
        l:= 1/lcoeff(c);
        c:= multcoeffs(c, l);
        c1:= extop(dom::one,1);
        d1:= extop(dom::zero,1);
        c2:= d1;
        d2:= c1;
        d:= P;
        while not iszero(d) do
          q:= divide(c,d); r:= op(q,2); q:= op(q,1);
          r1:= c1-q*d1; r2:= c2-q*d2;
          c:= d; c1:= d1; c2:= d2;
          d:= r; d1:= r1; d2:= r2;
        end_while;
        new(dom, multcoeffs(c1, l / lcoeff(c)))
      end_proc
    end_if;

    conjNorm:=
    if F::constructor = Dom::IntegerMod then
      proc()
      begin
        F::convert(polylib::resultant(P, extop(args(1),1), Alias))
      end_proc
    else
      proc()
      begin
        polylib::resultant(P,extop(args(1),1), Alias)
      end_proc
    end_if;
    
    conjTrace:=
    proc(aa)
      local i,result,x, a;
    begin
      a:=extop(aa,1);
      x:=poly(Alias, op(a,2..3));
      result:=coeff(a,0);
      for i from 1 to degree(P)-1 do
        a:=divide(a*x, P, hold(Rem));
        result:=result + coeff(a,Alias,i)
      end_for;
      F::convert(result)
    end_proc;


    gcd:=
    proc(x,y)
    begin
      if args(0)=1 then
        x
      else  
        dom::new(gcd(content(extop(x,1)), content(extop(y,1))))
      end_if
    end_proc;

    /* convert -- convert expression into algebraic number
   
      The expression must be a polynomial of the same type as the polynomial P
      defining the adjoinded element. It is interpreted as a polynomial in the
      adjoined element. */
     
    convert:=
    if F::constructor = Dom::IntegerMod then
      proc(xx)
        local x;
      begin
        x:=xx;
        if domtype(x) = dom then
          return(x)
        end_if;
        if domtype(x)=F then
          x:=expr(x);
        end_if;
        x:= poly(x, op(P, 2..3));
        if x = FAIL then
          if testtype(xx, Type::Arithmetical) then
            x:=denom(xx);
            if has(x, dom::variable) then
              return(dom::_mult(dom::new(numer(xx)), dom::_invert(dom::new(x))))
            end_if;
          end_if;
          return(FAIL)
        end_if;
        new(dom, divide(x, P, Rem))
      end_proc
    elif F=Dom::Rational then
      // do this *only* for the rationals
      // *not* for domains inheriting from them
      // hence F::hasProp(Dom::Rational) is FALSE
      proc(xx)
        local x,y;
      begin
        x:=xx;
        if domtype(x) = dom then
          return(x)
        end_if;
        // take care that there are no further indets
        x:= poly(x, op(P,2), Expr);
        if type(x)<>DOM_POLY then
          if not testtype(xx, Type::Arithmetical) then
            return(FAIL)
          end_if;
          x:=denom(xx);
          if has(x, dom::variable) then
            x:=dom::convert(x);
            y:=dom::convert(numer(xx));
            if x=FAIL or y=FAIL then
              return(FAIL)
            else  
              return(dom::_mult(y, dom::_invert(x)))
            end_if
          end_if;
          return(FAIL)
        end_if;
        if not testtype(x, Type::PolyOf(Type::Rational)) then
          return(FAIL)
        end_if;
        new(dom, divide(x, P, Rem))
      end_proc
    else
      proc(xx)
        local x, numerator, denominator;
      begin
        x:=xx;
        if domtype(x) = dom then
          return(x)
        end_if;        
        x:= poly(x, op(P, 2..3));
        if not testtype(x, Type::PolyOf(F)) then
          if not testtype(xx, Type::Arithmetical) then
            return(FAIL)
          end_if;
          x:=denom(xx);
          if has(x, dom::variable) then
            numerator:=dom::convert(numer(xx));
            denominator:=dom::convert(x);
            if numerator=FAIL or denominator=FAIL then
              return(FAIL)
            else  
              return(dom::_mult(numerator, dom::_invert(denominator)))
            end_if
          end_if;
          return(FAIL)
        end_if;
        new(dom, divide(x, P, Rem))
      end_proc
    end_if;
   
    convert_to:=
    proc(xx, T)
      local x;
    begin
      x:=xx;
      if domtype(x) = dom then
        case T
          of dom do
            return(x);
          of DOM_POLY do
            return(extop(x,1));
          of DOM_EXPR do
            return(expr(extop(x,1)));
        end_case;
        x:= extop(x, 1);
        if degree(x) = 0 then
          x:= lcoeff(x);
          return((if domtype(x) = T then
                    x
                  elif domtype(T)=DOM_DOMAIN then
                    T::coerce(x)
                  else
                    FAIL
                  end_if))
        end_if
      end_if;
      FAIL
    end_proc;

    /* minimalPolynomial(a)

       returns the minimal polynomial of the domain element a
    */


    minimalPolynomial:=
    proc(a:dom)
      local minpol_a,ausdr,p1,unb,unb1,hilf,base_field;
    begin
      ausdr      := expr(dom::minpoly);
      base_field := dom::groundField;
      unb        := dom::variable;
      unb1       := genident();
      p1         := poly(unb1-dom::expr(a),[unb,unb1],base_field);
      hilf       := poly(ausdr,[unb,unb1],base_field);
      minpol_a   := polylib::resultant(hilf,p1,unb);
      // problem in characteristic p if resultant is a pth power!!
      // minpol_a   := divide(minpol_a, gcd(minpol_a, D(minpol_a)),Quo);
      op(polylib::sqrfree(minpol_a), 2)
    end_proc;

    

      
/* Qpoly: auxiliary entry to compute the unique extension of
  the derivation on F*/

    makeQpoly:=
      (if F::D <> FAIL and
         not iszero(F::D) then
         proc()
           begin
             D(P);
             if iszero(%) then
               error("Inseparable extension")
             else
               extop(dom(-mapcoeffs(P, F::D))/dom(%),1)
             end_if
           end_proc
       else
         FAIL
       end_if
       );

    Qpoly:=dom::makeQpoly();
    
    Q:= 
    (if F::diff <> FAIL
       and not iszero(F::diff) then

       proc(x)
         local dP;

       begin
         dP:=D(P);
         if iszero(%) then
           error("Inseparable extension")
         end_if;
         -mapcoeffs(P, F::diff, x);
         extop(dom(%)/dom(dP),1)
       end_proc

      end_if);

    D:= 
    (if F::D = FAIL then
       FAIL       
     elif iszero(F::D) then
       dom::zero
     else  
       proc(a)
         begin
         extop(a,1);
         dom::new(mapcoeffs(%, F::D)+dom::Qpoly*diff(%, Alias))	
         end_proc;

     end_if);

    diff:=
    (if F::diff = FAIL then
       FAIL
     elif iszero(F::diff) then
       dom::zero
     else  

       proc(a)
       begin
         if args(0)=1 then
           args(1)
         elif args(0)>2 then
           dom::diff(dom::diff(args(1),args(2)), args(3..args(0)))
         else
           extop(args(1),1);
           dom::new(mapcoeffs(%, F::diff, args(2))+
		     dom::Q(args(2))*diff(%, Alias))
         end_if
       end_proc
     end_if);

    expr:= a -> expr(extop(a,1));

    expr2text:= a -> expr2text(extop(a, 0))."(".expr2text(extop(a, 1)).")";
    
    print := generate::sortSums@dom::expr;
    Content := (Out, data) -> Out(generate::sortSums(dom::expr(data)));
        
    random:=
    (if F = Dom::Rational then
	 proc()
     begin
     new(dom, poly(polylib::randpoly([dom::variable], Expr,
         Degree=degree(P)-1,Terms=infinity),Expr) )
     end_proc
    elif F::constructor = Dom::IntegerMod then
	 proc()
     begin
       new(dom, polylib::randpoly([dom::variable], op(P,3),
           Degree=degree(P)-1,Terms=infinity) )
     end_proc
    else
	 proc()
     begin
       new(dom, polylib::randpoly([dom::variable], F,
           Degree=degree(P)-1,Terms=infinity) )
     end_proc
    end_if);



begin
  case args(0)
    of 2 do
      Alias:= null();
      break;
    of 3 do
      if domtype(Alias) <> DOM_IDENT then
        error("illegal variable type")
      end_if;
      Alias:= [ Alias ];
      break;
    otherwise
      error("wrong no of args")
  end_case;
  if domtype(F)<>DOM_DOMAIN then
    error("First argument must be a domain")
  end_if;
  if F::hasProp(Cat::Field) <> TRUE then
    error("no Cat::Field given")
  end_if;
  if not F::hasProp(Ax::normalRep) then
    error("field has no normal representation")
  end_if;
  if type(PExpr) = "_equal" then
    PExpr:= op(PExpr,1) - op(PExpr,2)
  end_if;
  if F = Dom::Rational then
    P:= poly(PExpr, Alias, Expr);
    if not testtype(P, Type::PolyOf(Type::Rational,1)) then
      error("not a univariate polynomial")
    end_if;
  elif F::constructor = Dom::IntegerMod then
    P:= poly(PExpr, Alias, hold(IntMod)(F::size));
    if P = FAIL then
      error("not a polynomial")
    end_if;
    if nops(op(P,2)) <> 1 then
      error("multivariate polynomial")
    end_if;
  else
    P:= poly(PExpr, Alias, F);
    if P = FAIL then
      error("not a polynomial")
    end_if;
    if nops(op(P,2)) <> 1 then
      error("multivariate polynomial")
    end_if;
  end_if;
  if degree(P) = 0 then
    error("constant polynomial")
  end_if;
  P:= multcoeffs(P, 1/lcoeff(P));
  PExpr:= (expr(P) = 0);
  Alias:= op(P,[2,1]);

end_domain:

// end of file 
