/*
    Dom::DenseMatrix -- the domains of matrices 

    Dom::DenseMatrix([R])

    R - (optional) coefficient domain (a Cat::Rng)

    A domain Dom::DenseMatrix(R) represents all matrices over the 
    Cat::Rng R.  If the argument 'R' is missing then the coefficient 
    domain will be the domain 'Dom::ExpressionField()'. 

    An element of a domain created by Dom::DenseMatrix (a matrix), has three
    operands 'r, c, a'. The first two operands give the dimension 
    [r,c] of the matrix and the third operand is a two dimensional 
    array a which holds the entries of the matrix.
*/

domain Dom::DenseMatrix( R )
    local Rcoerce, RhasSystemRep, Riszero, Rmult, Rplus, Rzero, Rnormal, Rid;
    inherits Dom::BaseDomain;
    category Cat::Matrix(R);
    axiom if R::hasProp( Ax::canonicalRep ) then Ax::canonicalRep end_if;

// -----------------------------------------
//               entries
// -----------------------------------------
    isSparse:= FALSE;
    randomDimen:= [10,10];
    Name:= if dom::constructor = Dom::DenseMatrix and
	      R = Dom::ExpressionField() then
	       hold(Dom::DenseMatrix)()
	   end;

// -----------------------------------------
//                methods 
// -----------------------------------------

/*--
    new  --  return a new matrix
--*/
   new:= proc(r,c,f)
        local i, j, a, n;
    begin
        case args(0)
        of 1 do
            if testargs() then
                if r::dom::hasProp( Cat::Matrix ) <> TRUE and
                   domtype(r) <> DOM_LIST and
                   (domtype(r) = DOM_ARRAY and op(r,[0,1]) > 2) and
                   (domtype(r) = DOM_HFARRAY and op(r,[0,1]) > 2) 
                then
                    error("invalid argument")
                end_if
            end_if;
            if (r:= dom::coerce( r )) = FAIL then
                error("unable to define matrix over ".expr2text(R))
            else
                return( r )
            end_if
        of 2 do
            if not testtype( r, Type::PosInt ) or not testtype( c, Type::PosInt ) then
                error("expecting dimension as positive integers")
            else
                return( new(dom,r,c,array(1..r,1..c,[[Rzero $ c] $ r])) )
            end_if
        of 3 do
            if (not testtype( r, Type::PosInt )) or 
               (not testtype( c, Type::PosInt )) then
                error("expecting dimension as positive integers")
            elif (domtype(f) = DOM_ARRAY or
                  domtype(f) = DOM_HFARRAY) then
                a:= dom::mkDense( r,c,f );
                if a = FAIL then
                    error("unable to define matrix over ".expr2text(R))
                else
                    return(new(dom,op(a)) )
                end_if:
            elif domtype( f ) <> DOM_LIST then
                // consider f as a function of (i,j)
                a:= array(1..r,1..c, [[Rcoerce( f(i,j) ) $ j=1..c] $ i=1..r]);
                if has( a, FAIL ) then
                    error("unable to convert function value");
                else
                    return( new(dom,r,c,a) )
                end_if
            else
                a:= dom::mkDense( r,c,f );
                if a = FAIL then
                    error("unable to define matrix over ".expr2text(R))
                else
                    return( new(dom,op(a)) )
                end_if
            end_if
        of 4 do 
            // create diagonal matrix 
            case args(4)
            of hold(Diagonal) do
                if domtype( f ) <> DOM_LIST then
                    a:= array(1..r,1..c,
                      [[Rzero $ i-1, Rcoerce(f(i)), Rzero $ c-i] $ i=1..stdlib::min(r,c), 
                       [Rzero $ c] $ r-stdlib::min(r,c)]
                    )
                else
                    a:= assignElements( array(1..r,1..c,[ [Rzero $ c] $ r ]),
                      [i,i] = Rcoerce( f[i] ) $ i=1..stdlib::min(r,c,nops(f)) 
                    )
                end_if;

                if has( a, FAIL ) then
                    error("unable to define matrix over ".expr2text(R))
                else
                    return( new(dom,r,c,a) )
                end_if
            of hold(Banded) do
                n:= 2*stdlib::min(r,c) - 1; // give at most n elements in f
                if domtype(f) <> DOM_LIST or nops(f) mod 2 = 0 or nops( f ) > n then
                    error("expecting a list with an odd number of elements which is not greater than ".expr2text(n))
                else
                    n:= nops(f) div 2;
                    a:= array(1..r,1..c,
                        [[ (if specfunc::abs(j-i) > n then Rzero else Rcoerce(f[n+j-i+1]) end_if) $ j=1..c] $ i=1..r]
                    );
                    if has( a,FAIL ) then 
                        error("unable to define matrix over ".expr2text(R))
                    else
                        return( new(dom,r,c,a) )
                    end_if
                end_if
            otherwise
                error("expecting 'Diagonal' or 'Banded' as options")
            end_case 
        otherwise
            error("wrong no of args")
        end_case
    end_proc;

/*--
    create  --  create matrices without testing the arguments
--*/ 
    create:= proc(r,c,f)
        local i, j, a, n;
    begin
        case args(0)
        of 2 do
            return( new(dom,r,c,array(1..r,1..c,[[Rzero $ c] $ r])) )
        of 1 do
            if domtype(r) = DOM_ARRAY then
                return( new(dom,op(r,[0,2,2]) - op(r,[0,2,1]) + 1, op(r,[0,3,2]) - op(r,[0,3,1]) + 1,r) )
            elif domtype(r) = DOM_HFARRAY then
                r:= array(op(r, [0, 2]), op(r, [0,3]), [op(r)]):
                return( new(dom,op(r,[0,2,2]) - op(r,[0,2,1]) + 1, op(r,[0,3,2]) - op(r,[0,3,1]) + 1,r) )
            elif domtype(r) = DOM_LIST then
                f:= r; r:= nops(f); c:= 1;
                if domtype(f[1]) = DOM_LIST then
                    c:= stdlib::max( op(map(f,nops)) )
                end_if
            elif r::dom::hasProp(Cat::Matrix) = TRUE then
                return( new(dom,extop(r)) )
            else
                error("invalid argument")
            end_if
        of 3 do
           if domtype(f) = DOM_ARRAY or 
              domtype(f) = DOM_HFARRAY and
              nops(f) = r*c then
              return(new(dom,r, c, array(1..r, 1..c, [op(f)])));
           end_if:
        of 4 do
        of 3 do
            if domtype(f) <> DOM_LIST then
                // consider f as a function of (i,j)
                if args(0) = 4 and args(4) = hold(Diagonal) then 
                    return( new(dom,r,c,array(1..r,1..c,
                      [[Rzero $ i-1, f(i), Rzero $ c-i] $ i=1..stdlib::min(r,c), 
                       [Rzero $ c] $ r-stdlib::min(r,c)]
                    )) )
                else
                    return( new(dom,r,c,array(1..r,1..c,
                      [[f(i,j) $ j=1..c] $ i=1..r]
                    )) )
                end_if
            end_if;

            if args(0) = 4 then
                if args(4) = hold(Diagonal) then
                    a := array( 1..r,1..c,[ [Rzero $ c] $ r ] );
                    a:= assignElements( a,([i,i] = f[i]) $ i=1..stdlib::min(r,c,nops(f)) )
                else
                    // option Banded:
                    n:= nops(f) div 2;
                    a:= array(1..r,1..c,
                        [[ (if specfunc::abs(j-i) > n then Rzero else f[n+j-i+1] end_if) $ j=1..c] $ i=1..r]
                    )
                end_if
            elif r > 1 and c > 1 then
                // define matrix by list of list:
                a:= array( 1..r,1..c,[ [Rzero $ c] $ r ] );
                a:= assignElements( a,( ([i,j] = op(f,[i,j])) $ j=1..nops(f[i]) ) $ i=1..nops(f) );
            elif r = 1 then
                if domtype(f[1]) = DOM_LIST then f:= f[1] end_if;
                a:= array( 1..r,1..c,[ [Rzero $ c] $ r ] );
                a:= assignElements( a,([1,i] = f[i]) $ i=1..nops(f) )
            elif domtype(f[1]) = DOM_LIST then
                // case [[1],[2],...]:
                a:= array( 1..r,1..c,[ [Rzero $ c] $ r ] );
                a:= assignElements( a,([i,1] = f[i][1]) $ i=1..nops(f) )
            else
                a:= array( 1..r,1..c,[ [Rzero $ c] $ r ] );
                a:= assignElements( a,([i,1] = f[i]) $ i=1..nops(f) )
            end_if;

            return(new( dom,r,c,a ))
        end_case
    end_proc;

/*--
    mkDense  --  convert an expression to an array 
--*/
    mkDense:= proc(x)
        local i, j, m, n, a, t, offsi, offsj, aux;
    begin
        case args(0)
        of 1 do
            case domtype(x)
            of DOM_HFARRAY do
               x:= array(op(x, [0, i]) $ i = 2..nops([op(x, 0)]), [op(x)]);
               // no break here: proceed to DOM_ARRAY
            of DOM_ARRAY do
                if has(x,NIL) then
                    userinfo(1,"unable to define matrix over ",R);
                    return( FAIL )
                end_if;
                a:= map( x,Rcoerce );
                if contains( {op(a)},FAIL ) then
                    userinfo(1,"unable to define matrix over ",R);
                    return( FAIL )
                end_if;  
                m:= op(x,[0,2,2]) - op(x,[0,2,1]) + 1;
                if op(x,[0,1]) = 2 then 
                    n:= op(x,[0,3,2]) - op(x,[0,3,1]) + 1;
                    if op(x,[0,2,1]) = 1 and op(x,[0,3,1]) = 1 then
                        return([m,n,a])
                    end_if;
                    // return( [m,n,subsop(array(1..m,1..n),i=op(a,i) $ i=1..nops(a))] )
                    t:= array(1..m,1..n);
                    offsi:= op(x,[0,2,1])-1;
                    offsj:= op(x,[0,3,1])-1;
                    for i from 1 to m do
                        aux:= i+offsi;
                        for j from 1 to n do
                            t[i,j]:= a[aux, j+offsj]
                        end_for
                    end_for;
                    return([m,n,t])
                else
                    return([m,1, array(1..m,1..1, map([op(a)], DOM_LIST)) ])
                end_if;
            of DOM_LIST do
                if (m:= nops(x)) = 0 then return(FAIL) end_if;
                n:= 1;
                if domtype(x[1]) = DOM_LIST then
                    n:= stdlib::max( op(map(x,nops)) );
                    if n = 0 then return(FAIL) end_if
                end_if;
                break
            otherwise
                return( FAIL )
            end_case;
            break
        of 3 do
            m:= x; 
            n:= args(2); 
            if not testtype( m,Type::PosInt ) or 
               not testtype( n,Type::PosInt ) then
                return( FAIL ) 
            end_if;
            x:= args(3);
            if domtype(x) = DOM_LIST and
               nops(x) = m*n and
               not contains(map({op(x)}, domtype), DOM_LIST) then
               // input as a plain list of operands. Convert to
               // a nested list:
               x:= [[x[(i-1)*n + j] $ j = 1.. n] $ i = 1 .. m];
            elif (domtype(x) = DOM_ARRAY or
                  domtype(x) = DOM_HFARRAY) and 
                  nops(x) = m*n then
                  x:= [op(x)];
                  x:= [[x[(i-1)*n + j] $ j = 1.. n] $ i = 1 .. m];
            end_if:
            break
        otherwise
            return( FAIL )
        end_case;

        if m > 1 and n > 1 then
          // define matrix by list of list:
          // avoid assignElements with a huge argument list
          a:= array( 1..m,1..n,
                    [[Rcoerce(op(x,[i,j])) $ j=1..nops(x[i]),
                      Rzero $ n-nops(x[i])]
                     $ i=1..nops(x),
                     [Rzero $ n] $ m-nops(x)] );
        elif m = 1 then
            if domtype(x[1]) = DOM_LIST then 
                // 1,n,[[a,b,...]]
                if nops(x) <> 1 then
                    return( FAIL )
                end_if;
                x:= x[1]
            end_if;
            if nops(x) > n then
                return( FAIL ) 
            end_if;
            a:= array( 1..m,1..n,[ [Rzero $ n] $ m ] );
            a:= assignElements( a,([1,i] = Rcoerce(x[i])) $ i=1..nops(x) )
        else
            // n = 1:
            if domtype(x[1]) = DOM_LIST then
                // m,1,[[a],[b],...]:
                if stdlib::max( op(map(x,nops)) ) > n then
                    return( FAIL )
                end_if;
                a:= array( 1..m,1..n,[ [Rzero $ n] $ m ] ); 
                a:= assignElements( a,([i,1] = (if nops(x[i]) = 0 then Rzero else Rcoerce(x[i][1]) end)) $ i=1..nops(x) )
            else

                a:= array( 1..m,1..n,[ [Rzero $ n] $ m ] );
                a:= assignElements( a,([i,1] = Rcoerce(x[i])) $ i=1..nops(x) )
            end_if
        end_if;

        if contains([op(a)],FAIL) <> 0 then
            return( FAIL )
        else
            return([m,n,a])
        end_if
    end_proc;

/*--
    _concat  --  appends matrices horizontallt (see concatMatrix)
--*/
    _concat:= proc()
        local e;
    begin
        if testargs() then
            for e in {args()} do
                if e::dom <> dom and e::dom <> Dom::DenseMatrix(R) and e::dom::getSuperDomain(e) <> Dom::DenseMatrix(R) then
                    error("illegal operands")
                end_if
            end_for
        end_if;
        dom::concatMatrix(args())
    end_proc;

/*--
    _index  --  indexing of matrices and vectors
--*/
    _index:= proc(x,i,j) 
        local a, r1, r2, b1, b2, e1, e2;
    begin
        if args(0) = 3 then
            if domtype(i) = DOM_INT and domtype(j) = DOM_INT then
                return(eval(_index(extop(x,3),i,j)))
            elif type(i) = "_range" and type(j) = "_range" then
                x:= extop(x,3);
                b1:= op(i,1); e1:= op(i,2);
                b2:= op(j,1); e2:= op(j,2);
                a:= array(1..e1-b1+1, 1..e2-b2+1,
                    ( (r1,r2)=x[b1+r1-1, b2+r2-1] $ r1=1..e1-b1+1 ) $ r2=1..e2-b2+1
                );
                return(new( dom, e1-b1+1, e2-b2+1, a ))
            else
                return(hold(_index)(args()));
            end_if
        elif extop(x,2) = 1 then
            // a column vector:
            if domtype(i) = DOM_INT then
                return(eval(_index(extop(x,3),i,1)))
            elif type(i) = "_range" then
                x:= extop(x,3);
                b1:= op(i,1); e1:= op(i,2);
                a:= array(1..e1-b1+1, 1..1, (r1,1)=x[b1+r1-1,1] $ r1=1..e1-b1+1);
                return(new( dom, e1-b1+1, 1, a ))
            else
                return(hold(_index)(args()));
            end_if
        elif extop(x,1) <> 1 then
            error("missing column index")
        // a row vector:
        elif domtype(i) = DOM_INT then
            return(eval(_index(extop(x,3),1,i)))
        elif type(i) = "_range" then
            x:= extop(x,3);
            b1:= op(i,1); e1:= op(i,2);
            a:= array(1..1, 1..e1-b1+1, (1,r1)=x[1,b1+r1-1] $ r1=1..e1-b1+1);
            return(new( dom, 1, e1-b1+1, a ))
        else
            return(hold(_index)(args()));
        end_if
    end_proc;
        
/*--
    set_index  --  indexing of matrices 
--*/
    set_index:= proc(x,i,y,y2)
        local r, c, a;
    begin
        if args(0) = 4 and type(i) = "_range" or type(y) = "_range" then 
          return(x::dom::set_index2(x,i,y,y2))
        end_if;
        if testargs() then
            // check correct type of entry only at the interactive level 
            if args(0) = 4 then 
                if (y2:= Rcoerce(y2)) = FAIL then
                    warning("invalid matrix component, assignment ignored");
                    return(x)
                end_if
            elif (y:= Rcoerce(y)) = FAIL then
                warning("invalid vector component, assignment ignored");
                return(x)
            end_if
        end_if;

        r:= extop(x,1);
        c:= extop(x,2);
        a:= extop(x,3);
        if args(0) = 4 then
            if domtype(i) = DOM_INT and domtype(y) = DOM_INT 
               and i > 0 and i <= r and y > 0 and y <= c
            then
                a[i,y]:= y2;
                return(extsubsop( x,3=a ))
            else
                warning("invalid matrix indices, assignment ignored");
                return(x)
            end_if
        elif r = 1 then
            if domtype(i) = DOM_INT and i > 0 and i <= c then
                a[1,i]:= y;
                extsubsop( x,3=a ) 
            else
                warning("invalid column index, assignment ignored");
                return(x)
            end_if
        elif c <> 1 then
            warning("missing column index, assignment ignored");
            return(x)
        elif domtype(i) = DOM_INT and i > 0 and i <= r then
            a[i,1]:= y;
            return(extsubsop( x,3=a ))
        else
            warning("invalid row index, assignment ignored");
            return(x)
        end_if
    end_proc;

/*-------------------------------------------------------------------
  set_index2  --  substituting submatrices via indexing
      set_index(x, i1..i2, j1..j2, y)  <--> x[i1..i2,j1..j2]:= y;

  The matrix y must be of the same type as x or has to be converted
  to a matrix of the same type as x. If this fails, the matrix x 
  is returned. Otherwise, the submatrix x[i1..i2, j1..j2] of x is
  substituted by y.

  NOTE: This procedure can only be called from set_index. And 
        this is only the case, if set_index is called with exactly 
        four argument, at least one of them a range. 
      
-------------------------------------------------------------------*/

set_index2:= proc(x, r_range, c_range, y)
  local rx, cx, ry, cy, i, j, xx, yy;
begin 
  if ((y:= x::dom::convert(y))) = FAIL then 
    warning("new submatrix is incompatible with the original matrix");
    return(x);
  end_if;

  rx:= extop(x,1);
  cx:= extop(x,2);
  ry:= extop(y,1);
  cy:= extop(y,2);
  xx:= extop(x,3);
  yy:= extop(y,3);

  if type(r_range) = "_range" and type(c_range) = "_range" then 

    if r_range[1] <= r_range[2] and 
       c_range[1] <= c_range[2] and 
       r_range[1] >= 1 and 
       r_range[2] <= rx and 
       c_range[1] >= 1 and 
       c_range[2] <= cx and 
       r_range[2] - r_range[1] + 1 = ry and 
       c_range[2] - c_range[1] + 1 = cy then 
      for i from r_range[1] to r_range[2] do 
        for j from c_range[1] to c_range[2] do 
          xx[i,j]:= yy[i - r_range[1] + 1, j - c_range[1] + 1];
        end_for;
      end_for;
      return(extsubsop( x,3=xx ));
    else 
      warning("incompatible range");
      return(x);
    end_if;
    
  elif type(r_range) = "_range" and type(c_range) = DOM_INT then 

    if cy = 1 and 
       r_range[1] <= r_range[2] and 
       r_range[1] >= 1 and 
       r_range[2] <= rx and 
       c_range    >= 1 and 
       c_range    <= cx and 
       r_range[2] - r_range[1] + 1 = ry then 
      for i from r_range[1] to r_range[2] do 
        xx[i,c_range]:= yy[i - r_range[1] + 1, 1];
      end_for;
      return(extsubsop( x,3=xx ));
    else 
      warning("incompatible range");
      return(x);
    end_if;
  
  elif type(c_range) = "_range" and type(r_range) = DOM_INT then 

    if ry = 1 and 
       c_range[1] <= c_range[2] and 
       c_range[1] >= 1 and 
       c_range[2] <= cx and 
       r_range    >= 1 and 
       r_range    <= rx and 
       c_range[2] - c_range[1] + 1 = cy then 
      for j from c_range[1] to c_range[2] do 
        xx[r_range,j]:= yy[1, j - c_range[1] + 1];
      end_for;
      return(extsubsop( x,3=xx ));
    else 
      warning("incompatible range");
      return(x);
    end_if;

  else 
    warning("expecting ranges or integers as matrix indices");
    return(x);
  end_if;
end_proc;

/*--
    print  --  print a matrix as an array  
               (for consistency to 'Dom::/*Sparse*/Matrix' large matrices
                will not be displayed, since especially Windows-Systems 
                tend to collapse)
--*/

print:= proc(x)
local r,c;
begin
   [r,c]:= x::dom::matdim(x);
   if r*c > 500 or r*c = 0 then
      if r*c > 500 then
        warning("This matrix is too large for display. ".
                "If you really want to see all non-zero entries ".
                "of this matrix (A, say), call A::dom::doprint(A). ");
      end_if;
      return(subsop(hold(dummy)(r, c, ["..."]), 0 = dom));
   end_if;
   return(map(extop(x,3),generate::sortSums));
end_proc;

/*--
    doprint  --  print a matrix as an array wihtout respect to its size
--*/
    doprint:= x -> map(extop(x,3),generate::sortSums);
      
/*--
    expr -- returns the matrix as an element of DOM_ARRAY, whereby
            each entry was converted to an expression (using R::expr).
--*/
    expr:= x -> map(extop(x,3),expr);

/*--
    float -- maps float to the elements of the matrix, but only if
             the domain of the elements has a float method.
--*/
    float:= if R::float <> FAIL then 
    proc(x)    
        local y;
    begin
        y:= map(extop(x,3),float);
        if testargs() then
            // check correct type only at the interactive level
            y:= map(y,Rcoerce);
            if contains({op(y)},FAIL) then
                error("floating point evaluation not possible over the component ring")
            end_if
        end_if;
        return(extsubsop( x,3=y ))
    end_proc
    end_if;

/*--
   convert -- convert an expression to a matrix 
--*/
    convert:= proc(x)
    begin
        if args(0) <> 1 then return( FAIL ) end_if;

        case domtype(x)
        of dom do 
            return( x )
        of DOM_ARRAY do
        of DOM_HFARRAY do
        of DOM_LIST do 
            x:= dom::mkDense( x );
            if x = FAIL then return( FAIL ) else return(new( dom,op(x) )) end_if
        otherwise
            case x::dom::constructor
            of dom::constructor do
                if x::dom::coeffRing = R then
                   return(extsubsop( x,0=dom ))
                elif x::dom::isSparse = TRUE then
                    // not yet implemented!
                    return(FAIL)
                else
                    x:= dom::mkDense( extop( x,3 ) );
                    if x = FAIL then return( FAIL ) else return(new( dom,op(x) )) end_if
                end_if
            otherwise
                return( FAIL )
            end_case
        end_case
    end_proc;

/*--
    convert_to  --  convert matrices 
--*/
    convert_to:= proc(e,F)
        local i, j, t;
    begin
        if args(0) <> 2 then return( FAIL ) end_if;
        if domtype(e) = F then return(e) end_if;
        if domtype(F) <> DOM_DOMAIN then F:= domtype(F) end_if;

        case F
        of DOM_ARRAY do 
            return( extop( e,3 ) ) 
        of DOM_HFARRAY do 
            return( hfarray(1..extop(e,1), 1..extop(e, 2), [op(extop( e,3 ))]) ); 
        of DOM_LIST do
            t:= extop(e,3);
            return( [[t[i,j] $ j=1..extop(e,2)] $ i=1..extop(e,1)] )
        otherwise
            if (t:=coerce(extop(e, 3), DOM_ARRAY)) <> FAIL then
              if (t:= F::convert(t)) <> FAIL then
                return(t)
              else
                userinfo(50, expr2text(F)." is unable to convert arrays")
              end_if
            end_if
        end_case;

        case F::constructor
        of Dom::DenseMatrix do
            // same internal representation of matrices
            if F::isSparse = TRUE then
                // not yet implemented!
                return(FAIL)
            elif F::coeffRing = e::dom::coeffRing then
                return( extsubsop( e,0=F ) )
            else
                t:= F::mkDense( extop( e,3 ) );
                if t <> FAIL then 
                    return( new( F,op(t) ) )
                end_if
            end_if
        otherwise
            return( FAIL )
        end_case
    end_proc;

/*--
    matdim  --  return the dimension of a matrix 
--*/
    matdim:= x -> [extop(x,1),extop(x,2)];

/*--
    _mult  --  multiply matrices

    Syntax:

    _mult(x1 [,x2,x3])

    Synopsis:

    Multiplies two or more matrices.
    If an argument is not a matrix then it will be converted to an element 
    of R if possible, and scalar multiplication is performed. Otherwise 
    the domain of the arument is asked to perform the multiplication
    (which could return FAIL).
--*/
    _mult:= if R::hasProp( Ax::systemRep ) then
        proc(x,y)
            local r, c, i, j, cy, k;
        begin
            // Only one argument given? Useful for direct use of A::_mult!
            if args(0) = 1 then return( x )
            elif args(0) > 2 then
                x:= _mult( args(i) $ i=1..(args(0) div 2) );
                y:= _mult( args(i) $ i=((args(0) div 2) + 1)..args(0) );
                if x::dom::_mult <> FAIL then x:= x::dom::_mult(x, y)
                elif y::dom::_mult <> FAIL then x:= y::dom::_mult(x, y)
                else x:= _mult(x, y)
                end_if;
                return(x);
            end_if;
    
            case domtype(x) 
            of dom do
                case domtype(y)
                of dom do
                    r:= extop(x,1);
                    c:= extop(x,2);
                    x:= extop(x,3);
                    if c <> extop(y,1) then
                        error("dimensions don't match")
                    end_if;
                    cy:= extop(y,2);
                    y:= extop(y,3);
                    return( new(dom,r,cy,array(1..r,1..cy,[
                      [Rplus(Rmult(x[k,i],y[i,j]) $ i=1..c ) $ j=1..cy]
                            $ k=1..r]
                    )) )
    
                // Is y an integer? Note, that R has system representation.
                of DOM_INT do
                    case y 
                    of 1 do return( x )
                    of 0 do
                        r:= extop(x,1);
                        c:= extop(x,2);
                        return( extsubsop(x,3=array(1..r,1..c,[[0 $ c] $ r])) )
                    otherwise
                        return( extsubsop(x,3=map(extop(x,3),_mult,y)) )
                    end_case
    
                // Is y an element of R?
                  of R do
                    return( extsubsop(x,3=map(extop(x,3),_mult,expr(y))) )
    
                // Could y be converted into dom or R?
                otherwise
                    if (k:= dom::coerce(y)) <> FAIL then
                        return(dom::_mult( x,k ))
                    elif coerce(y,R ) <> FAIL then
                        return( extsubsop(x,3=map(extop(x,3),_mult, expr(y))) )
                    // do not use y::expr, y could be an element of a base domain!
                    else 
                        // Is the domain of y be able to perform the multiplication?
                        return( y::dom::_mult(x, y) )
                    end_if
                end_case

            // Is x an integer? Note, that R has system representation.
            of DOM_INT do
                case x 
                of 1 do 
                    return( y )
                of 0 do
                    r:= extop(y,1);
                    c:= extop(y,2);
                    return( extsubsop(y,3=array(1..r,1..c,[[0 $ c] $ r])) )
                otherwise
                    return( extsubsop(y,3=map(extop(y,3),_mult,x)) )
                end_case

            // Is x an element of R?
            of R do return( extsubsop(y,3=map(extop(y,3),_mult,expr(x))) )
    
            // Could x be converted into dom or R?
            otherwise
                if (k:= dom::coerce(x)) <> FAIL then
                    return(dom::_mult( k,y ))
                elif coerce( x,R ) <> FAIL then
                    return( extsubsop(y,3=map(extop(y,3),_mult,expr(x))) )
                else
                    // unable to perform the multiplication!
                    return( FAIL )
                end_if
            end_case
        end_proc
    elif R::hasProp( Cat::CommutativeRing ) then
        proc(x,y)
            local r, c, i, j, cy, k;
        begin
            // Only one argument given? Useful for direct use of A::_mult.
            if args(0) = 1 then return( x )
            elif args(0) > 2 then
                x:= _mult( args(i) $ i=1..(args(0) div 2) );
                y:= _mult( args(i) $ i=((args(0) div 2) + 1)..args(0) );
                if x::dom::_mult <> FAIL then x:= x::dom::_mult(x, y)
                elif y::dom::_mult <> FAIL then x:= y::dom::_mult(x, y)
                else x:= _mult(x, y)
                end_if;
                return(x);
            end_if;
    
            case domtype(x)
            of dom do
                case domtype(y)
                of dom do
                    r:= extop(x,1);
                    c:= extop(x,2);
                    x:= extop(x,3);
                    if c <> extop(y,1) then
                        error("dimensions don't match")
                    end_if;
                    cy:= extop(y,2);
                    y:= extop(y,3);
                    return( new(dom,r,cy,array(1..r,1..cy,[
                        [Rplus(Rmult(x[k,i],y[i,j]) $ i=1..c ) $ j=1..cy] $ k=1..r]
                    )) );

                of DOM_INT do
                    case y 
                    of 1 do return( x )
                    of -1 do return( dom::_negate(x) )
                    of 0 do
                        r:= extop(x,1);
                        c:= extop(x,2);
                        return( extsubsop(x,3=array(1..r,1..c,[[Rzero $ c] $ r])) )
                    otherwise
                        // Is R a ring with unit?
                        if (i:=R::one) <> FAIL then
                            // c = i+i+...+i (|y| times) becomes an element of R:
                            c:= Rplus( i $ specfunc::abs(y) );
                            x:= extsubsop(x,3=map(extop(x,3),Rmult,c));
                            if y > 0 then return( x )
                            else return( dom::_negate(x) )
                            end_if
                        elif y > 0 then return( dom::_plus(x $ y) )
                        else return( dom::_negate(dom::_plus(x $ -y)) )
                        end_if;
                    end_case

                of R do return( extsubsop(x,3=map(extop(x,3),Rmult,y)) )

                otherwise
                    if (k:= dom::coerce( y )) <> FAIL then
                        return(dom::_mult( x,k ))
                    elif (k:= Rcoerce(y)) <> FAIL then
                        return( extsubsop(x,3=map(extop(x,3),Rmult,k)) )
                    else
                        // Is the domain of y be able to perform the multiplication?
                        return( y::dom::_mult(x, y) )
                    end_if
                end_case


            of DOM_INT do
                case x
                of 1 do return( y )
                of -1 do return( dom::_negate(y) )
                of 0 do
                    r:= extop(y,1);
                    c:= extop(y,2);
                    return( extsubsop(y,3=array(1..r,1..c,[[Rzero $ c] $ r])) )
                otherwise
                    // Is R a ring with unit? 
                    if (i:=R::one) <> FAIL then
                        // c = i+i+...+i (|x| times) becomes an element of R!
                        c:= Rplus( i $ specfunc::abs(x) );
                        y:= extsubsop(y,3=map(extop(y,3),Rmult,c));
                        if x > 0 then return( y )
                        else return( dom::_negate(y) )
                        end_if
                    elif x > 0 then return( dom::_plus(y $ x) )
                    else return( dom::_negate(dom::_plus(y $ -x)) )
                    end_if;
                end_case

            // scalar * Matrix ? Use of map is possible because R is commutative
            of R do 
                return( extsubsop(y,3=map(extop(y,3),Rmult,x)) )

            otherwise
                if (k:= dom::coerce(x)) <> FAIL then
                    return(dom::_mult( k,y ))
                elif (k:= Rcoerce(x)) <> FAIL then
                    return( extsubsop(y,3=map(extop(y,3),Rmult,k)) )
                else
                    // unable to perform the multiplication
                    return( FAIL )
                end_if
            end_case;
        end_proc
    else
        proc(x,y)
            local r, c, i, j, cy, k;
        begin
            // Only one argument given? Useful for direct use of A::_mult!
            if args(0) = 1 then return( x )
            elif args(0) > 2 then
                x:= _mult( args(i) $ i=1..(args(0) div 2) );
                y:= _mult( args(i) $ i=((args(0) div 2) + 1)..args(0) );
                if x::dom::_mult <> FAIL then x:= x::dom::_mult(x, y)
                elif y::dom::_mult <> FAIL then x:= y::dom::_mult(x, y)
                else x:= _mult(x, y)
                end_if;
                return(x);
            end_if;
    
            case domtype(x)
            of dom do
                case domtype(y)
                of dom do
                    r:= extop(x,1);
                    c:= extop(x,2);
                    x:= extop(x,3);
                    if c <> extop(y,1) then
                        error("dimensions don't match")
                    end_if;
                    cy:= extop(y,2);
                    y:= extop(y,3);
                    return( new(dom,r,cy,array(1..r,1..cy,[
                      [Rplus(Rmult(x[k,i],y[i,j]) $ i=1..c ) $ j=1..cy] $ k=1..r]
                    )) );
    
                of DOM_INT do
                    case y
                    of 1 do return( x )
                    of -1 do return( dom::_negate(x) )
                    of 0 do
                        r:= extop(x,1);
                        c:= extop(x,2);
                        return( extsubsop(x,3=array(1..r,1..c,[[Rzero $ c] $ r])) )
                    otherwise
                        // Is R a ring with unit?
                        if (i:=R::one) <> FAIL then
                            // c = i+i+...+i (|y| times) becomes an element of R!
                            c:= Rplus( i $ specfunc::abs(y) );
                            x:= extsubsop(x,3=map(extop(x,3),Rmult,c));
                            if y > 0 then return( x )
                            else return( dom::_negate(x) )
                            end_if
                        elif y > 0 then return( dom::_plus(x $ y) )
                        else return( dom::_negate(dom::_plus(x $ -y)) )
                        end_if;
                    end_case
    
                of R do return( extsubsop(x,3=map(extop(x,3),Rmult,y)) )
    
                otherwise
                    if (k:= dom::coerce(y)) <> FAIL then
                        return(dom::_mult( x,k ))
                    elif (k:= Rcoerce(y)) <> FAIL then
                        return( extsubsop(x,3=map(extop(x,3),Rmult,k)) )
                    else
                        // Is the domain of y be able to perform the multiplication?
                        return( y::dom::_mult(x, y) )
                    end_if
                end_case
    
    
            of DOM_INT do
                case x
                of 1 do return( y )
                of -1 do return( dom::_negate(y) )
                of 0 do
                    r:= extop(y,1);
                    c:= extop(y,2);
                    return( extsubsop(y,3=array(1..r,1..c,[[Rzero $ c] $ r])) )
                otherwise
                    // Is R a ring with unit?
                    if (i:=R::one) <> FAIL then
                        // c = i+i+...+i (|x| times) becomes an element of R!
                        c:= Rplus( i $ specfunc::abs(x) );
                        y:= extsubsop(y,3=map(extop(y,3),Rmult,c));
                        if x > 0 then return( y )
                        else return( dom::_negate(y) )
                        end_if
                    elif x > 0 then return( dom::_plus(y $ x) )
                    else return( dom::_negate(dom::_plus(y $ -x)) )
                    end_if;
                end_case
    
            of R do
                // scalar * Matrix; use of map is not possible because R is not commutative!
                r:= extop(y,1);
                c:= extop(y,2);
                cy:= extop(y,3);
                ((cy[i,j] := Rmult(x,cy[i,j])) $ j=1..c) $ i=1..r;
                return( extsubsop(y,3=cy) )
    
            otherwise
                if (k:= dom::coerce(x)) <> FAIL then
                    return(dom::_mult( k,y ))
                elif (k:= Rcoerce(x)) <> FAIL then
                    r:= extop(y,1);
                    c:= extop(y,2);
                    cy:= extop(y,3);
                    ((cy[i,j] := Rmult(k,cy[i,j])) $ j=1..c) $ i=1..r;
                    return( extsubsop(y,3=cy) )
                else
                    // unable to perform the multiplication!
                    return( FAIL )
                end_if
            end_case;
        end_proc
    end_if;

/*--
    _plus  --  add matrices 
--*/
    _plus:= proc()
        local x, y, r, c, k, l, argu, d, tmpr, xx, i, i0;
    begin
        if args(0) = 1 then return( args(1) ) end_if;
        for i from 1 to args(0) do
            if domtype(args(i)) = dom then
                 i0:= i;
                 break;
            else next;
            end_if;
        end_for:
        if i > args(0) then
           // should not happen: no term of type dom in a sum?
           return(FAIL);
        end_if;
        x:= args(i0);
        argu:= [args(1 .. i0 - 1), args(i0 + 1 .. args(0))];
        d:= Rzero:
        for c from 1 to args(0) - 1 do
            if domtype( argu[c] ) <> dom then
                r:= argu[c];
                if (tmpr:= dom::coerce(r)) <> FAIL then
                    argu[c]:= tmpr;
                else
                    if (tmpr:= Rcoerce(r)) = FAIL then
                        userinfo(1,"operands are not compatible");
                       return( FAIL )
                    else
                       // case: matrix A + scalar tmpr
                       d:= d + tmpr;
                       argu[c]:= NIL;
                    end_if:
                end_if
            end_if
        end_for;

        r:= extop(x,1);
        c:= extop(x,2);
        xx:= extop(x,3);
        for y in argu do
            if y = NIL then next; end_if;
            if [extop(y,1),extop(y,2)] <> [r,c] then
                error ("dimensions don't match")
            end_if;
            l:= zip([op(xx)],[op(extop(y,3))],Rplus);
            xx:= subsop( array(1..r,1..c),k=l[k] $ k=1..r*c )
        end_for;
        if d <> Rzero then
           if r <> c then
              return(FAIL);
           end_if;
           (xx[i,i]:= xx[i,i] + d;) $ i=1..min(r,c);
        end_if;
        return(extsubsop(x,3=xx))
    end_proc;

/*--
    _power  --  multiply a matrix with itselfes 

    Syntax:

    _power(A,k)

    A  -- matrix
    k  -- integer

    Synopsis:

    Multiplies A k times. There are three cases:
    - k > 0: computes A*A*...*A (k times) using repeated squaring.
    - k = 0: returns the identity matrix if R has the unit 1.
    - k < 0: computes B*...*B (-k times) with B:= A^(-1) using repeated  
             squaring.

    Returns FAIL if i is not an integer or in the case of
    negative i and the inverse of A can not be computed.
    If k = 0 and R has not the unit 1 then FAIL is returned.
--*/
    _power:= proc(A,k)
        local n, j, P;
    begin
        // the special case k = 1 must be treated first, before any
        // error of the type "not a square matrix" can occur
        if k = 1 then return(A): end_if;
        if extop(A,1) <> extop(A,2) then
            error("not a square matrix")
        elif domtype(k) <> DOM_INT then
          if domtype(k) = DOM_RAT and denom(k) = 2 then 
            return(_power(linalg::sqrtMatrix(A), numer(k)));
          else
            error("illegal matrix power");
          end_if;
        else
            case k
            of  0 do
                if R::one = FAIL then
                    userinfo(1,"coefficient domain is without the unit");
                    return( FAIL )
                else
                    n:= extop( A,1 );
                    return( new( dom,n,n,
                        array(1..n,1..n,[[Rzero $ j-1,R::one,Rzero $ n-j] $ j=1..n])
                    ) )
                end_if
            of  1 do return( A )
            of -1 do  
                if (A:= dom::_invert( A )) = FAIL then
                     userinfo(1,"matrix is singular")
                end_if;
                return( A )
            otherwise
                if k < 0 then
                    if (A:= dom::_invert(A)) = FAIL then
                        userinfo(1,"matrix is singular");
                        return( FAIL )
                    end_if;
                    k:= -k
                end_if;

                // compute A^k (k>0) using repeated squaring:
                if k mod 2 = 1 then
                    P:= A;
                    k:= k - 1
                else
                    P:= dom::_mult( A,A );
                    k:= k - 2
                end_if;
                while TRUE do
                    if k mod 2 = 1 then
                        P:= dom::_mult( P,A )
                    end_if;
                    if k < 2 then return( P ) end_if;

                    k:= k div 2;
                    A:= dom::_mult( A,A )
                end_while;
                // should never arive here ...
            end_case
        end_if
    end_proc;

/*--
    _negate  --  _negate a matrix 
--*/
    _negate:= x -> extsubsop( x,3=map(extop( x,3 ),R::_negate) );

/*
    _invert  --  compute the invert of a matrix 

    Syntax:

    _invert(A)

    A  --  matrix

    Synopsis:
 
    Returns A^(-1) if A is regular, FAIL otherwise.
    numeric::inverse is used if A is defined over one of the
    following coefficient domains: Dom::Integer, Dom::Rational,
    Dom::Real, Dom::Complex, Dom::Float or domains created by
    Dom::ExpressionField.
*/
    _invert:= case R
    of Dom::ExpressionField() do
       linalg::inverse;
       break;
    of Dom::Integer do
        proc(x)
        begin
            x:= numeric::inverse(extop(x,3),Symbolic);
            if x = FAIL then
                userinfo(1,"matrix is not invertible");
                return( FAIL )
            elif (x:= dom::convert(x)) = FAIL then
                userinfo(1,"computation fails over its coefficient domain");
                return( FAIL )
            else
                return( x )
            end_if
        end_proc;
        break
    of Dom::Rational do
    of Dom::Complex do // assumed to have Ax::systemRep!
    of Dom::Real do // assumed to have Ax::systemRep!
        proc(x)
            local y;
        begin
            y:= numeric::inverse(extop(x,3),Symbolic);
            if y = FAIL then
                userinfo(1,"matrix is not invertible");
                return( FAIL )
            else
                return( extsubsop( x,3=y ) )
            end_if
        end_proc;
        break
    of Dom::Float do
        proc(x)
            local y;
        begin
            y:= numeric::inverse(extop(x,3));
            if y = FAIL then
                userinfo(1,"matrix is not invertible");
                return( FAIL )
            else
                return( extsubsop( x,3=y ) )
            end_if
        end_proc;
        break
    otherwise
        if R::constructor = Dom::ExpressionField then
            proc(x)
                local y;
            begin
                y:= numeric::inverse(expr(x),Symbolic);
                if y = FAIL then
                    userinfo(1,"matrix is not invertible");
                    return( FAIL )
                else
                    return( dom::convert( y ) )
                end_if
            end_proc
        elif R::hasProp( Cat::IntegralDomain ) then 
        // in MuPAD an Cat::IntegralDomain has the unit element
        proc(x)
            local n, i, j, t, k, R_divide, R_negate;
        begin
            if extop( x,1 ) <> extop( x,2 ) then
                userinfo(1,"expecting a square matrix");
                return( FAIL )
            end_if;

            // append the n x n identity matrix to x
            n := extop( x,1 ); x := extop( x,3 );
            t := array( 1..n,1..2*n,
                [[ x[i,j]$j=1..n,Rzero$i-1,R::one,Rzero$n-i ] $ i=1..n]
            );

            // rank(x) < n ?
            x := extop( op(dom::gaussElim( new(dom,n,2*n,t) ),1),3 );
            t := { Riszero( x[n,i] ) $ i=1..n };
            if t = {TRUE} then
                userinfo(1,"matrix is not invertible");
                return( FAIL )
            end_if;

            R_divide := R::_divide; R_negate := R::_negate;

            // transform the submatrix x(1..n,1..n) to identity matrix 
            // by backsolving
            for k from n downto 1 do
                t := x[k,k];
                for j from n+1 to 2*n do
                    x[k,j] := R_divide( x[k,j],t );
                    if x[k,j] = FAIL then
                        userinfo(1,"computation fails over its coefficient domain");
                        return( FAIL )
                    end_if
                end_for;
                for i from k-1 downto 1 do
                    t := R_negate( x[i,k] );
                    for j from n+1 to 2*n do
                        if not Riszero( x[k,j] ) then
                            x[i,j] := Rplus(x[i,j],Rmult(x[k,j],t))
                        end_if
                    end_for
                 end_for
            end_for;

            return(new( dom,n,n,
                array(1..n,1..n,[[x[i,j] $ j=n+1..2*n] $ i=1..n])
            ))
        end_proc

        elif R::hasProp( Cat::Ring ) then

          //------------------------------------------------
          // a special fast routine for modular inverses
          // over Dom::IntegerMod(N). Here, N is not a prime.
          //------------------------------------------------
          if R::constructor = Dom::IntegerMod then
             Dom::Utilities::IntModInverse;
          else
             //------------------------------------------------
             // Last resort:  Try to determine the inverse matrix 
             // via the formula 
             //
             //          det(x)^-1 * adjoint(x) 
             //
             // Beware: this is terribly slow!
             //------------------------------------------------
             proc(x) 
               local d; 
             begin
               // STRATEGY: 
               // Compute the inverse of the determinant first. 
               // If d = FAIL, the matrix is not invertible and 
               // there is no need to compute the adjoint matrix. 
               d:= linalg::det(x)^-1:
               if d = FAIL then 
                 return(FAIL);
               else 
                 return( d * linalg::adjoint(x) )
               end_if
             end_proc
          end_if;
        else
          FAIL
        end_if
    end_case;

/*--
    _divide  --  compute a/b for a matrix a.
--*/
    _divide:= if R::hasProp( Cat::Ring ) then
        proc(a,b)
        begin
            b:= 1/b;
            if b <> FAIL then 
                return( dom::_mult( a,b ) )
            else 
                userinfo(1,"inversion of the 2nd matrix can not be computed");
                return( FAIL )
            end_if
        end_proc
    end_if;


/*--
    _mod, mods, modp   -- map modulo operator to matrix

--*/

    _mod:=
    proc(A: dom, n: DOM_INT)
    begin
      if iszero(n) then
        error("Division by zero")
      end_if;
      map(A, _mod, n)
    end_proc;

    mods:=
    proc(A: dom, n: DOM_INT)
    begin
      if iszero(n) then
        error("Division by zero")
      end_if;
      map(A, mods, n)
    end_proc;

    modp:=
    proc(A: dom, n: DOM_INT)
    begin
      if iszero(n) then
        error("Division by zero")
      end_if;
      map(A, modp, n)
    end_proc;
      
      
/*--
    iszero  --  test if a matrix is the zero matrix 
--*/
    iszero:= if RhasSystemRep then
        x -> bool(map({op(extop(x,3))},iszero@Rnormal) = {TRUE});
    else
        x -> bool(map({op(extop(x,3))},Riszero) = {TRUE});
    end_if;

/*----------------------------------------------------------------------
  testeq  --  overloads the function 'testeq' for matrices 
----------------------------------------------------------------------*/
    testeq:= proc(x,y)
      local d;
    begin
        if x::dom <> dom then error("should not happen"); end_if;
        if domtype(x) <> domtype(y) then 
              y:= x::dom::coerce(y);
              if y = FAIL then return(FALSE) end_if;
              // do not try x:= y::dom::coerce(x), because
              // y::dom may lead to an error message (e.g., if y = a number)
        end_if;
        if extop(x,1) <> extop(y,1) then return(FALSE) end_if;
        if extop(x,2) <> extop(y,2) then return(FALSE) end_if;
        d:= map({op(x - y)}, elem -> testeq(elem, Rzero));
        if contains(d, FALSE) then return(FALSE) end_if;
        if contains(d, UNKNOWN) then return(UNKNOWN) end_if;
        return(TRUE);
    end_proc;

/*--
    equal  --  test if two matrices are equal 
--*/
    equal:= if R::equal = bool@_equal then 
        if RhasSystemRep then
            proc(x,y)
            begin
                if [extop(x,1),extop(x,2)] <> [extop(y,1),extop(y,2)] then
                    return( FALSE )
                else
                    return(bool( map(extop(x,3),Rnormal)=map(extop(y,3),Rnormal) ))
                end_if
            end_proc
        else
            bool@_equal
        end_if
    else
        proc(x,y)
            local r, c, i, j, t, e;
        begin
            r := extop(x,1); c := extop(x,2);
            if [r,c] <> [extop(y,1),extop(y,2)] then
                return( FALSE )
            end_if;
            x := extop(x,3); 
            y := extop(y,3);
            t := R::equal;
            for i from 1 to r do
                for j from 1 to c do
                    if (e:= bool(t(x[i,j],y[i,j]))) <> TRUE then return( e ) end_if
                end_for
            end_for;
            return( TRUE )
        end_proc
    end_if;

/*--
    gaussElim  --  Gaussian elimination

    Syntax:

    gaussElim(A)

    A  --  matrix

    Synopsis:

    Performs Gaussian elimination on A and returns a list with 
    a row echolon form of A, the rank of A, the determinant of 
    A if dom is defined (otherwise FAIL will be the third 
    operand of the list) and the set of characteristic indices.

    If R has a method "pivotSize" then the pivot element of
    smallest size will be choosen, whereby pivotSize must return
    a natural number representing 'the size' of an element.
--*/
    gaussElim:= if R::hasProp(Cat::Field) and R::hasProp( Ax::efficientOperation("_invert") ) then
        // -----------------------------
        // ordinary Gaussian elimination
        // -----------------------------
        proc(x)
            option remember;
            local det, m, n, i, j, l, o, k, t, p, ps, detsign, charind,
                  Rinvert, Rnegate, Rone, zeroTest;                             
            begin
            //===================================================================
            /* Zero test heuristic for matrices with floatable entries:
              ===================================================================
            
               For matrices whose components can be converted to floating point 
               enrties, we will use a special heuristic a zero test instead of 
               only using 'isnonzero'. This is the case of args(2) = `#FloatCheck`.  
            */
            //===================================================================
            if args(0) = 2 and args(2) = `#FloatCheck` then 
              zeroTest:= x -> if iszero(x) = TRUE then 
                                TRUE;
                              elif numeric::isnonzero(x) <> TRUE then 
                                TRUE;
                              else
                                FALSE;                             
                              end_if:
            else 
              zeroTest:= Riszero; 
            end_if:
            //===================================================================

            m:= extop(x,1); // number of rows
            n:= extop(x,2); // number of columns
            x:= extop(x,3);

            ps:= R::pivotSize; // if ps = FAIL then no pivot strategy will be used, otherwise
                               // the pivot in respect to < will be choosen
    
            Rnegate:= R::_negate;
            Rinvert:= R::_invert;
            Rone:= R::one;
    
            userinfo(1,"perform (ordinary) Gaussian elimination");
    
            j:= 1;
            detsign:= 1;  // sign of the determinant
            det:= Rone;   // the determinant 
            charind:= {}; // the characteristic column indices
            for i from 1 to n do
                if j = m then 
                    while i <= n do
                        if not zeroTest(Rnormal(x[j,i])) then 
                            charind:= charind union {i};
                            break
                        else
                            i:= i + 1
                        end_if
                    end_while;
                    break // i
                end_if;
    
                userinfo(3,"search for pivot in column ",i);
                p:= 0; k:= -1;
                for l from j to m do
                    if not zeroTest(Rnormal(x[l,i])) then 
                        if ps = FAIL then
                            p:= l;
                            break // l
                        elif k = -1 then
                            k:= ps(x[l,i]);
                            p:= l
                        elif (t:= ps(x[l,i])) < k then
                            k:= t;
                            p:= l
                        end_if
                    end_if
                end_for;
                if p = 0 then
                    userinfo(3,"no pivot found");
                    det:= Rzero;
                    next // i
                end_if;
        
                if p <> j then
                    userinfo(3,"swap rows: ",p, j);
                    x:= assignElements( x,([j,o]=x[p,o], [p,o]=x[j,o]) $ o=i..n );
                    detsign:= - detsign
                end_if;
     
                t:= x[j,i]; // the pivot element
    
                if m = n and det <> Rzero then det:= Rnormal(Rmult( det,t )) end_if;
    
                charind:= charind union {i};
    
                t:= Rinvert(t);
                // row operation
                for k from j+1 to m do
                    x:= assignElements( x,
                        ([k,o]=Rnormal(Rplus( x[k,o], Rnegate(Rmult(t,x[j,o],x[k,i])) ))) $ o=i+1..n
                    );
                    x[k,i]:= Rzero
                end_for;
    
                j:= j + 1
            end_for;
    
            if m = n then 
                if det <> Rzero then det:= Rnormal(Rmult( det,x[n,n] )) end_if;
                if detsign = -1 then 
                    return( [new(dom,m,n,x), nops(charind), Rnegate(det), charind] )
                else        
                    return( [new(dom,m,n,x), nops(charind), det, charind] )
                end_if
            else 
                return( [new(dom,m,n,x), nops(charind), FAIL, charind] )
            end_if
        end_proc
    elif R::hasProp( Cat::IntegralDomain ) then
        // -------------------------------------------
        // two-step fraction free Gaussian elimination
        // -------------------------------------------
        proc(x)
            option remember;
            local n, m, i, j, l, f, f0, f1, f2, r, k, o, p, l0, ps, sig, charind, 
                  Rnegate, Rdivide;
        begin
            Rnegate := R::_negate; 
            Rdivide := (if R::constructor=Dom::DistributedPolynomial and R::quo <> FAIL then 
                           R::quo // we know the division is always exact here,
                                  // and R::_divide may return FAIL for Dom::Numerical
                       else 
                            R::_divide 
                       end_if);

            ps := R::pivotSize; // if ps = FAIL then no pivot strategy will be used, otherwise
                                // the pivot in respect to < will be choosen

            m := extop(x,1); // number of rows
            n := extop(x,2); // number of columns
            x := extop(x,3);

            userinfo(1,"perform 2-step fraction free Gaussian elimination");

            sig := 1;      // the sign of the determinant
            f0 := R::one;  // the determinant

            charind:= {k $ k=1..n}; // the characteristic column indices

            k := 2; 
            r := 2;
            for k from 2 to 2*(n div 2) step 2 do
                if r > m then break end_if;

                userinfo(3,"search for non-zero determinant");
                l0:= p:= FAIL; // initial value
                for i from r-1 to m do
                    f1 := Rnegate( x[i,k] );
                    f2 := x[i,k-1];
                    for j from i+1 to m do 
                        f := Rplus( Rmult( f2,x[j,k] ), Rmult( f1,x[j,k-1] ) );
                        if not Riszero(Rnormal(f)) then
                            if ps = FAIL then
                                p := [i,j,f];
                                break
                            elif l0 = FAIL then
                                l0 := ps(f);
                                p := [i,j,f]
                            elif (l:=ps(f)) < l0 then
                                l0 := l;
                                p := [i,j,f]
                            end_if
                        end_if
                    end_for;
                    if j <= m and ps = FAIL then break end_if
                end_for;
                if p = FAIL then
                    userinfo(3,"  non non-zero determinant found in step ",k);
                    // initialize single step
                    if r = 2 then f0 := R::one else f0:= x[r-2,k-2] end_if;
                    break
                end_if;
                if p[1] <> r-1 then 
                    i := p[1];
                    userinfo(3,"swap rows ",r-1,i);
                    x:= assignElements( x, ( [r-1,o]=x[i,o], [i,o]=x[r-1,o]) $ o=k-1..n );
                    sig := -sig
                end_if;
                if p[2] <> r then
                    j := p[2];
                    userinfo(3,"swap rows ",j,r);
                    x:= assignElements( x, ( [r,o]=x[j,o], [j,o]=x[r,o] ) $ o=k-1..n );
                    sig := -sig
                end_if;

                // x[r-1,k-1] = 0 ?
                if Riszero( Rnormal(x[r-1,k-1]) ) then
                    userinfo(3,"swap rows ",r-1,r);
                    x:= assignElements( x, ( [r-1,o]=x[r,o], [r,o]=x[r-1,o] ) $ o=k-1..n );
                    p[3] := Rnegate( p[3] ); // |0 a\\c d| = -ac = -|c d\\0 a|
                    sig := -sig
                end_if;

                f := Rdivide(p[3],f0);
                if f = FAIL then
                    error("no exact division (numerical errors occurred?)")
                end_if;
                for i from r+1 to m do
                    f1 := Rnormal(Rdivide( 
                            Rplus( Rmult( x[r-1,k],x[i,k-1] ),
                            Rnegate(Rmult( x[r-1,k-1],x[i,k] )) ),
                          f0 ));
                    f2 := Rnormal(Rdivide( 
                            Rplus( Rmult( x[r,k-1],x[i,k] ), Rnegate( Rmult( x[r,k],x[i,k-1] )) ),
                          f0 ));
                    for j from k+1 to n do
                        x[i,j] := Rnormal(Rdivide(
                            Rplus(Rmult(f,x[i,j]),Rmult(f1,x[r,j]), Rmult(f2,x[r-1,j]) ),
                            f0
                        ));
                    end_for;
                    x[i,k] := Rzero; x[i,k-1] := Rzero
                end_for;
                for j from k+1 to n do
                    x[r,j] := Rnormal(Rdivide( 
                        Rplus( Rmult( x[r-1,k-1], x[r,j] ),
                        Rnegate(Rmult( x[r-1,j],x[r,k-1] )) ),
                    f0 ))
                end_for;
                x[r,k-1] := Rzero; x[r,k] := f;
                // new divisor
                f0 := f; 
                // next step
                r := r + 2
            end_for;
    
            r:= r - 2;
    
            if n mod 2 = 1 or r <= n then
                userinfo(2,"perform single-step algorithm");
                f1 := k - 1; 
                r := r + 1;
                for k from f1 to n do
                    if r > m then 
                        break 
                    end_if;
    
                    userinfo(2,"searching for pivot element in row/column ",r,k);
                    p := m+1; l0:= FAIL; // initial value
                    for i from r to m do
                        if not Riszero(Rnormal(x[i,k])) then
                            if ps = FAIL then p := i; break
                            elif l0 = FAIL then l0 := ps(x[i,k]); p := i
                            elif (l:=ps(x[i,k])) < l0 then l0 := l; p := i 
                            end_if
                        end_if
                    end_for;
                    if p <= m then
                        if r <> p then 
                            userinfo(3,"swap rows ",r,p);
                            x:= assignElements( x,( [r,l]=x[p,l], [p,l]=x[r,l] ) $ l=k..n );
                            sig := -sig
                        end_if;
    
                        for i from r+1 to m do
                            f2:= Rnegate( x[i,k] );
                            for j from k+1 to n do
                                x[i,j] := Rnormal(Rdivide( 
                                   Rplus( Rmult( x[r,k],x[i,j] ), Rmult( x[r,j],f2 ) ),f0 
                                ))
                            end_for;
                            x[i,k] := Rzero
                        end_for;
                        f0 := x[r,k]; 
                        r := r + 1
                    else
                        charind:= charind minus {k}
                    end_if
                end_for;
    
                charind:= charind minus {l $ l = k..n};
    
                r := r - 1
            end_if;
      
            if n = m then
                if r = m then if sig = -1 then f0 := Rnegate(f0) end_if
            else 
                f0 := Rzero
            end_if
        else
            f0 := FAIL
        end_if;

        [new(dom,m,n,x),r,f0,charind]
    end_proc
end_if;

/*--
    transpose  --  compute the transpose of a matrix 
--*/
    transpose:= proc(x)
        local r, c, i, j, t;
    begin
        r := extop( x,1 ); c := extop( x,2 ); x := extop( x,3 );
        t := array( 1..c,1..r );
        ((t[j,i] := indexval(x,i,j)) $ j=1..c) $ i=1..r;

        new( dom,c,r,t )
    end_proc;

/*--
    nonZeros  --  return the number of non-zero elements of a matrix 
--*/
    nonZeros:= if RhasSystemRep then
        x -> nops( select([op(extop(x,3))],_not@iszero@Rnormal) );
    else
        x -> nops( select([op(extop(x,3))],_not@Riszero) );
    end_if;

/*--
    nonZeroes  --  return the number of non-zero elements of a matrix 
--*/
    nonZeroes:= if RhasSystemRep then
        x -> nops( select([op(extop(x,3))],_not@iszero@Rnormal) );
    else
        x -> nops( select([op(extop(x,3))],_not@Riszero) );
    end_if;


/*--
    stackMatrix  --  concat two matrices vertically 
--*/
    stackMatrix:= proc(a,b)
        local rb, r, c, t, j, i;
    begin
        case args(0)
        of 0 do return( null() )
        of 1 do return( a )
        of 2 do break
        otherwise
            return( dom::stackMatrix(a,dom::stackMatrix(b,args(3..args(0)))) )
        end_case;

        // the case of two arguments:
        r := extop( a,1 ); c := extop( a,2 ); a := extop( a,3 );
        rb := extop( b,1 ); b := extop( b,3 );
        t := array(1..r+rb,1..c);
        (( t[i,j] := indexval(a,i,j) ) $ i=1..r;
        ( t[r+i,j] := indexval(b,i,j) ) $ i=1..rb) $ j=1..c;

        new( dom,r+rb,c,t )
    end_proc;

/*--
    concatMatrix  --  append two matrices horizontally 
--*/
    concatMatrix:= proc(a,b)
        local cb, r, c, t, i, j;
    begin
        case args(0)
        of 0 do return( null() )
        of 1 do return( a )
        of 2 do break
        otherwise
            return( dom::concatMatrix(a,dom::concatMatrix(b,args(3..args(0)))) )
        end_case;

        // the case of two arguments:
        r := extop( a,1 ); c := extop( a,2 ); a := extop( a,3 );
        cb := extop( b,2 ); b := extop( b,3 );
        t := array(1..r,1..c+cb);

        (( t[i,j] := indexval(a,i,j) ) $ j=1..c;
         ( t[i,c+j] := indexval(b,i,j) ) $ j=1..cb ) $ i=1..r;

        new( dom,r,c+cb,t )
    end_proc;

/*--
    setRow  --  replace a row of a matrix by a new one 
--*/
    setRow:= proc(x,p,row)
        local c, j, pp, Rgn;
    begin
        if extop(row,2) <> extop(x,2) then 
          error("wrong number of columns of the row vector");
        end_if;
        if x::dom <> row::dom and x::dom::constructor <> Dom::SquareMatrix then 
          if traperror((row:= x::dom::new(row))) <> 0 then 
            error("cannot convert row vector to a vector of type ".expr2text(x::dom));
          end_if;
        elif x::dom::constructor = Dom::SquareMatrix then 
          Rgn:= x::dom::coeffRing;
          if has(map({op(row)}, elem -> coerce(elem,Rgn)), FAIL) then 
            error("cannot convert coefficient ring of the row vector to ".expr2text(x::dom::coeffRing));
          end_if;
        end_if;
        //if traperror((row:= x::dom::new(row))) <> 0 then  
        //  error("cannot convert row vector to a vector of type ".expr2text(x::dom));
        //end_if;
        row:= extop( row,3 );
        c:= extop( x,2 );
        pp:= (p-1)*c;
        extsubsop( x,3=subsop(extop(x,3),
           pp+j=indexval(row,1,j) $ j=1..c,hold(Unsimplified)
        ))
    end_proc;

/*--
    setCol  --  replace a column of a matrix by a new one 
--*/
    setCol:= proc(x,p,cll)
        local c, i, col, Rgn;
    begin
        if extop(cll,1) <> extop(x,1) then 
          error("wrong number of columns of the row vector");
        end_if;
        if x::dom <> cll::dom and x::dom::constructor <> Dom::SquareMatrix then 
          if traperror((cll:= x::dom::new(cll))) <> 0 then 
            error("cannot convert column vector to a vector of type ".expr2text(x::dom));
          end_if;
        elif x::dom::constructor = Dom::SquareMatrix then 
          Rgn:= x::dom::coeffRing;
          if has(map({op(cll)}, elem -> coerce(elem,Rgn)), FAIL) then 
            error("cannot convert coefficient ring of the column vector to ".expr2text(x::dom::coeffRing));
          end_if;
        end_if;
        //if traperror((cll:= x::dom::new(cll))) <> 0 then  
        //  error("cannot convert row vector to a vector of type ".expr2text(x::dom));
        //end_if;
        c:= extop(x,2);
        col:= extop( cll,3 );
        extsubsop( x,3=subsop(
            extop(x,3),p+i*c = indexval(col,i+1,1) $ i=0..extop(x,1)-1
            ,hold(Unsimplified)
        ))
    end_proc;

/*--
    row  --  extract a row of a matrix 
--*/
    row:= proc(A,p)
        local c, j, x;
    begin
        c:= extop( A,2 );
        x:= extop( A,3 );
        new( dom,1,c,array( 1..1,1..c,[[indexval(x,p,j) $ j=1..c]] ) )
    end_proc;

/*--
    col  --  extract a column of a matrix 
--*/
    col:= proc(A,p)
        local r, i, x;
    begin
        r:= extop( A,1 ); 
        x:= extop( A,3 );
        new( dom,r,1,array( 1..r,1..1,[ [indexval(x,i,p)] $ i=1..r ] ) )
    end_proc;

/*--
    delRow  --  delete a row of a matrix 
--*/
    delRow:= proc(A,p)
        local r, c, t, j, i, x;
    begin
        r:= extop( A,1 ); 
        c:= extop( A,2 );
        if r = 1 then return( null() ) end_if;
        x:= extop( A,3 );

        t:= array( 1..r-1,1..c );
        ( ( t[i,j] := indexval(x,i,j) ) $ j=1..c ) $ i=1..p-1;
        ( ( t[i-1,j] := indexval(x,i,j) ) $ j=1..c ) $ i=p+1..r;

        new( dom,r-1,c,t )
    end_proc;

/*--
    delCol  --  delete a column of a matrix 
--*/
    delCol:= proc(xx,p)
        local r, c, t, i, j, x;
    begin
        r:= extop( xx,1 ); 
        c:= extop( xx,2 );
        if c = 1 then return( null() ) end_if;
        x:= extop( xx,3 );

        t:= array( 1..r,1..c-1 );
        ( ( t[i,j] := indexval(x,i,j) ) $ i=1..r ) $ j=1..p-1;
        ( ( t[i,j-1] := indexval(x,i,j) ) $ i=1..r ) $ j=p+1..c;

        new( dom,r,c-1,t )
    end_proc;

/*--
    swapRow  --  swap two rows of a matrix 
--*/
    swapRow:= proc(x,k,l,b)
        local a, j;
    begin
        if k = l then return( x ) end_if;

        a:= extop( x,3 );
        if args(0) = 3 then
            a:= assignElements(a, 
                ([[l,j],indexval(a,k,j)], [[k,j],indexval(a,l,j)]) $ j=1..extop(x,2)
            );
        else
            a:= assignElements(a, 
                ([[l,j],indexval(a,k,j)], [[k,j],indexval(a,l,j)]) $ j=op(b,1)..op(b,2)
            );
        end_if;

        extsubsop( x,3=a )
    end_proc;

/*--
    swapCol  --  swap two columns of a matrix 
--*/
    swapCol:= proc(x,k,l,b)
        local a, i;
    begin
        if k = l then return( x ) end_if;

        a:= extop( x,3 );
        if args(0) = 3 then
            a:= assignElements(a, ([[i,l], indexval(a,i,k)],
                                  [[i,k], indexval(a,i,l)]) $ i=1..extop(x,1)) ;
        else
            a:= assignElements(a, ([[i,l], indexval(a,i,k)], 
                                  [[i,k], indexval(a,i,l)]) $ i=op(b,1)..op(b,2)) ;
        end_if;

        extsubsop( x,3=a )
    end_proc;

/*--
    assignElements  --  overloads the function 'assignElements'
--*/
    assignElements:= x -> extsubsop( x,3=assignElements( extop(x,3),args(2..args(0)) ) );

/*--
    TeX  --  'TeX' for matrices 
--*/
    TeX:= x -> "\\left(".generate::TeX(extop(x,3))."\\right)";

/*--
    map  --  'map' for matrices 
--*/
    map:= proc(x)
        local y;
    begin
        y:= map(extop(x,3),args(2..args(0)));
        y:= map(y,Rcoerce);
        if contains({op(y)},FAIL) then
           error("operation lead to invalid matrix component(s)")
        end_if;
        return(extsubsop( x,3=y ))
    end_proc;


// needed for misc::maprec
enableMaprec:= TRUE;


/*--
    mapNonZeroes  --  'mapNonZeroes' for matrices.  
--*/
    mapNonZeroes:= proc(x)
        local seq, y;
    begin
        seq:= args(2..args(0));
        y:= map(extop(x,3), ind -> if ind = Rzero then ind else map(ind, seq) end_if);
        if testargs() then
            // check correct type only at the interactive level
            y:= map(y,Rcoerce);
            if contains({op(y)},FAIL) then
                error("operation lead to invalid matrix component(s)")
            end_if
        end_if;
        return(extsubsop( x,3=y ))
    end_proc;

/*--
    nonZeroOperands  --  'nonZeroOperands' for matrices.  
--*/
    nonZeroOperands:= x -> map(op(extop(x,3)), ind -> if iszero(ind) then null() else ind end_if);


/*--
    subs  --  'subs' for matrices 
--*/
    subs:= proc(x)
        local y;
    begin
        y:= subs( extop(x,3),args(2..args(0)) );
        if testargs() then
            // check correct type only at the interactive level
            y:= map(y,Rcoerce);
            if contains({op(y)},FAIL) then
                error("substitution lead to invalid matrix components")
            end_if
        end_if;
        return( extsubsop(x,3=y) )
    end_proc;
 
/*--
    subsex  --  'subsex' for matrices 
--*/
    subsex:= proc(x)
        local y;
    begin
        y:= subsex( extop(x,3),args(2..args(0)) );
        if testargs() then
            // check correct type only at the interactive level
            y:= map(y,Rcoerce);
            if contains({op(y)},FAIL) then
                error("substitution lead to invalid matrix components")
            end_if
        end_if;
        return( extsubsop(x,3=y) )
    end_proc;

/*--
    subsop  --  'subsop' for matrices
--*/
    subsop:= proc(x)
        local y;
    begin
        y:= subsop( extop(x,3),args(2..args(0)) );
        if testargs() then
            // check correct type only at the interactive level
            y:= map(y,Rcoerce);
            if contains({op(y)},FAIL) then
                error("substitution lead to invalid matrix components")
            end_if
        end_if;
        return( extsubsop(x,3=y) )
    end_proc;

/*--
    evalAt -- the operator | for matrices
--*/
    evalAt:= x -> dom::map(x, evalAt, args(2..args(0)));

/*--
    has  --  the function 'has' for matrices
--*/
    has:= x -> has(extop(x,3),args(2));

/*--
    simplify  --  simplification of matrix elements
--*/
    simplify:= x -> dom::map(x, simplify, args(2..args(0)));

    Simplify:= x -> dom::map(x, Simplify, args(2..args(0)));

/*--
    normal  --  normalization of matrix elements
--*/
    normal:= if Rnormal = Rid then
        Rid
    else
        x -> extsubsop(x,3=map(extop(x,3), Rnormal, args(2..args(0))))
    end_if;

/*--
    length  --  returns the lenght of a matrix 

    The length of a matrix is defined by the length of its 3rd operand, i.e.,
    the length of an array.
--*/
    length:= x -> length(extop(x,3));

/*--
    abs        --  compute the absolute value of matrix entries
    conjugate  --  compute the complex conjugate of a matrix 
    Re, Im     --  real and imaginary part of a matrix
--*/
/* old (through MuPAD 4.01)
    conjugate:= if R::conjugate <> FAIL then 
        x -> extsubsop(x,3=map(extop(x,3),R::conjugate))
    else id end_if;
*/

    abs:= x -> map(x, abs);
    conjugate:= x -> map(x, conjugate);
    Re:= x -> map(x, Re);
    Im:= x -> map(x, Im);

/*--
    random  --  create random matrices

    Syntax:

    random()

    Synopsis:

    Returns a random matrix whereby the dimension is randomly choosen
    in the interval [1..dom::randomDimen[1],1..dom::randomDimen[2]]
    The elements of the random matrix are created by the method
    "random" of R. If such a method does not exist, then FAIL is 
    returned.
--*/
    random:= if R::random <> FAIL then 
    proc()
        local m, n, j, i;
    begin
        m:= (random() mod dom::randomDimen[1]) + 1;
        n:= (random() mod dom::randomDimen[2]) + 1;
        new( dom,m,n,array( 1..m,1..n,
           [[R::random() $ j=1..n] $ i=1..m]
        ))
    end_proc
    end_if;

/*--
    tr  --  compute the trace of a square matrix

    Syntax:

    tr(A)

    A  --  matrix

    Synopsis:

    A trace of a square matrix A is defined to be the sum of the
    diagonal elements of A.
--*/
    tr:= proc(x)
        local n, i;
    begin
        n:= extop(x,1);
        if n <> extop(x,2) then
            error("expecting a square matrix")
        end_if;
        x:= extop(x,3);
        Rplus( Rzero,x[i,i] $ i=1..n )
    end_proc;

/*--
    norm()  --  compute the norm of a matrix

    Syntax: 

    norm(A)
    norm(A,Frobenius)
    norm(v,k)

    A                    --  matrix
    v                    --  vector
    k                    --  positive integer
    Frobenius, Infinity  --  identifier

    Synopsis:

    norm(A) and norm(A,Infinity) compute the infinity norm of a matrix.
    That is the maximum row sum, where the row sum is the sum of norms
    of each element.
    The infinity norm of a vector is the maximum norm of all
    elements.

    norm(A,1) computes the 1-norm of a matrix which is the maximum sum 
    of the norm of the elements of each column.

    norm(A,Frobenius) computes the Frobenius norm of a matrix which is 
    the square root of the sum of the squares of the norm of each element.

    norm(v,k) computes the k-norm of a vector for k >= 2.
    Dom norm is defined to be the kth root of the sum of the norm of each 
    element raised to the kth power.
    For the 2-norm the function 'linalg::scalarProduct' will be used
    if it is defined. Otherwise the Euclidian scalar product of v will
    be returned, which is defined by

        v[1]*conjugate(v[1]) + v[2]*conjugate(v[2]) + ...

    If the coeffRing of v does not provide a method "conjugate"
    (which must return the complex conjugate of an element) then 
    the sum

            v[1]*v[1] + v[2]*v[2] + ...

    will be returned.


    Dom method returns FAIL if the result is not longer an element
    of the coefficient ring of A and v, respectively.

    For all cases except of the 2-norm, the coefficient ring of the
    matrix has to provide the method 'norm', which result for any element 
    has to be a number. FAIL will be returned if such a method not exists.
--*/
    norm:= proc(x, normtype=hold(Infinity))
        local i, j, c, r, n, Rnorm;
    begin
        if args(0) < 1 or args(0) > 2 then
            error("wrong no of args")
        end_if;

        if domtype(normtype) = DOM_FLOAT and iszero(frac(normtype)) then
           normtype:= round(normtype);
        end_if:

        if normtype <> 2 then
            Rnorm := R::norm;
            if Rnorm = FAIL then
                userinfo(1,"the coefficient domain has not the method \"norm\"");
                return( FAIL )
            end_if
        end_if;
    
        r:= extop(x,1);
        c:= extop(x,2);
        case normtype
        of 1 do 
            x:= dom::transpose(x);
            n:= c; c := r; r := n
    
        of hold(Infinity) do
        of hold(Maximum) do 
            x:= extop(x,3);
            n:= stdlib::max(_plus(Rnorm(x[i,j]) $ j=1..c) $ i=1..r);
            break

        of hold(Frobenius) do 
            x:= extop(x,3);            
            n:= _plus((_mult(Rnorm(x[i,j]) $ 2) $ j=1..c) $ i=1..r);
            n:= n^(1/2);
            break
        of 2 do 
        of hold(Euclidean) do 
        of hold(Spectral) do
            // compute the 2-norm of vector x:
            if stdlib::min(r,c) <> 1 then
                error("invalid matrix norm (expecting a vector)")
            end_if;
    
            normtype:= linalg::scalarProduct;
            if normtype <> FAIL then
                n:= normtype(x,x)^(1/2)
            else
                x:= extop(x,3);
                normtype:= R::conjugate;
                if normtype <> FAIL then
                    if r = 1 then
                        n:= Rplus( Rmult(x[1,i],normtype(x[1,i])) $ i=1..c )
                    else
                        n:= Rplus( Rmult(x[i,1],normtype(x[i,1])) $ i=1..r )
                    end_if
                elif r = 1 then
                    n:= Rplus( Rmult(x[1,i],x[1,i]) $ i=1..c )
                else // c = 1
                    n:= Rplus( Rmult(x[i,1],x[i,1]) $ i=1..r )
                end_if;
                n:= n^(1/2)
            end_if;
            break

        otherwise 
            // compute the k-norm of vector x:
            if not testtype( normtype,Type::PosInt ) then
                error("invalid matrix norm")
            end_if;
            if r = 1 then
                x:= extop(x,3);
                n:= _plus( Rnorm(x[1,i])^normtype $ i=1..c )^(1/normtype)
            elif c = 1 then
                x:= extop(x,3);
                n:= _plus( Rnorm(x[i,1])^normtype $ i=1..r )^(1/normtype)
            else
                error("invalid matrix norm (expecting a vector)")
            end_if
        end_case;

        return( n )
    end_proc;

    exp:= if R::hasProp( Cat::Field ) then 
        /*--    
            exp  --  compute the exponential of a matrix over a field
    
            Syntax:
            exp(A <,x>)

            A: square matrix
            x: element of R

            Synopsis:
            exp(A,x) computes the matrix exponential defined by
            exp(A,x) = I + A*x + 1/2!*A^2*x^2 + ...
            If the second argument is not given then 1 is set for x.
            When the eigenvalues of A can not be computed then an 
            error message will be returned.

            Background: 
            This method first computes the Jordan form of A such that 
            A = 1/P*A*P. Then exp(A) = P*exp(J)*1/P, where exp(J) = 
            ( exp(J[1]),...,exp(J[k]) ) with the k Jordan blocks J[i]. 
            Suppose that

                     [ 0, 1, 0 ]
              J[i] = [ 0, 0, 1 ],
                     [ 0, 0, 0 ]

            then we get:

                          [ 1, t, 1/2*t^2]
              exp(J[i]) = [ 0, 1,    t   ].
                          [ 0, 0,    1   ]
        --*/
        proc(A,t=R::one)
            local k, i, j, x, e, n, r, J, P, Rone, Cols, b;
        begin
            if args(0) < 1 or args(0) > 2 then
                error("wrong no of args")
            end_if;
            n:= extop(A,1);
            if n <> extop(A,2) then
                error("expecting a square matrix")
            elif args(0) = 2 then
                if domtype(t) <> R then
                    t:= Rcoerce( t );
                    if t = FAIL then
                        error("2nd argument is not compatible with the component ring");
                    end_if
                end_if
            end_if;

            if R = Dom::Float and iszero(t-1) then
                userinfo(1,"call 'numeric::expMatrix'");
                return(extsubsop( A,3=numeric::expMatrix(extop(A,3)) ))
            end_if;

            k:= linalg::jordanForm( A,hold(All) );
            if k = FAIL then
                userinfo(1,"linalg::jordanForm fails");
                return( FAIL )
            end_if;
            [J,P]:= k;

            Rone:= R::one;
            k:= 1;
            while k <= n do
                e:= J[k,k]; // eigenvalue of A
                // Compute the exponential of a jordan block:
                x:= Rcoerce( exp(Rmult(t,e)) );
                if x = FAIL then
                   userinfo(1,"cannot compute the exponential of a ring element");
                   return( FAIL )
                end_if;
                // get the length of the corresponding Jordan chain:
                i:= k;
                while i < n and J[i,i] = e and J[i,i+1] = Rone do i:= i + 1 end_while;
                for r from k to i do
                    J[r,r]:= x;
                    if r > 1 and not Riszero( J[r-1,r] ) then
                        e:= Rmult( J[r,r],J[r-1,r] );
                        (J[r-j,r] := Rmult( e,t^j/fact(j) )) $ j=1..r-k
                    end_if
                end_for;
                k:= r
            end_while;
            //=========================================
            if n < 7 and max(map({op(P)},length)) > 100 then 
              Cols:= [0 $ n];
              for i from 1 to n do 
                b:= P::dom(n,1);
                b[i]:= 1;
                Cols[i]:= linalg::SSS(P,b)[1];
              end_for;  
              return(P*J*(P::dom::concatMatrix(op(Cols))));
            else   
              return(dom::_mult(P,J,dom::_invert(P)) )
            end_if;          
            //=========================================
        end_proc
    else
        // exp(A) = FAIL, if A is not defined over a 'Cat::Field':
        () -> FAIL
    end_if;


/*----------------------------------------------

sin -- the sine for matrices

-----------------------------------------------*/

sin:= proc(A,t=R::one)
   local n, B;
   begin
   if R::hasProp( Cat::Field ) then 
     /*--    
     If there are foating point numbers within the 
     matrix we try to compute sin(A) with the help 
     'numeric::exp'. The strategy of searching for 
     floating point numbers within the matrix A is 
     the same as above in the procedure 'exp'. 
     --*/
     if args(0) < 1 or args(0) > 2 then
         error("wrong no of args")
     end_if;
     n:= extop(A,1);
     if n <> extop(A,2) then
         error("expecting a square matrix")
     elif args(0) = 2 then
         if domtype(t) <> R then
             t:= Rcoerce( t );
             if t = FAIL then
                 error("2nd argument is not compatible with the component ring");
             end_if
         end_if
     end_if;
     
     if (B:= linalg::checkForFloats(A)) <> FALSE and iszero(t-1) then
        userinfo(1,"call 'numeric::expMatrix'");
        return(dom::convert(map(
                (1/2 * I) * (numeric::expMatrix(-I*A) - numeric::expMatrix(I*A)), 
                numeric::complexRound
               )));
     end_if;

     if R::hasProp(Dom::ExpressionField) or R::hasProp(Dom::Complex) then
       return(map((1/2 * I) * (exp(-I*A) - exp(I*A)), rewrite, sincos))
     end_if;
   else 
     return(FAIL);
   end_if;
   end_proc;

/*
  sin:= proc(A) 
        begin
            if R::hasProp(Dom::ExpressionField) then
            expXA:= exp(`#X` * A);
            // use sin(z) = 1/2 I (exp(-Iz) - exp(Iz))
            // works also if exp failed, since 1/2*I*(FAIL - FAIL) = FAIL
            return(map((1/2 * I) * (subs(expXA, `#X` = -I) - subs(expXA, `#X` = I)),
                       rewrite, sincos));
            // map((1/2 * I) * (exp(-I*A) - exp(I*A)), rewrite, sincos)
          if R::hasProp(Dom::ExpressionField) or R::hasProp(Dom::Complex) then 
            return(map((1/2 * I) * (exp(-I*A) - exp(I*A)), rewrite, sincos));
          else
            return(FAIL);
          end_if; 
      end_proc;
*/

/*--
cos -- the cosine for matrices
--*/

cos:= proc(A,t=R::one)
   local n, B;
   begin
   if R::hasProp( Cat::Field ) then 
     /*--    
     If there are foating point numbers within the 
     matrix we try to compute sin(A) with the help 
     'numeric::exp'. The strategy of searching for 
     floating point numbers within the matrix A is 
     the same as above in the procedure 'exp'. 
     --*/
     if args(0) < 1 or args(0) > 2 then
         error("wrong no of args")
     end_if;
     n:= extop(A,1);
     if n <> extop(A,2) then
         error("expecting a square matrix")
     elif args(0) = 2 then
         if domtype(t) <> R then
             t:= Rcoerce( t );
             if t = FAIL then
                 error("2nd argument is not compatible with the component ring");
             end_if
         end_if
     end_if;
     
     if (B:= linalg::checkForFloats(A)) <> FALSE and iszero(t-1) then
        userinfo(1,"call 'numeric::expMatrix'");
        return(dom::convert(map(
                1/2 * (numeric::expMatrix(-I*A) + numeric::expMatrix(I*A)), 
                numeric::complexRound
               )));
     end_if;

     if R::hasProp(Dom::ExpressionField) or R::hasProp(Dom::Complex) then
       return(map(1/2 * (exp(-I*A) + exp(I*A)), rewrite, sincos))
     end_if;
   else 
     return(FAIL);
   end_if;
   end_proc;

/*
  cos:= proc(A) 
        begin
            if R::hasProp(Dom::ExpressionField) then
            expXA:= exp(`#X` * A);
            // use sin(z) = 1/2 I (exp(-Iz) - exp(Iz))
            // works also if exp failed, since 1/2*I*(FAIL - FAIL) = FAIL
            return(map(1/2 * (subs(expXA, `#X` = -I) + subs(expXA, `#X` = I)),
                       rewrite, sincos));
            // map((1/2 * I) * (exp(-I*A) - exp(I*A)), rewrite, sincos)
          if R::hasProp(Dom::ExpressionField) or R::hasProp(Dom::Complex) then 
            return(map(1/2 * (exp(-I*A) + exp(I*A)), rewrite, sincos));
          else
            return(FAIL);
          end_if; 
      end_proc;
*/
      /*cos:=
      if R::hasProp(Dom::ExpressionField) or R::hasProp(Dom::Complex) then
        proc(A)
          local expXA;
        begin
          // expXA:= exp(`#X` * A);
          // use sin(z) = 1/2 I (exp(-Iz) - exp(Iz))
          // this works also if exp failed
          // L:= map(1/2 * (subs(expXA, `#X` = -I) + subs(expXA, `#X` = I)),
          //         rewrite, sincos);
          map(1/2 * (exp(-I*A) + exp(I*A)), rewrite, sincos)
        end_proc
      else
        () -> FAIL
      end_if; */

// -------------------------------------------------------------------
// sqrt:  overload 'sqrt'
// -------------------------------------------------------------------

sqrt:= A -> linalg::sqrtMatrix(A,args(2..args(0)));
     
      
/*--
    zip  --  overload the function zip for matrices
--*/
    zip:= proc(a,b,f)
        local ia, ib, ic, ra, rb, ca, cb, i, j, rmin, cmin, xfarg;
    begin
        xfarg:= args(4..args(0));

        ia := extop( a,3 ); ib := extop( b,3 );
        ra := extop( a,1 ); ca := extop( a,2 );
        rb := extop( b,1 ); cb := extop( b,2 );
        rmin := stdlib::min(ra,rb); 
        cmin := stdlib::min(ca,cb);

        ic := array( 1..rmin,1..cmin );
        for i from 1 to rmin do
            (ic[i,j] := f(ia[i,j],ib[i,j],xfarg)) $ j=1..cmin
        end_for;

        if testargs() then
            // check correct type only at the interactive level
            ic:= map( ic,Rcoerce );
            if contains( {op(ic)},FAIL ) then
                error("operation lead to invalid matrix component(s)")
            end_if
        end_if;
        new( dom,rmin,cmin,ic )
    end_proc;

/*--
    expr2text  --  convert a matrix to a string
--*/
    expr2text :=
    if R = Dom::ExpressionField() then
      x -> "densematrix(".expr2text(coerce(x,DOM_LIST)).")";
    else
      x -> expr2text(dom::key)."(".expr2text(coerce(x,DOM_LIST)).")";
    end_if;

/*--
    op  --  overload the function 'op' as it operates on arrays
--*/
    op:= x -> op(extop(x,3),args(2..args(0)));

/*--
    nop  --  overload the function 'nops' as it operates on arrays
--*/
    nops:= x -> nops(extop(x,3));

/*--
    expand  --  overload the function 'expand'
--*/
    expand:= if R::expand <> FAIL then
        x -> extsubsop(x,3= map(extop(x,3), R::expand, args(2..args(0))))
    elif RhasSystemRep then
        x -> extsubsop(x,3= map(extop(x,3), expand, args(2..args(0))))
    end;

/*--
    diff  --  overload the function 'diff'
--*/
    diff:= if R::diff <> FAIL then
        x -> extsubsop(x,3= map(extop(x,3), diff, args(2..args(0))))
    end;

/*--
    int  --  overload the function 'int'
--*/
    int:= if R::int <> FAIL then
        x -> extsubsop(x,3= map(extop(x,3), int, args(2..args(0))))
    end_if;

/*--
    factor -- try to write a matrix A as s*B, where s is an element 
              of the coefficient ring
--*/
    factor:= if R::hasProp(Cat::GcdDomain) then
        proc(A)
            local g, r, c;
        begin
            if R = Dom::ExpressionField(normal, iszero@normal) then
               g:= R(gcd( map(op(extop(A,3)), expr)));
            else 
               g:= gcd( op(extop(A,3)));
            end_if;
            if g = R::one or g = FAIL then
                return( Factored::create( [1,A,1], "unknown"))
            elif g = Rzero then 
                r:= extop(A,1);
                c:= extop(A,2);
                return( Factored::create( [1,new(dom,r,c,array(1..r,1..c,[[Rzero $ c] $ r])),1], "unknown"))
            else
               if R = Dom::ExpressionField(normal, iszero@normal) then 
                  A:= extsubsop( A,3=map(extop(A,3),R::_divide,g) );
                  return(Factored::create( [g,A,1],"unknown" ))
               else 
                  A:= extsubsop( A,3=map(extop(A,3),Rnormal@R::_divide,g) );
                  return(Factored::create( [g,A,1],"unknown" ))
               end_if;
            end_if
        end_proc
    end_if;

/*--
    evalp -- overload 'evalp' for matrices over polynomial coefficient domains
--*/
    evalp:= if R::hasProp(Cat::Polynomial) then
    proc(A)
        local x, y;
    begin
        x:= map(extop(A,3),eval@evalp,args(2..args(0)));
        // Check whether the matrix now consists only of components 
        // of the coefficient ring of the polynomial domain:
        y:= map({op(x)},testtype,R::coeffRing);
        if contains(y,FALSE) then
            // No, hence define the matrix over R:
            return(new( dom,extop(A,1..2),x ))
        else
            // Yes, hence define the matrix over R::coeffRing:
            return(new( Dom::DenseMatrix(R::coeffRing),extop(A,1..2),x ))
        end_if
    end_proc
    end_if;

/*--
    identity(n) -- return the n x n identity matrix
--*/
    identity:= 
    proc(n)
        local x, i;
    begin
        if args(0) > 2 or args(0) = 0 then 
            error("expecting one argument")
        else
            if not testtype( n,Type::PosInt ) then
                error("expecting a positive integer")
            end_if
        end_if;

        x:= R::one;
        x:= array(1..n,1..n,[[Rzero $ i-1, x, Rzero $ n-i] $ i=1..n]);
        return( new(dom,n,n,x) )
    end_proc;

/*--
    unapply  --  overloads fp::unapply
--*/
    unapply:= proc(x)
        local i, e, Args;
    begin
        e:= map(extop(x,3),expr);
        if args(0) = 1 then
            Args:= [ op(indets(e)) ]
        else
            Args:= [ args(2..args(0)) ]
        end_if;

        if Args = [] then
            subsop(proc() option arrow; begin end, 4=x, Unsimplified)
        else
            subsop(proc() option arrow; begin end, 
                1=op(Args),
                4=hold(extsubsop)(x,3=hold(map)(subs(e, [ Args[i]=DOM_VAR(0,1+i) $ i=1..nops(Args) ],Unsimplified),R::coerce@eval))
            )
        end_if
    end_proc;

    // to be sure that everything works well ....
    // (currently, no matrix properties are available)
    is:= () -> FALSE;

//---------------------------------
// Free module methods
//---------------------------------

    multcoeffs := _mult; // could be optimized

    mapcoeffs := map;

// -----------------------------------------
//          body of the domain
// -----------------------------------------
begin
    if args(0) > 1 then error("wrong no of args") end_if;

    if args(0) = 0 then 
        R := Dom::ExpressionField()
    elif R::dom <> DOM_DOMAIN or R::hasProp( Cat::Ring ) <> TRUE then
        error("expecting a coefficient domain of category 'Cat::Ring'")
    end_if;

    RhasSystemRep := R::hasProp( Ax::systemRep );
    Riszero:= R::iszero; 
    Rzero:= R::zero;
    Rplus:= R::_plus; 
    Rmult:= R::_mult;
    Rcoerce:= R::coerce;

    Rid:= () -> args(1); 
    if (Rnormal := R::normal) = FAIL then 
        if RhasSystemRep and 
           R <> Dom::Integer and 
           R <> Dom::Rational then 
            Rnormal:= normal;
        else 
            Rnormal:= Rid; // normal may receive more than 1 argument
        end_if
    end_if;
end_domain:
