//    


/*++

Dom::GaloisField -- domain constructor for finite fields

Dom::GaloisField(P,n <,pol>)
or
Dom::GaloisField(q)

P               - Prime number
                  or Dom::IntegerMod(p) where p is a prime number
		  or Dom::GaloisField 
n               - integer 
pol (optional) - an irreducible polynomial of degree n over Z/pZ or over
		  the underlying Dom::GaloisField, respectively
q		- integer (= p^n)

Dom::GaloisField creates the finite field Fp^n = Fp[X]/(poly).
If poly is not given as a parameter, it is chosen at random.
An element is represented as Domain element with operands

0 -  domain Dom::GaloisField(P,n,pol)
1 -  polynomial of degree <= n-1 with coefficient ring
        -   IntMod(P)    if P integer 
        -   IntMod(p)    if P=Dom::IntegerMod(p)
        -   P            if P Dom::GaloisField 

Entries:

PrimeField - Dom::IntegerMod(p), where p is the characteristic
Matrices  - the matrices over the field
size      - the number of elements in the field

   
Methods:

frobenius - the frobenius automorphism x->x^q, where p is the size of the underlying
            Dom::IntegerMod or Dom::GaloisField, respectively
conjugates(x) - list of all conjugates of x
             (= images of x under all F-automorphisms,
	     where F is the underlying Dom::GaloisField / Dom::IntegerMod , 
	     respectively)
	     Any conjugate is only listed once even if it occurs as the
	     image of x under several automorphisms          
conjTrace(x) -  computes the trace (= sum of conjugates ) of an element x
conjNorm(x)  -  computes the norm  (= product of conjugates ) of an element x
order(x)     - computes the order (= least integer k with x^k=1 ) of an element
               x  
isSquare(x) - tests whether there exists y such that y^2=x 
randomPrimitive() - returns a generator of the multiplicative Group of the field
		    chosen at random                
isBasis(l) - checks whether the operands of the list l form a basis of the field
isNormal(x) - tests whether the set of conjugates of x is a basis 
randomNormal() - returns a normal element chosen at random      
++*/

domain Dom::GaloisField(P:Type::Union(Type::PosInt, DOM_DOMAIN), n, pol)
  local characteristic,size,PrimeField,VariablesInUse,LogTable;
	

  inherits if domtype(P)=DOM_INT then
             Dom::AlgebraicExtension(Dom::IntegerMod(P),pol)
           else
             Dom::AlgebraicExtension(P,pol)
           end_if;
  
         
  
  category
    Cat::Field,Cat::Algebra(PrimeField),Cat::VectorSpace(PrimeField),
    if domtype(P)=DOM_DOMAIN then Cat::VectorSpace(P) end_if,
    if domtype(P)=DOM_DOMAIN then Cat::Algebra(P) end_if;
    
  axiom Ax::canonicalRep;

    // entries 

    size := size;

    // degreeOverPrimeField inherited from Dom::AlgebraicExtension 

    zero := new(dom,poly(0,op(pol,2..3)));

    one := new(dom,poly(1,op(pol,2..3)));

    iszero := x -> iszero(extop(x,1));

    characteristic :=characteristic;

    Matrices := Dom::/*Dense*/Matrix(dom);

    PrimeField :=PrimeField;

    Variable :=op(pol,[2,1]);
   
    VariablesInUse := VariablesInUse;

	 
    convert :=
    proc(x)
      local xx;
    begin
      if domtype(x)=dom then
        return(x);
      end_if;
      
      xx:=(dom::getSuperDomain())::convert(x);
      if xx = FAIL then return(FAIL) end_if;
      extsubsop(xx,0=dom)
    end_proc;


    convert_to :=
    proc(x, T)
	
    begin
      case T
        of DOM_POLY do return(extop(x,1));
        of dom do return(x);
      end_case;
      FAIL
    end_proc;

    /* elementNumber : orders the field elements as
      0,1, ...,P-1,X,X+1,...,X+P-1,X^2,....,
      (P-1)X^(n-1)+(P-1)X^(n-2)+...+(P-1) */

    elementNumber:=
    if type(P)=DOM_DOMAIN and P::hasProp(Dom::GaloisField) then
      proc(x:dom)
      begin
        evalp(mapcoeffs(subsop(extop(x,1), 3=Expr),P::elementNumber@P),
            dom::Variable=P::size)
      end_proc
    else  
      proc(x:dom)
      begin
        evalp(mapcoeffs(subsop(extop(x,1),3=Expr),_mod,P),
              dom::Variable=P)
      end_proc
    end_if;


      /*   expr and print are inherited from SuperDomain 
      	Dom::AlgebraicExtension(....)  */

      //  _plus, _subtract, and _mult inherited from SuperDomain 

    _power :=
    proc(x:dom, n:DOM_INT)

    begin
      if iszero(n) then dom::one
      elif iszero(x) then dom::zero
      else
        new(dom, powermod(extop(x,1), n mod (size-1), pol))
      end_if
    end_proc;
  
      // faster than the inherited _power would be 

      //  _invert,_divide,intmult,random,TeX inherited from SuperDomain 

        
    frobenius :=
    proc(x:dom)

    begin
      if domtype(P) = DOM_DOMAIN then
        dom::_power(x, P::size)
      else
        dom::_power(x, characteristic)
      end_if
    end_proc;

    conjugates :=
    proc(x:dom)
      local result,a;

    begin
      result:=[x];
      a:=dom::frobenius(x);
      while a<>x do
        result:=result.[a];
        a:=dom::frobenius(a);
      end_while;
      return(result);
    end_proc;

    /*  conjTrace inherited from AlgebraicExtension; faster than        
     	    
       conjTrace := proc(x)
		 local a;

         	begin
		a:=dom::conjugates(x);
	        dom::intmult(dom::_plus(op(a)),n div nops(a));
	 	end_proc;

       would be */
 
     	   
     // conjNorm inherited from AlgebraicExtension 


    order :=
    proc(a:dom)
      local k,l,primes,lngth,i,j,suf,b,c,powarr,bas2l,no;
      
    begin
      if a=dom::zero then return(infinity); end_if;
      if a=dom::one then return(1); end_if;
      k:=size-1;
      if isprime(k) then return(k); end_if;
      l:=stdlib::ifactor(k);
      no:=floor(ln(k div op(l,2))/ln(2));
      // number of powers to be computed 
      lngth:=nops(l) div 2;
      primes:=[op(l,2*i) $ i=1..lngth ];
      suf:=[op(l,2*i+1) $ i=1..lngth ];
      powarr:=array(0..no);
      powarr[0]:=a;
      for i from 1 to no do
        powarr[i]:=powarr[i-1]*powarr[i-1];
      end_for;
      // now, powarr[i] = a^(2^i) for all i 
      for i from 1 to lngth do
        repeat
          c:=k div primes[i];
          bas2l:=[];
          b:=c;
          for j from 0 to no do
            if b mod 2=1 then bas2l:=append(bas2l,j); end_if;
            b:=b div 2;
            if b=0 then break; end_if;
          end_for;
          /* now, bas2l contains all numbers j such that the
            binary representation of c has a "1" at position j */
          
          b:=dom::_mult(powarr[op(bas2l,j)] $ j=1..nops(bas2l));
          // dom gives b=a^c 
          if b=dom::one then
            suf[i]:=suf[i]-1;
            k:=c;
          else
            break;
          end_if;
        until suf[i]=0 end_repeat;
      end_for;
      return(k);
    end_proc;

	
    isSquare:=
	 (if dom::characteristic=2 then 
      TRUE
	  else
      proc(x)
      begin
	   bool(dom::_power(x,(dom::size-1) div 2) =dom::one)
      end_proc
	  end_if);

    ln :=
    proc(x:dom,g:dom)
      // compute log(x) w.r.t. the base g 
      local ord,Q,gQ,gQk,lastg,i,ginv,xg,position;
      
    begin
      if iszero(g) then
        error("Zero cannot be base of logarithm")
      end_if;
      if iszero(x) then
        return(infinity)
      end_if;
      userinfo(3, "Computing giant steps");
      ord:=dom::order(g);
      Q:=ceil(sqrt(ord));
    	userinfo(5, "Size of giant step: ".expr2text(Q));
      gQ:=g^Q;
      gQk:=[dom::one];
    	lastg:=dom::one;
    	for i from 1 to Q do
        lastg:=lastg*gQ;
        gQk:=append(gQk, lastg);
        if lastg=x then
          userinfo(3, "Found logarithm in giant step");
          return(Q*i)
        end_if
      end_for;
      ginv:=g^(-1);
      xg:=x*ginv;
      for i from 1 to Q do
        if (position:=contains(gQk,xg))<>0 then
          return(((position-1)*Q+i) mod ord)
        end_if;
        xg:=xg*ginv;
      end_for;
      userinfo(2, expr2text(x). " is not in the subgroup generated by ".
               expr2text(g));
      infinity
    	end_proc;

    // companion matrix of pol 

    companionMatrix :=
    proc()
      option remember;
      local i: DOM_INT;
    begin
        (Dom::/*Dense*/Matrix(dom::groundField))
        (n,n, [[ 0 $n-1 , -coeff(pol,0)],
               [0 $i-1, 1, 0 $ n-i-1, -coeff(pol,i)] $i=1..n-1]);
    end_proc;


    // companionPowers: list of powers of the companion matrix 
    
    companionPowers :=
    proc()
      option remember;
      local i;
    begin
      [];
      (dom::companionMatrix())^0;
      for i from 1 to n-1 do
        append(%2,%);
        %2*dom::companionMatrix()
      end_for;
      append(%2,%)
    end_proc;

    matrixRepresentation :=
    proc(x:dom)
      local xcoeffs,i;  
    begin
      xcoeffs:=dom::convert_to(x, DOM_POLY);
      xcoeffs:=[coeff(xcoeffs,i) $i=0..n-1];
      _plus(op(zip(xcoeffs, dom::companionPowers(), _mult)))
    end_proc;

    randomPrimitive :=
    proc()
      local a,c,k,l,result,i,j;

    begin
      k:=dom::size-1;
      if isprime(k) then
        repeat
          result:=dom::random();
        until result<>(dom::one) and result<>(dom::zero) end_repeat;
        return(result);
      end_if;
      l:=stdlib::ifactor(k);
      result:=dom::one;
      for i in [[l[2*j],l[2*j+1]] $ j=1..nops(l) div 2] do
        c:=k div i[1];
        repeat
          a:=dom::random();
        until not iszero(a) and a^c<>dom::one end_repeat;
        result:=result*a^(k div (i[1]^i[2]));
      end_for;
      return(result);
    end_proc;
	
		

	  		

    isBasis :=
    proc(x:Type::ListOf(dom))
      local M;
      
    begin
      if nops(x) <> n then return(FALSE); end_if;
      M:=dom::Matrices(map(x,dom::conjugates));
      if linalg::rank(M) = n then
        return(TRUE)
      else
        return(FALSE);
      end_if;
    end_proc;
	

/*

   too slow:

	"isNormal"=proc(x:dom)
	local a,j,i,l;

	begin
	a:=dom::conjugates(x);
	if nops(a)<>n then return(FALSE); end_if;
	// generate circulant
        l:=[([op(a,j) $ j=i..n].[op(a,j) $ j=1..i-1])
			 $ i=1..n];
	return(bool(linalg::rank(dom::Matrices(l))=n));
	end_proc,
	
*/   

   isNormal :=
   proc(x:dom)
     local a,j;

   begin
     a:=dom::conjugates(x);
     if nops(a)<>n then return(FALSE); end_if;
     gcd(poly(`#y`^n-1,[`#y`],dom),poly([[op(a,j),n-j] $j=1..n],[`#y`],dom));
     return(bool(degree(%)=0))
   end_proc;
   

   randomNormal :=
   proc()
     local a;
   begin
     repeat
       a:=dom::random();
     until dom::isNormal(a) end_repeat;
     a;
   end_proc;

   isPrimitivePolynomial :=
   proc(f:Type::PolyOf(dom))
   local dg, x,G,o;

   begin
   if nops(op(f,2))>1 then
      error("Polynomial must be univariate")
   else
      x:=op(f,[2,1])
   end_if;
   if (dg:=degree(f))=0 then
      return(FALSE)
   end_if;
   if coeff(f,x,0)=dom::zero then
      return(FALSE)
   end_if;
   if lcoeff(f)<>dom::one then
      return(FALSE)
   end_if;
   if not irreducible(f) then
      userinfo(3, "Polynomial is reducible");
      return(FALSE)
   end_if;
   G:=Dom::GaloisField(dom, dg, f);
   o:=G::order(G(x));
   userinfo(3, "Polynomial has order ".expr2text(o));
   return(bool(o=G::size-1))
   end_proc;

// initialization 
        
begin
if args(0) < 1 or args(0) > 3 then error("wrong no of args"); end_if; 
   if args(0) = 1 then		
		if not testtype(P, Type::PosInt) then
			error("Number of elements must be positive integer");
		end_if;
		size:=P;
		if isprime(P) then 
			n:=1
		else
			n:=numlib::ispower(P);
         if n=FALSE then
		        	error("Number of elements must be prime power");
         end_if;
			P:=op(n,1);
			n:=op(n,2);
		end_if;
		characteristic:=P;
		PrimeField:=Dom::IntegerMod(P);
		VariablesInUse:=[];
    elif domtype(P)= DOM_INT then 
		if not isprime(P) then
        error("P must be prime"); 
		else 
			characteristic:=P;
			size:=P^n;
			PrimeField:=Dom::IntegerMod(P);		
                        VariablesInUse:=[];
		end_if;
	 elif 
	  (P::constructor <> Dom::IntegerMod and 
      P::constructor <> Dom::GaloisField )
      or not P::hasProp(Cat::Field) then 
         error("First Argument must be prime or Dom::IntegerMod(p),".
               " p prime or Dom::GaloisField");
    else 
		characteristic := P::characteristic;
      if P::constructor = Dom::IntegerMod then
			PrimeField:=P;
			P:=characteristic;
			size:=P^n;
         VariablesInUse:=[];
		else   // Dom::GaloisField over another Dom::GaloisField 
			PrimeField:=P::PrimeField;
			size:=(P::size)^n;
         VariablesInUse:=P::VariablesInUse;
		end_if;
	 end_if;       
    if domtype(n) <> DOM_INT then
		error("Second argument must be integer");
	 end_if;
	 if n < 1 then 
		error("Degree over underlying Field must be >= 1 ")
	 end_if;
	 if args(0)=3 then
		userinfo(5,"Testing argument",pol);
		if domtype(pol) = DOM_EXPR or domtype(pol)= DOM_IDENT
	   	or domtype(pol) = DOM_POLY  then
			if domtype(P)=DOM_INT then                          
				pol:= poly(pol,IntMod(P));
		        elif P::constructor = Dom::IntegerMod then
                                pol:= poly(pol,IntMod(P::characteristic));
			else pol:= poly(pol,P);
			end_if;
	        else 
			error("Third argument must be polynomial");
	 	end_if;
	        if nops(op(pol,2))<> 1 then
			error("Polynomial in one variable needed");
		end_if;
      if has(VariablesInUse,op(pol,[2,1])) then 
         error("Variable already used in underlying Dom::GaloisField");
      end_if;
		if degree(pol) <> n then
			error("Polynomial has wrong degree");
		end_if;
		if not faclib::is_irred_gf(pol) then 
			error("Polynomial not irreducible");
		end_if;
		
		VariablesInUse:=VariablesInUse.op(pol,2); 

          else  // i.e. if args(0) = 2 
		if domtype(P)=DOM_INT then
			pol:=faclib::random_irred_gf(IntMod(P),n);
                elif P::constructor = Dom::IntegerMod then
    			pol:=faclib::random_irred_gf(IntMod(P::characteristic),n);
		else
                	pol:=faclib::random_irred_gf(P,n);
		end_if;
		userinfo(3,"Choosing random irreducible polynomial",pol);
		VariablesInUse:=VariablesInUse.op(pol,2);      
end_if
end_domain:

// end of file 
