/*   
*/
/*++
Dom::IntegerMod -- domain constructor for the residue class rings Z/nZ

Dom::IntegerMod(n)

n - integer > 1

Dom::IntegerMod(n) creates the residue class ring (or field) Z/nZ.
An element is represented as a domain element with operands:

0 - domain Dom::IntegerMod(n)
1 - integer of {0,...,n-1} representing the residue class

++*/

domain Dom::IntegerMod(Mod: Type::PosInt)

local rndgen, x /* to avoid warnings */ ;   
inherits Dom::BaseDomain;
category if isprime(Mod) then 
            Cat::Field 
         else 
            Cat::CommutativeRing 
         end_if;
axiom Ax::canonicalRep,  
      Ax::efficientOperation("_divide"), Ax::efficientOperation("_mult"),
      Ax::efficientOperation("_invert");


    characteristic := Mod;
 
    size := Mod;
    
    zero := new(dom, 0);

    one := new(dom, 1);

    iszero := x -> iszero(extop(x,1));

    normal := () -> args(1);

    convert :=
    proc(x)
    begin
      case domtype(x)
        of dom do return(x);
        of DOM_INT do return(new(dom, x mod Mod));
        of DOM_RAT do
          if not iszero(op(x,2) mod Mod) then
            return(dom::_divide(dom::new(op(x,1)),
                                dom::new(op(x,2))));
          end_if;
      end_case;
      FAIL
    end_proc;

    convert_to :=
    proc(x, T)
    begin
      case T
        of DOM_INT do
        of Dom::Integer do return(extop(x,1));
        of dom do return(x);
      end_case;
      FAIL
    end_proc;

  expr := x -> extop(x,1);

  expr2text:= x -> expr2text(extop(x, 0))."(".expr2text(extop(x, 1)).")";
  
  print := x -> hold(_mod)(extop(x,1), Mod);

  _plus :=
  proc()
    local argv;
  begin
    argv:= map([args()], dom::convert);
    if map(argv, domtype) <> [dom $nops(argv)] then
      return(FAIL)
    end_if;
    new(dom, _plus(op(map(argv, extop, 1))) mod Mod)
  end_proc;

  _subtract :=
  proc(x, y)
  begin
    if domtype(x) <> dom then
      x:= dom::convert(x);
    end_if;
    if domtype(y) <> dom then
       y:= dom::convert(y);
    end_if;
    if type(x) <> dom or type(y) <> dom then
      FAIL
    else
      new(dom, (extop(x,1) - extop(y,1)) mod Mod)
    end_if
  end_proc;

  _mult :=
  proc(x,y)
    local t1, t2;
  begin
    if args(0) = 2 then
      if domtype(y) <> dom then
        case domtype(y)
          of DOM_INT do
            return(dom::intmult(args()))
          of DOM_POLY do
            if op(y,3) = dom then
              return(mapcoeffs(y, dom::_mult, x))
            else
              // of course, one could try to convert x to the domain op(y,3),
              // if this is what the user might expect ...
              return(FAIL)
            end_if
          of DOM_RAT do
            return(dom::_mult(x, dom::convert(y)));
            break;
          otherwise
            return((domtype(y))::_mult(args()))
        end_case
      elif domtype(x) <> dom then
        case domtype(x)
          of DOM_INT do
            return(dom::intmult(y, x))
          of DOM_POLY do
            if op(x,3) = dom then
              return(mapcoeffs(x, dom::_mult, y))
            else
              // of course, one could try to convert y to the domain op(x,3),
              // if this is what the user might expect ...
              return(FAIL)
            end_if
          of DOM_RAT do
            return(dom::_mult(dom::convert(x), y));
            break;
          otherwise
            return(FAIL)
        end_case
      else
        new(dom, _mult(map(args(), extop, 1)) mod Mod)
      end_if
    elif args(0) = 1 then
      x
    elif args(0) = 0 then
      dom::one
    else
      t1 := _mult(args(1..(args(0) div 2)));
      t2 := _mult(args(((args(0) div 2) + 1)..args(0)));
      if (domtype(t1))::_mult <> FAIL then
        (domtype(t1))::_mult(t1, t2)
      elif (domtype(t2))::_mult <> FAIL then
        (domtype(t2))::_mult(t1, t2)
      else
        _mult(t1, t2)
      end_if
    end_if
  end_proc;

  _power :=
  (if isprime(Mod) then
     proc(x,y)
     begin
       if domtype(y) <> DOM_INT then
         error("no integer exponent")
       end_if;
       assert(x::dom = dom);
       if y < 0 then
         new(dom, powermod((1 / extop(x,1)) mod Mod, -y, Mod))
       elif iszero(y) then
         new(dom,1)
       elif iszero(x) then
         new(dom,0)
       else
         new(dom, powermod(extop(x,1), y mod (Mod-1),
                           Mod))
       end_if
     end_proc
   else
     proc(x,y)
     begin
       if testargs() then
         if not testtype(y, DOM_INT) then
           error("integer exponent expected");
         end_if;
       end_if;

       assert(x::dom = dom);
       if y < 0 then
         if dom::_invert(x) = FAIL then
           userinfo(1, expr2text(x)." not invertible");
           FAIL;
         else
           new(dom, powermod(extop(dom::_invert(x), 1) mod Mod, -y, Mod))
         end_if;
       elif iszero(y) then
         new(dom,1)
	    elif iszero(x) then
		   new(dom,0)
	    else
		   new(dom, powermod(extop(x,1), y, Mod))
	    end_if
   end_proc
    end_if);

    _negate := x -> new(dom, (-extop(x,1)) mod Mod);

    intmult :=
    proc(x:dom, n:DOM_INT)
    begin
      new(dom, (extop(x,1) * n) mod Mod)
    end_proc;

    _invert := (if isprime(Mod) then
	x -> new(dom, (1 / extop(x,1)) mod Mod)
    else
        proc(x)
	begin
	    if igcd(extop(x,1), Mod) = 1 then
		new(dom, (1 / extop(x,1)) mod Mod)
	    else
		FAIL
	    end_if
	end_proc
    end_if);

    _divide :=
    (if isprime(Mod) then
      proc(xx, yy)
        local x, y;
      begin
        x := xx; y := yy;
        if x::dom = DOM_INT or x::dom = DOM_RAT then
          x := dom::new(x);
        end_if;
        if y::dom = DOM_INT or y::dom = DOM_RAT then
          y := dom::new(y);
        end_if;
        if x::dom = dom and y::dom = dom then
          new(dom, (extop(x,1) / extop(y,1)) mod Mod);
        else
          FAIL
        end_if
      end_proc
      else
      proc(xx, yy)
        local x, y, yinv;
      begin
        x := xx; y := yy;
        if x::dom = DOM_INT or x::dom = DOM_RAT then
          x := dom::new(x);
        end_if;
        if y::dom = DOM_INT or y::dom = DOM_RAT then
          y := dom::new(y);
        end_if;
        if x::dom = dom and y::dom = dom then
          if igcd(extop(y,1), Mod) = 1 then
            new(dom, (extop(x,1) / extop(y,1)) mod Mod)
          else
            FAIL
          end_if;
        else
        yinv := y::dom::_invert(y);
        if yinv <> FAIL then
          x*yinv
        else
          FAIL
        end_if;
       end_if;
     end_proc
    end_if);

    isSquare := x -> bool(numlib::jacobi(extop(x,1),Mod)=1);

    ln :=
    proc(x:dom, g:dom)
      // compute log(x) w.r.t. the base g
      local ord,Q,gQ,gQk,lastg,i,ginv,xg,position;
      
    begin
      if iszero(g) then
        error("Zero cannot be base of logarithm")
      end_if;
      if iszero(x) then
        return(infinity)
      end_if;
      if (i:=igcd(expr(g), Mod))<>1 then
        return(FAIL)
      end_if;  
      userinfo(3, "Computing giant steps");
      ord:=dom::order(g);
      Q:=ceil(sqrt(ord));
      userinfo(5, "Size of giant step: ".expr2text(Q));
      gQ:=g^Q;
      gQk:=[dom::one];
      lastg:=dom::one;
      for i from 1 to Q do
        lastg:=lastg*gQ;
        gQk:=append(gQk, lastg);
        if lastg=x then
                userinfo(3, "Found logarithm in giant step");
                return(Q*i)
        end_if
      end_for;
      ginv:=g^(-1);
      xg:=x*ginv;
      for i from 1 to Q do
        if (position:=contains(gQk,xg))<>0 then
                return((position-1)*Q+i)
        end_if;
        xg:=xg*ginv;
      end_for;    
      userinfo(2, expr2text(x). " is not in the subgroup generated by ".
        expr2text(g));
      infinity
    end_proc;

  randomPrimitive:=
  if Mod = 3 then
    () -> new(dom, 2)
  else                    
    proc()
      local a,c,k,l,result,i,j;
    begin
      k:=Mod-1;
      l:=stdlib::ifactor(k);
      result:=dom::one;
      for i in [[l[2*j],l[2*j+1]] $ j=1..nops(l) div 2] do
        c:=k div i[1];
        repeat
          a:=dom::random();
        until not iszero(a) and a^c<>dom::one end_repeat;
        result:=result*a^(k div (i[1]^i[2]));
      end_for;
      return(result)
    end_proc
  end_if;


  

  // solve_poly - return the zeroes of a polynomial
  solve_poly:=
  proc(f: DOM_POLY, x: DOM_IDENT, options: DOM_TABLE)
    local S: DOM_SET, thisdom;
  begin
    thisdom:= dom; // because dom is not visible in the inner proc
    if options[Multiple] then
      // this makes no sense if dom is not a field
      if dom::hasProp(Cat::Field) then
        S:= numlib::mroots(poly(f, [x], hold(Expr)), Mod);
        Dom::Multiset::convert
        ([op(map(S,
                proc(y: DOM_INT): DOM_LIST
                  // given a root y, return a pair [y, m]
                  // where m is the multiplicity of y
                  local i, g, d;
                begin
                  i:=0;
                  g:= poly(f, [x], IntMod(Mod));
                  d:= poly(x-y, [x], IntMod(Mod));
                  while (g:= divide(g, d, Exact)) <> FAIL do
                    i:=i+1
                  end_while;
                  [thisdom::new(y), i]
                end_proc
                ))])
      else
        error("Option multiple is only allowed for fields")
      end_if
    else
      {op(map(numlib::mroots(poly(f, [x], hold(Expr)), Mod), dom::new))}
    end_if
  end_proc;
  
    order:= x -> numlib::order(extop(x,1), Mod);

    TeX := x -> expr2text(extop(x,1))."\\,\\text{mod}\\,".expr2text(Mod);
    
    TeXrep := x -> "\\mathbb{Z}_{".expr2text(Mod)."}";
    
    random:= proc() begin new(dom, rndgen()) end_proc;


begin
  if Mod < 2 then
    error("modulus must be > 1")
  end_if;
    rndgen:=random(Mod);
end_domain:

// end of file 
