//    
/*++
Dom::Product -- the domain constructor of homogeneous finite products

Dom::Product(S [,n])

S - set domain
n - positive integer (optional, default is 1)

A homogeneous finite product is a n-fold product of identical sets S.

Entries:-
card                - the dimension n

Methods:-
map(p,f,...)        - maps f(p.i,...) to each component of p
zip(p,q,f)          - maps f(p.i,q.i) to each corresponding component pair of p and q
mapCanFail(p,f,...) - like map but returns FAIL if one call of f returns FAIL
zipCanFail(p,q,f)   - like zip but returns FAIL if one call of f returns FAIL
testEach(p,f,...)   - returns TRUE if f(p.i,...) returns TRUE for each component of p
testOne(p,f,...)    - returns TRUE if f(p.i,...) returns TRUE for one component of p
_index(p,i)         - returns the i-th component of p
set_index(p,i,v)    - changes the i-th component of p to v
nops(p)             - returns n
op(p)               - returns the sequence (p[1],...p[n])
op(p,i)             - returns p[i]
subsop(p,i=v)       - changes the i-th element op p to v
subs(p,x=y)         - substitutes x by y in each element of p
++*/

domain Dom::Product(Set, Dimen)
    inherits Dom::BaseDomain;
    category Cat::HomogeneousFiniteProduct(Set);
    axiom 
        if Set::hasProp(Ax::canonicalRep) then Ax::canonicalRep end_if,
        if Set::hasProp(Cat::AbelianMonoid) then
           if Set::hasProp(Ax::normalRep) then Ax::normalRep end_if
        end_if;

/* entries */

    dimen:= Dimen;

    size:= if Set::size = FAIL then
             FAIL
           else
             (Set::size)^Dimen
           end_if;
      
      
    convert:= (if Dimen = 1 then
        proc(e) begin
            if args(0) <> 1 then return(FAIL) end_if;
            case domtype(e)
            of dom do return(e);
            of DOM_LIST do 
		if nops(e)<>1 then
			return(FAIL)
		else
			e:= e[1]; //fall through
		end_if
            otherwise e:= Set::convert(e);
            end_case;
            if e = FAIL then FAIL else new(dom, e) end_if
        end_proc
    else
	proc(l) begin
	    if args(0) = 1 then
	    	case domtype(l)
	    	of dom do return(l);
	    	of DOM_LIST do 
			if nops(l)<>Dimen then
				FAIL
			else
				break
			end_if
	    	otherwise return(FAIL);
	    	end_case;
	    elif args(0) = Dimen then
	    	l:= [ args() ];
	    else
	    	return(FAIL)
	    end_if;
	    l:= map(l, Set::convert);
	    if contains(l, FAIL) = 0 then new(dom, l) else FAIL end_if
	end_proc
    end_if);
    
    expr:= (if Dimen = 1 then
    	(()->(Set::expr(extop(args(1),1))))
    else
    	(()->(map(extop(args(1),1), Set::expr)))
    end_if);
    
    sort:= x -> dom::convert(sort(expr(x),args(2..args(0))));

    equal:= (if dom::hasProp(Ax::canonicalRep) then
	bool @ _equal
    elif Dimen = 1 then
    	(()->(Set::equal(extop(args(1),1), extop(args(2),1))))
    else
    	proc(x,y) local i; begin
    	    x:= extop(x,1); y:= extop(y,1);
    	    for i from 1 to Dimen do
    	    	if not Set::equal(x[i], y[i]) then return(FALSE) end_if
    	    end_for;
    	    TRUE
    	end_proc
    end_if);

    map:= (if Dimen = 1 then
    	(()->(new(dom, args(2)(extop(args(1),1), args(3..args(0))))))
    else
    	(()->(new(dom, map(extop(args(1),1), args(2..args(0))))))
    end_if);
    
    zip:= (if Dimen = 1 then
    	(()->(new(dom, args(3)(extop(args(1),1), extop(args(2),1)))))
    else
    	(()->(new(dom, zip(extop(args(1),1), extop(args(2),1), args(3)))))
    end_if);

    mapCanFail:= (if Dimen = 1 then
    	proc(x,f) local i; begin
    	    x:= f(extop(x,1), args(i) $ i=3..args(0));
    	    if x = FAIL then FAIL else new(dom, x) end_if
    	end_proc
    else
    	proc(x,f) local e, l, a; begin
    	    a:= args(e) $ e=3..args(0);
    	    l:= [];
    	    for e in extop(x,1) do
    	    	e:= f(e, a);
    	    	if e = FAIL then return(FAIL) end_if;
    	    	l:= append(l, e)
    	    end_for;
    	    new(dom, l)
    	end_proc
    end_if);
    
    zipCanFail:= (if Dimen = 1 then
    	proc(x,y,f) begin
    	    x:= f(extop(x,1), extop(y,1));
    	    if x = FAIL then FAIL else new(dom, x) end_if
    	end_proc
    else
    	proc(x,y,f) local e, l, i; begin
    	    x:= extop(x,1);
    	    y:= extop(y,1);
    	    l:= [];
    	    for i from 1 to Dimen do
    	    	e:= f(x[i], y[i]);
    	    	if e = FAIL then return(FAIL) end_if;
    	    	l:= append(l, e)
    	    end_for;
    	    new(dom, l)
    	end_proc
    end_if);

    testEach:= (if Dimen = 1 then
    	(()->(args(2)(extop(args(1),1), args(3..args(0)))))
    else
    	proc(x,f) local e, a; begin
    	    a:= args(e) $ e=3..args(0);
    	    for e in extop(x,1) do
    	    	if not f(e,a) then return(FALSE) end_if
    	    end_for;
    	    TRUE
    	end_proc
    end_if);

    testOne:= (if Dimen = 1 then
    	dom::testEach
    else
    	proc(x,f) local e, a; begin
    	    a:= args(e) $ e=3..args(0);
    	    for e in extop(x,1) do
    	    	if f(e,a) then return(TRUE) end_if
    	    end_for;
    	    FALSE
    	end_proc
    end_if);

    op:= (if Dimen = 1 then
    	(()->(extop(args(1),1)))
    else
    	(()->((
    	    if args(0) = 1 then op(extop(args(1),1))
    	    else extop(args(1),1)[args(2)] end_if
    	)))
    end_if);

    _index:= (if Dimen = 1 then
    	(()->(extop(args(1),1)))
    else
    	(()->(extop(args(1),1)[args(2)]))
    end_if);
    
    set_index:= (if Dimen = 1 then
    	(()->(new(dom, args(args(0)))))
    else
    	(()->(new(dom, subsop(extop(args(1),1), args(2)=args(args(0))))))
    end_if);
    
    _less:= (if Set::hasProp(Cat::OrderedSet) then
    	if Dimen = 1 then
    	    (()->(Set::_less(extop(args(1),1), extop(args(2),1))))
    	else
    	    // order lexicographically 
    	    proc(x,y) local i; begin
    	    	x:= extop(x,1);
    	    	y:= extop(y,1);
    	    	for i from 1 to Dimen do
    	    	    if not Set::equal(x[i], y[i]) then
    	    	    	return(bool(Set::_less(x[i], y[i])))
    	    	    end_if
    	    	end_for;
    	    	FALSE
    	    end_proc
    	end_if
    end_if);

    subs:= (if Dimen = 1 then
        (()->(new(dom, subs(extop(args(1),1), args(2)))))
    else
        (()->(new(dom, map(extop(args(1),1), subs, args(2)))))
    end_if);
    
    subsop:= (if Dimen = 1 then
    	(()->(new(dom, args(3))))
    else
        (()->(new(dom, subsop(extop(args(1),1), args(2)=args(3)))))
    end_if);
    
    _plus:= (if Set::hasProp(Cat::AbelianSemiGroup) then
	if Dimen = 1 then
	    (()->((if map({args()}, domtype) <> {dom} then return(FAIL) end_if;
	         new(dom, Set::_plus(op(map([args()], extop, 1)))))))
	else
	    (()->((
	    	case args(0)
	    	of 2 do
	    	    if map({args()}, domtype) <> {dom} then return(FAIL) end_if;
	    	    new(dom, zip(extop(args(1),1), extop(args(2),1), Set::_plus));
		    break;
	    	of 1 do args(1); break;
		otherwise
	    	    _plus(args(1..(args(0) div 2)));
	    	    _plus(args(((args(0) div 2)+1)..args(0)));
	    	    if (domtype(%2))::_plus <> FAIL then (domtype(%2))::_plus(%2, %1)
	    	    elif (domtype(%1))::_plus <> FAIL then (domtype(%1))::_plus(%2, %1)
	    	    else _plus(%2, %1)
	    	    end_if
	    	end_case
	    )))
	end_if
    end_if);
    
    intmult:= (if Set::hasProp(Cat::AbelianSemiGroup) then
	if Dimen = 1 then
	    (()->(new(dom, Set::intmult(extop(args(1),1), args(2)))))
	else
	    (()->(new(dom, map(extop(args(1),1), Set::intmult, args(2)))))
	end_if
    end_if);
    
    zero:= (if Set::hasProp(Cat::AbelianMonoid) then
	if Dimen = 1 then
	    new(dom, Set::zero)
	else
	    new(dom, [ Set::zero $ Dimen ])
	end_if
    end_if);
    
    iszero:= (if Set::hasProp(Cat::AbelianMonoid) then
	if Dimen = 1 then
	    (()->(Set::iszero(extop(args(1),1))))
	else
	    proc(x) local i; begin
	    	for i in extop(x,1) do 
	    	    if not Set::iszero(i) then return(FALSE) end_if 
	    	end_for;
	    	TRUE
	    end_proc
	end_if
    end_if);

    _subtract:= (if Set::hasProp(Cat::AbelianGroup) then
	if Dimen = 1 then
	    (()->(new(dom, Set::_subtract(extop(args(1),1), extop(args(2),1)))))
	else
	    (()->(new(dom, zip(extop(args(1),1), extop(args(2),1), Set::_subtract))))
	end_if
    elif Set::hasProp(Cat::CancellationAbelianMonoid) then
	if Dimen = 1 then
	    proc(x,y) begin
	    	x:= Set::_subtract(extop(x,1), extop(y,1));
	    	if x = FAIL then FAIL else new(dom, x) end_if
	    end_proc
	else
	    proc(x,y) begin
	    	x:= zip(extop(x,1), extop(y,1), Set::_subtract);
	    	if contains(x,FAIL) = 0 then new(dom, x) else FAIL end_if
	    end_proc
	end_if
    end_if);
    
    _negate:= (if Set::hasProp(Cat::AbelianGroup) then
	if Dimen = 1 then
	    (()->(new(dom, Set::_negate(extop(args(1),1)))))
	else
	    (()->(new(dom, map(extop(args(1),1), Set::_negate))))
	end_if
    elif Set::hasProp(Cat::CancellationAbelianMonoid) then
	if Dimen = 1 then
	    proc(x) begin
	    	x:= Set::_negate(extop(x,1));
	    	if x = FAIL then FAIL else new(dom, x) end_if
	    end_proc
	else
	    proc(x) begin
	    	x:= map(extop(x,1), Set::_negate);
	    	if contains(x,FAIL) = 0 then new(dom, x) else FAIL end_if
	    end_proc
	end_if
    end_if);
    
    _mult:= (if Set::hasProp(Cat::SemiGroup) then
    	if Dimen = 1 then
    	    if Set::hasProp(Ax::systemRep) then
    	    	(()->((
    	    	    if args(0) = 2 then
    	    	        if domtype(args(2)) <> dom then
    	    	            Set::_mult(extop(args(1),1), args(2));
    	    	            if testtype(%, Set) then new(dom, %) else FAIL end_if;
    	    	        elif domtype(args(1)) <> dom then
    	    	            Set::_mult(args(1), extop(args(2),1));
    	    	            if testtype(%, Set) then new(dom, %) else FAIL end_if;
    	    	        else
    	    	            new(dom, Set::_mult(extop(args(1),1), extop(args(2),1)))
    	    	        end_if
    	    	    elif args(0) = 1 then
    	    	    	args(1)
    	    	    else
			_mult(args(1..(args(0) div 2)));
			_mult(args(((args(0) div 2) + 1)..args(0)));
			if (domtype(%2))::_mult <> FAIL then (domtype(%2))::_mult(%2, %1)
			elif (domtype(%1))::_mult <> FAIL then (domtype(%1))::_mult(%2, %1)
			else _mult(%2, %1)
			end_if
    	    	    end_if
    	    	)))
	    else
    	    	(()->((
    	    	    if args(0) = 2 then
    	    	        if domtype(args(2)) <> dom then
    	    	            Set::_mult(extop(args(1),1), args(2));
    	    	            if domtype(%) = Set then new(dom, %) else FAIL end_if;
    	    	        elif domtype(args(1)) <> dom then
    	    	            Set::_mult(args(1), extop(args(2),1));
    	    	            if domtype(%) = Set then new(dom, %) else FAIL end_if;
    	    	        else
    	    	            new(dom, Set::_mult(extop(args(1),1), extop(args(2),1)))
    	    	        end_if
    	    	    elif args(0) = 1 then
    	    	    	args(1)
    	    	    else
			_mult(args(1..(args(0) div 2)));
			_mult(args(((args(0) div 2) + 1)..args(0)));
			if (domtype(%2))::_mult <> FAIL then (domtype(%2))::_mult(%2, %1)
			elif (domtype(%1))::_mult <> FAIL then (domtype(%1))::_mult(%2, %1)
			else _mult(%2, %1)
			end_if
    	    	    end_if
    	    	)))
	    end_if
	else
    	    if Set::hasProp(Ax::systemRep) then
    	    	(()->((
    	    	    if args(0) = 2 then
    	    	        if domtype(args(2)) <> dom then
    	    	            map(extop(args(1),1), Set::_mult, args(2));
    	    	            if testtype(%, Type::ListOf(Set)) then new(dom, %)
    	    	            else FAIL end_if
    	    	        elif domtype(args(1)) <> dom then
    	    	            zip([args(1) $ Dimen], extop(args(2),1), Set::_mult);
    	    	            if testtype(%, Type::ListOf(Set)) then new(dom, %)
    	    	            else FAIL end_if
    	    	        else
    	    	            new(dom, zip(extop(args(1),1), extop(args(2),1), Set::_mult))
    	    	        end_if
    	    	    elif args(0) = 1 then
    	    	    	args(1)
    	    	    else
			_mult(args(1..(args(0) div 2)));
			_mult(args(((args(0) div 2) + 1)..args(0)));
			if (domtype(%2))::_mult <> FAIL then (domtype(%2))::_mult(%2, %1)
			elif (domtype(%1))::_mult <> FAIL then (domtype(%1))::_mult(%2, %1)
			else _mult(%2, %1)
			end_if
    	    	    end_if
    	    	)))
    	    else
    	    	(()->((
    	    	    if args(0) = 2 then
    	    	        if domtype(args(2)) <> dom then
    	    	            map(extop(args(1),1), Set::_mult, args(2));
    	    	            if {op(map(%, domtype))} = {Set} then new(dom, %)
    	    	            else FAIL end_if
    	    	        elif domtype(args(1)) <> dom then
    	    	            zip([args(1) $ Dimen], extop(args(2),1), Set::_mult);
    	    	            if {op(map(%, domtype))} = {Set} then new(dom, %)
    	    	            else FAIL end_if
    	    	        else
    	    	            new(dom, zip(extop(args(1),1), extop(args(2),1), Set::_mult))
    	    	        end_if
    	    	    elif args(0) = 1 then
    	    	    	args(1)
    	    	    else
			_mult(args(1..(args(0) div 2)));
			_mult(args(((args(0) div 2) + 1)..args(0)));
			if (domtype(%2))::_mult <> FAIL then (domtype(%2))::_mult(%2, %1)
			elif (domtype(%1))::_mult <> FAIL then (domtype(%1))::_mult(%2, %1)
			else _mult(%2, %1)
			end_if
    	    	    end_if
    	    	)))
    	    end_if
    	end_if
    end_if);
    
    _power:= (if Set::hasProp(Cat::SemiGroup) then
	if Dimen = 1 then
	    (x,y)->(
	        if domtype(x) <> dom then error("illegal power") end_if;
                x:= extop(x,1) ^ y;
                if x = FAIL then return( FAIL ) else return( new(dom,x) ) end 
	     )
	else
            (x,y) -> (
	        if domtype(x) <> dom then error("illegal power") end_if;
                x:= map(extop(x,1), _power, y);
                if contains(x,FAIL) <> 0 then return( FAIL ) else return( new(dom,x) ) end
	    )
	end_if
    end_if);
    
    one:= (if Set::hasProp(Cat::Monoid) then
	if Dimen = 1 then
	    new(dom, Set::one)
	else
	    new(dom, [ Set::one $ Dimen ])
	end_if
    end_if);
    
    _invert:= (if Set::hasProp(Cat::Group) then
	if Dimen = 1 then
	    x -> (
                x:= Set::_invert(extop(x,1));
                if x = FAIL then return( FAIL ) else return( new(dom,x) ) end
            )
	else
	    x -> (
                x:= map(extop(x,1), Set::_invert);
                if contains( x,FAIL ) <> 0 then return( FAIL ) else return( new(dom,x) ) end
            )
	end_if
    elif Set::hasProp(Cat::Monoid) then
	if Dimen = 1 then
	    proc(x) begin
	    	x:= Set::_invert(extop(x,1));
	    	if x = FAIL then FAIL else new(dom, x) end_if
	    end_proc
	else
	    proc(x) begin
	    	x:= map(extop(x,1), Set::_invert);
	    	if contains(x,FAIL) <> 0 then return( FAIL ) else return( new(dom,x) ) end
	    end_proc
	end_if
    end_if);
    
    _divide:= (if Set::hasProp(Cat::Group) then
	if Dimen = 1 then
            proc(x,y)
            begin
                x:= Set::_divide(extop(x,1),extop(y,1));
                if x = FAIL then return( FAIL ) else return( new(dom,x) ) end
            end
	else
            proc(x,y)
            begin
                x:= zip(extop(x,1),extop(y,1),Set::_divide);
                if contains(x,FAIL) <> 0 then return( FAIL ) else return( new(dom,x) ) end
            end
	end_if
    end_if);
    
    D:= (if Set::hasProp(Cat::PartialDifferentialRing) then
	if Dimen = 1 then
	    x -> new(dom,
	    	  (if args(0) = 1 then Set::D(extop(x,1))
	    	   else Set::D(x, extop(args(2),1)) end_if)
                 )
	else
	    x -> new(dom,
	    	    (if args(0) = 1 then map(extop(x,1), Set::D)
	    	     else map(extop(args(2),1), 
	    	     	      eval(subsop(hold((()->(Set::D(A, args(1))))), [1,1]=x)))
	    	     end_if)
                )
	end_if
    end_if);
    
    diff:= (if Set::hasProp(Cat::PartialDifferentialRing) then
	if Dimen = 1 then
	    x -> new(dom, Set::diff(extop(x,1), args(2..args(0))) )
	else
	    x -> new(dom, map(extop(x,1), Set::diff, args(2..args(0))) )
	end_if
    end_if);
    
    random:= (if Dimen = 1 then
        (()->(new(dom, Set::random())))
    else
        subsop(hold((()->(new(dom, l)))), [1,2] = [ hold(Set::random()) $ Dimen ])
    end_if);

    coeffRing:= Set;

/* body of the domain */

begin
      if args(0) = 1 then Dimen:= 1
      elif args(0) = 2 then
          if not testtype(Dimen, Type::PosInt) then
              error("illegal dimension")
          end_if
      else error("wrong no of args") end_if;
      if domtype(Set) <> DOM_DOMAIN or 
           Set::hasProp(Cat::BaseCategory) <> TRUE 
      then
          error("not of category 'Cat::BaseCategory'")
      end_if
end_domain:

