/*++
Dom::SparseMatrixOld(C,R) -- sparse matrices with entries from the commutative
			  ring R and column indices from the ordered set C

Besides general matrix operations the main functionality is a procedure to
compute efficiently row echelon forms.

Examples for the generation of sparse matrices:

	>> A := array(1..3,1..3,[[2,0,4],[0,0,5],[3,6,0]]);

	>> M := Dom::Matrix(Dom::Integer)([[2,0,4],[0,0,5],[3,6,0]]);


	>> S:=Dom::SparseMatrixOld(Dom::Integer, Dom::Integer);

                  Dom::SparseMatrixOld(Dom::Integer, Dom::Integer)

                                +-       -+
                                | 2, 0, 4 |
                                |         |
                                | 0, 0, 5 |
                                |         |
                                | 3, 6, 0 |
                                +-       -+


	>> S(3, [1,2,3], [[[1,2],[3,4]],[[3,5]],[[1,3],[2,6]]]);

	>> S(3, [1,2,3], [[2,0,4],[0,0,5],[3,6,0]]);

	>> S([1,2,3], [[[1,2],[3,4]],[[3,5]],[[1,3],[2,6]]]);

	>> S([[[1,2],[3,4]],[[3,5]],[[1,3],[2,6]]]);

	>> S([1,2,3], [[2,0,4],[0,0,5],[3,6,0]]);

	>> S([1,2,3], A);

	>> S([1,2,3], M);

	>> S(%);
++*/


alias(NROWS(x) = extop(x,1),
      NCOLS(x) = extop(x,2),
      ALL_INDICES(x) = extop(x,3),
      ROWS(x) = extop(x,4),
      ARG_ERR = error("wrong no of args"),
      COL_ERR = error("column index out of range"),
      ROW_ERR = error("row index out of range"),
      TYPE_ERR = error("illegal type of argument")):


domain Dom::SparseMatrixOld(C,R)

    local Rzero, Rone, Risone, Riszero, Rcoerce, Rnegate, Rplus, Rmult, Rexpr,
	  Mat, Q, QMat, Qinvert, Qconvert, Qmult, Qiszero, remDup, sublist, 
	  less, Ccoerce, TyLO, TyAT;
    inherits Dom::BaseDomain;
    category Cat::Matrix(R);

    /* The representation has four slots:
      NCols is the number of columns,
      NRows the number of rows
      AllIndices is a list containing all labels of columns; it must always
	 have length NCols and must always be orderd
      Rows is a list; each element represents one row;
	 a row is a list; each element represents one entry;
	 an entry is a list with two elements: the first one is the label,
	 the second one is the corresponding value
      The representation is neither canonical nor normal, as there are no
      checks whether some entries vanish. */


    // =============== 
    // "local" methods 
    // =============== 

    sortRow :=
        proc(l:DOM_LIST):DOM_LIST
          option escape;
        begin sort(l, ((l1,l2) -> less(l1[1],l2[1]))) end_proc;


    containsIndex :=
        proc(r:DOM_LIST,c:C):DOM_INT
	local pos:DOM_INT;
	begin
	    pos := 1;
	    while r<>[] and r[1][1]<>c do
		delete r[1];
		pos := pos+1;
	    end_while;
	    if r=[] then 0 else pos end_if;
	end_proc;


    indices :=
        proc(l:DOM_LIST):DOM_LIST
        begin map(l, ((j) -> j[1])) end_proc;


    entries :=
        proc(l:DOM_LIST):DOM_LIST
        begin map(l, ((j) -> j[2])) end_proc;


    greater :=
        proc(r1:DOM_LIST,r2:DOM_LIST):DOM_BOOL
        begin
	    if r1=[] then FALSE
	    elif r2=[] then TRUE
	    elif r1[1][1]=r2[1][1] then bool(nops(r2)<nops(r1))
	    else bool(less(r1[1][1],r2[1][1]))
	    end_if;
	end_proc;


    // removes the gcd of a row 
    makePrimitive :=
        proc(r:DOM_LIST)
	local ld:DOM_LIST, g:R, i:DOM_INT;
	begin
	    if nops(r)=1 then return(r[1][2],[[r[1][1],Rone]]) end_if;
	    ld := dom::entries(r);
	    g := gcd(op(ld));
	    if Risone(g) then return(Rone,r) end_if;
	    ld := map(ld,R::_divide,g);
	    for i from 1 to nops(ld) do
		r[i][2] := ld[i];
	    end_for;
	    return(g,r);
	end_proc;


    // computes linear combination of two rows 
    addRows :=
        proc(d1:R,r1:DOM_LIST,d2:R,r2:DOM_LIST):DOM_LIST
	local res:DOM_LIST,c:C,e:R,summe:R,k:DOM_INT;
	begin
	    if r1=[] then
		if not(Risone(d2)) then
		    r2 := map(r2, ((l,d) -> [l[1],Rmult(l[2],d)]), d2);
		end_if;
		return(r2);
	    end_if;
	    if r2=[] then
		if not(Risone(d1)) then
		    r1 := map(r1, ((l,d) -> [l[1],Rmult(l[2],d)]), d1);
		end_if;
		return(r1);
	    end_if;

	    if not(Risone(d1)) then
		r1 := map(r1, ((l,d) -> [l[1],Rmult(l[2],d)]), d1);
	    end_if;
	    if not(Risone(d2)) then
		r2 := map(r2, ((l,d) -> [l[1],Rmult(l[2],d)]), d2);
	    end_if;

	    res := [ 0 $ nops(r1)+nops(r2)];
	    k := 0;
	    while r1<>[] do
	        if r2=[] then return([op(res,1..k)].r1) end_if;
		c := r1[1][1];
		e := r1[1][2];
		delete r1[1];
		while r2<>[] and less(r2[1][1],c) do
		    k := k+1;
		    res[k] := [r2[1][1],r2[1][2]];
		    delete r2[1];
		end_while;
		if r2<>[] and r2[1][1]=c then
		    summe := Rplus(e,r2[1][2]);
		    if not(Riszero(summe)) then
		        k := k+1;
			res[k] := [c,summe];
		    end_if;
		    delete r2[1];
		else
		    k := k+1;
		    res[k] := [c,e];
		end_if;
	    end_while;

	    [op(res,1..k)] . r2;
	end_proc;


    extractIndices :=
        proc(l:DOM_LIST):DOM_LIST
        local res:DOM_LIST,i:DOM_INT,r:DOM_LIST;
        begin
            res := [];
            for r in l do
                if r<>[] then
                    for i in r do
                        if i<>[] then
                            if iszero(contains(res,i[1])) then
                                res := append(res,i[1]);
                            end_if;
                        end_if;
                    end_for;
                end_if;
            end_for;
	    res;
        end_proc;


    coerceEntry :=
        proc(r:DOM_LIST):DOM_LIST
        begin
            if nops(r)<>2 then FAIL
            else [Ccoerce(r[1]),Rcoerce(r[2])] end_if;
        end_proc;


    newMat :=
        proc(inds:DOM_LIST,x):dom
	local i,j,k1,k2,ni,ind,nr,nc,rows,r,e;
	begin
	    if inds=[] then return(FAIL) end_if;
	    inds := map(inds,Ccoerce);
	    if not(iszero(contains(inds,FAIL))) then return(FAIL) end_if;
	    inds := sort(inds,less);

	    case domtype(x)
	    of DOM_ARRAY do       // only two-dimensional arrays allowed! 
		if op(x,[0,1])<>2 then return(FAIL) end_if;
		nr := op(x,[0,2,2]);
		nc := op(x,[0,3,2]);
		break;
	    of DOM_LIST do
		x := Mat::convert(x);
		if x=FAIL then return(FAIL) end_if;
	    otherwise:
		if not(x::dom::hasProp(Cat::Matrix)) then 
		    return(FAIL);
	        end_if;
		i := x::dom::matdim(x);
		nr := op(i,1);
		nc := op(i,2);
	    end_case;

	    rows := [ 0 $ nr ];
	    k1 := 0;
	    ni := nops(inds);
	    for i from 1 to nr do
		r := [ 0 $ ni ];
		k2 := 0;
		j := 1;
		for ind in inds do
		    e := Rcoerce(x[i,j]);
		    if not(Riszero(e)) then
		        k2 := k2+1;
		        r[k2] := [ind,e];
		    end_if;
		    j := j+1;
		end_for;
		k1 := k1+1;
		rows[k1] := [op(r,1..k2)];
	    end_for;
	    return(new(dom,nr,nc,inds,rows));
	end_proc;


    // ========================= 
    // representation, in/output 
    // ========================= 


    new := dom::convert;
    create := dom::convert;
    
	
    convert :=
        proc(x)
	local i, nr, inds, rows;
	begin
	    if testargs() then
		if (args(0)<1) or (args(0)>4) then ARG_ERR end_if;
	    end_if;

	    case args(0)
	    of 1 do
		if domtype(x)=dom then return(x) end_if;
		if not(testtype(x,TyLO(TyLO(TyLO(TyAT))))) then
		    return(FAIL)
		end_if;
		nr := nops(x);
		rows := x;
		inds := dom::extractIndices(x);
		break;
	    of 2 do
		case domtype(x)
		of DOM_INT do
		    nr := args(1);
		    inds := args(2);
		    rows := [];
		    break;
		of DOM_LIST do
		    if testtype(args(2),TyLO(TyLO(TyLO(TyAT)))) then
			nr := nops(args(2));
			inds := args(1);
			rows := args(2);
		    else return(dom::newMat(args())) end_if;
		    break;
		otherwise return(FAIL);
		end_case;
		break;
	    of 3 do
		nr := args(1);
		inds := args(2);
		if not(testtype(args(3),TyLO(TyLO(TyLO(TyAT))))) then
		    return(dom::newMat(inds,args(3)));
		end_if;
		rows := args(3);
		break;
	    otherwise return(FAIL);
	    end_case;

	    if testargs() then
		if (domtype(nr)<>DOM_INT) or (nr<1) or
		   (domtype(inds)<>DOM_LIST) or (inds=[]) then
		    return(FAIL)
		end_if;
		inds := map(inds,Ccoerce);
		if not(iszero(contains(inds,FAIL))) then return(FAIL) end_if;
		rows := map(rows, map, dom::coerceEntry);
                if has(rows,FAIL) then return(FAIL) end_if;
	    end_if;

	    for i from 1 to nops(rows) do
		rows[i] := dom::sortRow(rows[i]);
	    end_for;
	    if nops(rows)<nr then
		rows := rows . [[] $ nr-nops(rows)];
	    end_if;
	    new(dom,nr,nops(inds),sort(inds,less),rows);
        end_proc;


    convert_to :=
        proc(A:dom,t)
	begin
	    case t
	    of DOM_LIST do return(ROWS(A));
	    of DOM_ARRAY do return(dom::expr(A));
	    otherwise FAIL;
	    end_case;
	end_proc;


    ncols :=
        proc(A:dom):DOM_INT
        begin NCOLS(A) end_proc;


    nrows :=
        proc(A:dom):DOM_INT
        begin NROWS(A) end_proc;


    matdim :=
        proc(A:dom):DOM_LIST
        begin [NROWS(A),NCOLS(A)] end_proc;


    allIndices :=
        proc(A:dom):DOM_LIST
        begin ALL_INDICES(A) end_proc;


    rows :=
        proc(A:dom):DOM_LIST
        begin ROWS(A) end_proc;


    isSparse := TRUE;


    _index :=
        proc(A:dom,i:DOM_INT,c):R
	local r:DOM_LIST,pos:DOM_INT;
	begin
	    c := Ccoerce(c);
	    if testargs() then
		if c=FAIL then COL_ERR end_if;
		if iszero(contains(ALL_INDICES(A),c)) then COL_ERR end_if;
		if (i<1) or (i>NROWS(A)) then ROW_ERR end_if;
	    end_if;

	    r := ROWS(A)[i];
	    pos := dom::containsIndex(r,c);
	    if iszero(pos) then Rzero else r[pos][2] end_if;
	end_proc;


    row :=
        proc(A:dom,i):DOM_LIST
        begin
	    case type(i)
	    of DOM_INT do
		if testargs() then
		    if (i<0) or (i>NROWS(A)) then ROW_ERR end_if;
	    	end_if;
		return(ROWS(A)[i]);
	    of DOM_LIST do return(dom::extract(A,i));
	    of "_range" do return(dom::extract(A,[ $ i ]));
	    otherwise TYPE_ERR
	    end_case;
	end_proc;


    /* extracts the rows with indices in the list l and
      returns them as a new matrix */
    extract :=
        proc(A:dom,l:DOM_LIST): dom
	local i,j:DOM_INT,ra:DOM_LIST,nrows:DOM_LIST;
	begin
	    ra := ROWS(A);
	    nrows := [ 0 $ nops(l)];
	    j := 0;
	    for i in l do
		j := j+1;
		nrows[j] := ra[i];
	    end_for;
	    new(dom,nops(l),NCOLS(A),ALL_INDICES(A),nrows);
	end_proc;


    // expr returns a sparse matrix as an array in the usual MuPAD form 
    expr :=
        proc(A:dom):DOM_ARRAY
	local AA:DOM_ARRAY,i:DOM_INT,j:DOM_INT,allinds:DOM_LIST,
	      ind:C,nr:DOM_INT,nc:DOM_INT,r:DOM_LIST;
	begin
	    nc := NCOLS(A);
	    nr := NROWS(A);
	    if iszero(nc) then error("matrix with zero columns") end_if;
	    if iszero(nr) then error("matrix with zero rows") end_if;

	    AA := array(1..nr, 1..nc, [[Rexpr(Rzero) $ nc] $ nr]);
	    i := 0;
	    allinds := ALL_INDICES(A);
	    for r in ROWS(A) do
		i := i+1;
		j := 0;
		for ind in allinds do
		    if r=[] then break end_if;
		    j := j+1;
		    if ind=r[1][1] then
			AA[i,j] := Rexpr(r[1][2]);
			delete r[1];
		    end_if;
		end_for;
	    end_for;
	    AA;
	end_proc;


    iszero :=
        proc(A:dom):DOM_BOOL
        begin bool(ROWS(A) = [[] $ NROWS(A)]) end_proc;


    map :=
        proc(A:dom,f):dom
        begin
            new(dom,NROWS(A),NCOLS(A),ALL_INDICES(A),
	        map(ROWS(A),
		    (()->(map(args(1),
			      (()->([args(1)[1],
				     args(2)(args(1)[2],args(3..args(0)))])),
			      args(2..args(0))))),
		    f,args(3..args(0))));
	end_proc;


    // ======================================= 
    // manipulation and generation of matrices 
    // ======================================= 

    set_index :=
        proc(A:dom,i:DOM_INT,c,d):dom
	local rs:DOM_LIST,r:DOM_LIST,k:DOM_INT,pos:DOM_INT,newr:DOM_LIST;
	begin
	    if domtype(c)<>C then
		c := Ccoerce(c);
		if c=FAIL then COL_ERR end_if;
	    end_if;
	    if iszero(contains(ALL_INDICES(A),c)) then COL_ERR end_if;
	    if (i<1) or (i>NROWS(A)) then ROW_ERR end_if;
	    if domtype(d)<>R then
		d := Rcoerce(d);
		if d=FAIL then TYPE_ERR end_if;
	    end_if;

	    if Riszero(d) then return(A) end_if;
	    rs := ROWS(A);
	    pos := dom::containsIndex(rs[i],c);
	    if iszero(pos) then
		r := rs[i];
		newr := [ 0 $ NROWS(A) ];
		k := 0;
		while r<>[] and less(r[1][1],c) do
		    k := k+1;
		    newr[k] := r[1];
		    delete r[1];
		end_while;
		newr[k+1] := [c,d];
		newr := [op(newr,1..k+1)] . r;
		rs[i] := newr;
	    else rs[i][pos][2] := d end_if;
	    new(dom,NROWS(A),NCOLS(A),ALL_INDICES(A),rs);
	end_proc;


    setRow :=
        proc(A:dom,i:DOM_INT,r:DOM_LIST):dom
	local nr:DOM_INT,inds:DOM_LIST,ind:C;
	begin
	    nr := NROWS(A);
	    if testargs() then
	        if i<0 or i>nr then ROW_ERR end_if;
	    end_if;
	    r := map(r,dom::coerceEntry);
	    if has(r,FAIL) then TYPE_ERR end_if;
	    r := dom::sortRow(r);
	    inds := ALL_INDICES(A);
	    for ind in dom::indices(r) do
		if iszero(contains(inds,ind)) then COL_ERR end_if;
	    end_for;
	    new(dom,nr,NCOLS(A),inds,subsop(ROWS(A),i=r))
	end_proc;


    /* removes columns which contain only zeros.
      This effects only the values of ALL_INDICES and NCOLS */
    elimZeroCols :=
        proc(A:dom):dom
	local inds:DOM_LIST,r:DOM_LIST;
	begin
	    inds := [];
	    for r in ROWS(A) do
		inds := remDup(listlib::merge(inds,dom::indices(r),less));
	    end_for;
	    new(dom,NROWS(A),nops(inds),inds,ROWS(A));
	end_proc;


    // eliminates all columns with an index C such that crit(C)=TRUE 
    purge :=
        proc(A:dom,crit:DOM_PROC):dom
	local ra:DOM_LIST,inds:DOM_LIST,change:DOM_BOOL,k:DOM_INT,
	      c:C,rargs;
	begin
	    rargs := args(3..args(0));

	    inds := [ 0 $ nops(ALL_INDICES(A)) ];
	    change := FALSE;
	    k := 0;
	    for c in ALL_INDICES(A) do
		if not(crit(c,rargs)) then
		    k := k+1;
		    inds[k] := c
		else change := TRUE end_if;
	    end_for;

	    if change then
	        ra := map(ROWS(A), select,
			  ((r) -> not(crit(r[1],args(2..args(0))))), rargs);
	        new(dom,NROWS(A),k,[op(inds,1..k)],ra);
	    else return(A) end_if;
	end_proc;


    /* like "purge", however, with the additional assumption that crit
      respects the ordering of the indices */
    sortedPurge :=
        proc(A:dom,crit:DOM_PROC):dom
	local ra:DOM_LIST,i:DOM_INT,inds:DOM_LIST,rargs;
	begin
	    rargs := args(3..args(0));

	    inds := ALL_INDICES(A);
	    ra := ROWS(A);
	    if crit(inds[1],rargs) then
		while inds<>[] and crit(inds[1],rargs) do
		    delete inds[1]
		end_while;
		for i from 1 to NROWS(A) do
		    while ra[i]<>[] and crit(ra[i][1][1],rargs) do
			delete ra[i][1]
		    end_while;
		end_for;
	    end_if;
	    new(dom,NROWS(A),nops(inds),inds,ra);
	end_proc;


    delRow :=
        proc(A:dom,i:DOM_INT):dom
	local ra:DOM_LIST;
        begin
	    if testargs() then
		if (i<0) or (i>NROWS(A)) then ROW_ERR end_if;
	    end_if;
	    ra := ROWS(A);
	    delete ra[i];
	    new(dom, NROWS(A)-1, NCOLS(A), ALL_INDICES(A),ra);
	end_proc;


    // inserts a row r at the top of the matrix 
    consRow :=
        proc(A:dom,r:DOM_LIST):dom
        begin
	    r := map(r,dom::coerceEntry);
	    if has(r,FAIL) then TYPE_ERR end_if;
	    r := dom::sortRow(r);
	    if iszero(sublist(ALL_INDICES(A),
		      dom::indices(r))) then COL_ERR end_if;
	    new(dom, NROWS(A)+1, NCOLS(A), ALL_INDICES(A), [r].ROWS(A));
	end_proc;


    // appends a row at the end of the matrix 
    appendRow :=
        proc(A:dom,r:DOM_LIST):dom
        begin
	    r := map(r,dom::coerceEntry);
	    if has(r,FAIL) then TYPE_ERR end_if;
	    r := dom::sortRow(r);
	    if iszero(sublist(ALL_INDICES(A),
		      dom::indices(r))) then COL_ERR end_if;
	    new(dom, NROWS(A)+1, NCOLS(A), ALL_INDICES(A),
		append(ROWS(A),r));
	end_proc;



    stackMatrix :=
        proc():dom
        local n:DOM_INT;
        begin
            case args(0)
	    of 0 do ARG_ERR;
	    of 1 do return(args(1));
	    of 2 do
		if testargs() then
		    if domtype(args(1))<>dom or
		       domtype(args(2))<>dom then TYPE_ERR end_if;
		    if ALL_INDICES(args(1))<>
		       ALL_INDICES(args(2)) then COL_ERR end_if;
		end_if;
	        return(new(dom,NROWS(args(1))+NROWS(args(2)),
			   NCOLS(args(1)),ALL_INDICES(args(1)),
			   ROWS(args(1)).ROWS(args(2))));
	    otherwise
		n := args(0) div 2;
		dom::stackMatrix(dom::stackMatrix(args(1..n)),
				  dom::stackMatrix(args(n+1..args(0))));
	    end_case;
	end_proc;


    concatMatrix :=
        proc():dom
        local n:DOM_INT,inds:DOM_LIST;
        begin
            case args(0)
	    of 0 do ARG_ERR;
	    of 1 do return(args(1));
	    of 2 do
		if testargs() then
		    if NROWS(args(1))<>NROWS(args(2)) then
		        error("incompatible dimensions");
		    end_if;
		    if not(less(ALL_INDICES(args(1))[nops(ALL_INDICES(args(1)))],
		                ALL_INDICES(args(2))[1])) then
		        error("incompatible matrices");
		    end_if;
		end_if;
	        inds := ALL_INDICES(args(1)) . ALL_INDICES(args(2));
	        return(new(dom,NROWS(args(1)),nops(inds),inds,
			   zip(ROWS(args(1)),ROWS(args(2)),_concat)));
	    otherwise
		n := args(0) div 2;
		dom::concatMatrix(dom::concatMatrix(args(1..n)),
				   dom::concatMatrix(args(n+1..args(0))));
	    end_case;
	end_proc;


    /* splits the matrix A into two at the column given by c.
      the first column of the right matrix is enumerated by the
      first index greater or equal to c. */
    horizSplit :=
        proc(A:dom,c)
	local Rinds:DOM_LIST,Linds:DOM_LIST,Rrows:DOM_LIST,Lrows:DOM_LIST,
              ra:DOM_LIST,r:DOM_LIST,l:DOM_LIST,
              k:DOM_INT,kl:DOM_INT,ki:DOM_INT,n:DOM_INT;
        begin
	    c := Ccoerce(c);
	    if c=FAIL then TYPE_ERR end_if;

	    Rinds := ALL_INDICES(A);
	    if less(c,Rinds[1]) or (c=Rinds[1]) then return((0,A))
	    elif less(Rinds[nops(Rinds)],c) then return((A,0)) end_if;
	    n := nops(Rinds);
	    Linds := [ 0 $ n ];
	    ki := 0;
	    while less(Rinds[1],c) do
	        ki := ki+1;
		Linds[ki] := Rinds[1];
		delete Rinds[1];
	    end_while;
	    Linds := [op(Linds,1..ki)];

            ra := ROWS(A);
	    Lrows := [ 0 $ NROWS(A) ];
	    Rrows := [ 0 $ NROWS(A) ];
	    for k from 1 to NROWS(A) do
	        r := ra[k];
		l := [ 0 $ nops(r) ];
		kl := 0;
		while r<>[] and less(r[1][1],c) do
		    kl := kl+1;
		    l[kl] := r[1];
		    delete r[1];
		end_while;
		Lrows[k] := [op(l,1..kl)];
		Rrows[k] := r;
	    end_for;
	    ( new(dom, NROWS(A), ki, Linds, Lrows),
	      new(dom, NROWS(A), n-ki, Rinds, Rrows) );
	end_proc;


    swapRow :=
        proc(A:dom,i:DOM_INT,j:DOM_INT):dom
	local ra:DOM_LIST,tmp:DOM_LIST;
	begin
	    if testargs() then
		if (i<0) or (j<0) or
		   (i>NROWS(A)) or (j>NROWS(A)) then ROW_ERR end_if;
	    end_if;

	    if (i=j) then return(A) end_if;
	    ra := ROWS(A);
	    tmp := ra[i];
	    ra[i] := ra[j];
	    ra[j] := tmp;
	    new(dom,NROWS(A),NCOLS(A),ALL_INDICES(A),ra);
	end_proc;


    transpose :=
        proc(A:dom):dom
	local ra:DOM_LIST,nrows:DOM_LIST,inds:DOM_LIST,idx:DOM_LIST,
	      nr:DOM_INT,nc:DOM_INT,i:DOM_INT,j:DOM_INT,pos:DOM_INT;
	begin
	    nr := NROWS(A);
	    nc := NCOLS(A);
	    if nr<>nc then error("only possible for square matrices") end_if;

	    nrows := [[ 0 $ nr] $ nc];
	    idx := [ 0 $ nc ];
	    inds := ALL_INDICES(A);
	    ra := ROWS(A);
	    for j from 1 to nr do
		for i in ra[j] do
		    pos := contains(inds,i[1]);
		    idx[pos] := idx[pos]+1;
		    nrows[pos][idx[pos]] := [inds[j],i[2]];
		end_for;
	    end_for;
	    for i from 1 to nr do
	        nrows[i] := [op(nrows[i],1..idx[i])];
	    end_for;
	    new(dom,nc,nr,inds,nrows);
	end_proc;


    // ================ 
    // row echelon form 
    // ================ 

    pivot :=
        proc(A:dom,i:DOM_INT):DOM_LIST
	local r:DOM_LIST;
	begin
	    if testargs() then
		if (i<0) or (i>NROWS(A)) then ROW_ERR end_if;
	    end_if;

	    r := ROWS(A)[i];
	    if r=[] then error("empty row") end_if;
	    [r[1][1],r[1][2]];
	end_proc;


    pivots :=
        proc(A:dom):DOM_LIST
	local res:DOM_LIST,r:DOM_LIST,k:DOM_INT;
	begin
	    res := [ 0 $ NROWS(A) ];
	    k := 0;
	    for r in ROWS(A) do
		if r=[] then next end_if;
		k := k+1;
		res[k] := r[1];
	    end_for;
	    if k<NROWS(A) then [op(res,1..k)] else res end_if;
	end_proc;


    /* For GCD domains a primitive row echelon form is provided.
      It is computed by a fraction-free elimination where every row is made
      primitive by division by its GCD. */
    gaussElim :=
        proc(A:dom)
	local all:DOM_BOOL,prim:DOM_BOOL,ltr,pivs:DOM_LIST,pivlen:DOM_INT,
	      pivind:C,rk:DOM_INT,pivrow:DOM_INT,anrows:DOM_INT,
	      newr:DOM_LIST, early:DOM_BOOL,tmp,piv:R,c:C,fd:R,
	      r:DOM_LIST,pr,len:DOM_INT,rind:DOM_LIST,i:DOM_INT,j,k,l,q,
	      DOM:DOM_DOMAIN, heapify:DOM_PROC, heapSize:DOM_INT;
	begin
	    case args(0)
	    of 1 do
		all := FALSE;
		prim := FALSE;
		break;
	    of 2 do
		all := bool(args(2)=hold(All));
		prim := bool(args(2)=hold(Primitive));
		if not(all or prim) then error("unknown option") end_if;
		break;
	    of 3 do
		if {args(2),args(3)}<>{hold(All),hold(Primitive)} then
		    error("unknown option");
		end_if;
		all := TRUE;
		prim := TRUE;
		break;
	    otherwise ARG_ERR;
	    end_case;
	    if testargs() then
		if not(R::hasProp(Cat::IntegralDomain)) then
		    error("not a Cat::IntegralDomain");
		end_if;
		if prim and not(R::hasProp(Cat::GcdDomain)) then
		    error("primitive form only possible for a Cat::GcdDomain");
		end_if;
	    end_if;

	    anrows := NROWS(A);
	    if prim then
		ltr := QMat(anrows,anrows, ()-> Qconvert(Rone),hold(Diagonal));
	    else
		ltr := Mat(anrows,anrows, ()-> Rone, hold(Diagonal));
	    end_if;
	    pivs := [ 0 $ anrows ];

	    r := ROWS(A);
	    for i from 1 to anrows do
		while r[i]<>[] do
		    if not(Riszero(r[i][1][2])) then break end_if;
		    delete r[i][1];
		end_while;
	    end_for;

	    /* bubble sort 
	    notSorted := TRUE;
	    while notSorted do
		notSorted := FALSE;
		oldr := r[1];
		for i from 2 to anrows do
		    newr := r[i];
		    if dom::greater(newr,oldr) then
			r[i] := oldr;
			r[i-1] := newr;
			ltr := ltr::dom::swapRow(ltr,i-1,i);
			notSorted := TRUE;
		    else oldr := newr end_if;
		end_for;
	    end_while; */
	    
	    // heap sort 
	    
	    DOM := dom;
	    heapify := proc(i)
		local left:DOM_INT, right:DOM_INT, smallest:DOM_INT, tmp;
		begin
		    left := 2 * i;
		    right := 2 * i + 1;
		    if (left <= heapSize 
			and DOM::greater(r[i], r[left]))
		    then
			smallest := left;
		    else
			smallest := i;
		    end_if;
		    if (right <= heapSize
			and DOM::greater(r[smallest], r[right]))
		    then
			smallest := right;
		    end_if;
		    if smallest <> i then
			tmp := r[i];
			r[i] := r[smallest];
			r[smallest] := tmp;
			ltr := ltr::dom::swapRow(ltr, i, smallest);
			heapify(smallest);
		    end_if;
		end_proc;
		
	    heapSize := anrows;
	    for k from (anrows div 2) downto 1 do
		heapify(k);
	    end_for;
	    for k from anrows downto 2 do
		tmp := r[1];
		r[1] := r[k];
		r[k] := tmp;
		ltr := ltr::dom::swapRow(ltr, 1, k);
		heapSize := heapSize - 1;
		heapify(1);
	    end_for;
		
		    early := FALSE;
		    for i from 1 to anrows do
			if r[i]=[] then
			    rk := i-1;
			    early := TRUE;
			    break;
			end_if;

			// search good pivot (with "short" row) 
			pivind := r[i][1][1];
			pivlen := nops(r[i]);
			pivrow := i;
			k := 0;
			for j from i+1 to anrows do
			    rind := dom::indices(r[j]);
		    if rind=[] or pivind<>rind[1] then break end_if;
		    len := nops(rind);
		    k := k+1;
		    if len<pivlen then
			pivlen := len;
			pivrow := j;
		    end_if;
		end_for;

		if prim then
		    tmp := dom::makePrimitive(r[pivrow]);
		    pivs[i] := op(tmp,1);
		    if op(tmp,1)<>Rone then
			r[pivrow] := op(tmp,2);
			q := Qinvert(Qconvert(pivs[i]));
			for l from 1 to anrows do
			    if not(Qiszero(ltr[pivrow,l])) then
				ltr[pivrow,l] := Qmult(q,ltr[pivrow,l]);
			    end_if;
			end_for;
		    end_if;
		else pivs[i] := r[pivrow][1][2] end_if;
		piv := r[pivrow][1][2];

		// elimination 
		if not(iszero(k)) then
		    if pivrow<>i then
			pr := r[pivrow];
			r[pivrow] := r[i];
			r[i] := pr;
			ltr := ltr::dom::swapRow(ltr,i,pivrow);
		    end_if;
		    if prim then pr := op(tmp,2)
		    else pr := r[i] end_if;
		    delete pr[1];
		    for j from i+1 to i+k do
			newr := r[i+1];
			c := newr[1][2];
			delete newr[1];
			newr := dom::addRows(piv,newr,-c,pr);
			for l from 1 to anrows do
			    fd := piv*ltr[i+1,l] - c*ltr[i,l];
			    ltr[i+1,l] := fd;
			end_for;
			for l from i+2 to i*2+k+1-j do
			    r[l-1] := r[l];
			    ltr := ltr::dom::swapRow(ltr,l-1,l);
			end_for;
			for l from i*2+k+2-j to anrows do
			    if not(dom::greater(r[l],newr)) then break	end_if;
			    r[l-1] := r[l];
			    ltr := ltr::dom::swapRow(ltr,l-1,l);
			end_for;
			r[l-1] := newr;
		    end_for;
		end_if;
	    end_for;

	    if early then pivs := [op(pivs,1..rk)]
	    else rk := anrows end_if;
	    A := new(dom,NROWS(A),NCOLS(A),ALL_INDICES(A),r);
	    if all then (A,ltr,pivs,rk)
	    else A end_if;
	end_proc;


    // =========== 
    // arithmetics 
    // =========== 

    _negate :=
        proc(A:dom):dom
        begin dom::map(A,Rnegate) end_proc;


    _plus :=
        proc():dom
        local n:DOM_INT;
        begin
            case args(0)
	    of 1 do return(args(1));
	    of 2 do return(dom::plus2(args(1),args(2)));
	    otherwise
		n := args(0) div 2;
		dom::plus2(dom::_plus(args(1..n)),
			    dom::_plus(args(n+1..args(0))));
	    end_case;
	end_proc;


    plus2 :=
        proc(a:dom,b:dom):dom
	local i,j,k,nr:DOM_INT,nc:DOM_INT,inds:DOM_LIST,
	      ra:DOM_LIST,rb:DOM_LIST,newrows:DOM_LIST,
	      newr:DOM_LIST,t:R,ta:DOM_LIST,tb:DOM_LIST;
	begin
	    nr := NROWS(a);
	    nc := NCOLS(a);
	    inds := ALL_INDICES(a);
	    if nr<>NROWS(b) or inds<>ALL_INDICES(b) then
		error("incompatible matrices");
	    end_if;
	    ra := ROWS(a);
	    rb := ROWS(b);

	    newrows := [ 0 $ nr ];
	    for i from 1 to nr do
		newr := [ 0 $ nc ];
		k := 0;
		for j from 1 to nc do
		    ta := select(ra[i], ((r,c) -> bool(r[1]=c)), inds[j]);
		    tb := select(rb[i], ((r,c) -> bool(r[1]=c)), inds[j]);
		    if ta<>[] then ta := [op(ta,[1,2])] end_if;
		    if tb<>[] then tb := [op(tb,[1,2])] end_if;
		    t := Rplus(op(ta),op(tb));
		    if not(Riszero(t)) then
		        k := k+1;
			newr[k] := [op(inds,j),t];
		    end_if;
		end_for;
		newrows[i] := [op(newr,1..k)];
	    end_for;
	    new(dom,nr,nc,inds,newrows);
	end_proc;


    _mult :=
        proc():dom
        local n:DOM_INT;
        begin
            case args(0)
	    of 1 do return(args(1));
	    of 2 do return(dom::mult2(args(1),args(2)));
	    otherwise
		n := args(0) div 2;
		dom::mult2(dom::_mult(args(1..n)),
			   dom::_mult(args(n+1..args(0))));
	    end_case;
	end_proc;


    mult2 :=
        proc(a,b):dom
	local nca:DOM_INT,nra:DOM_INT,ncb:DOM_INT,nrb:DOM_INT,i,k,newr:DOM_LIST,
	      r:DOM_LIST,res:DOM_LIST,ind:DM_IDENT,vtmp:DOM_ARRAY,
	      allinds:DOM_LIST,dim:DOM_LIST,idx:DOM_LIST;
	begin
	    if b::dom <> dom and a::dom = dom then
	      [a, b] := [b, a];
	    end_if;
	    if b::dom <> dom then
	      error("illegal argument ".a.", expected matrix of type ".dom);
	    end_if;
	    case domtype(a)
	    of dom do return(dom::thisMult(a,b));
	    of DOM_ARRAY do
		dim := [op(a,[0,2,2]),op(a,[0,3,2])];
		break;
	    of Mat do
		dim := Mat::matdim(a);
		break;
	    otherwise
		if domtype(a)<>R  then
		    a := Rcoerce(a);
		    if a=FAIL then TYPE_ERR end_if;
		end_if;
		return(dom::map(b, ((r,a) -> Rmult(a,r)), a));
	    end_case;

	    nra := op(dim,1);
	    nca := op(dim,2);
	    nrb := NROWS(b);
	    ncb := NCOLS(b);
	    allinds := ALL_INDICES(b);
	    if nca<>nrb then error("incompatible matrix dimensions") end_if;
	    res := [ [0 $ ncb] $ nra];
	    idx := [ 0 $ nra ];
	    r := ROWS(b);
	    for ind in allinds do
		vtmp := array(1..nra,[Rzero $ nra]);
		for i from 1 to nrb do
		    newr := r[i];
		    if (newr=[]) or (newr[1][1]<>ind) then next end_if;
		    for k from 1 to nra do
			if Riszero(a[k,i]) then next end_if;
			vtmp[k] := Rplus(vtmp[k],Rmult(a[k,i],newr[1][2]));
		    end_for;
		    delete newr[1];
		    r[i] := newr;
		end_for;
		for k from 1 to nra do
		    if Riszero(vtmp[k]) then next end_if;
		    idx[k] := idx[k]+1;
		    res[k][idx[k]] := [ind,vtmp[k]];
		end_for;
	    end_for;
	    for i from 1 to nra do
	        res[i] := [op(res[i],1..idx[i])];
	    end_for;
	    new(dom,nra,nca,allinds,res);
	end_proc;


    thisMult :=
        proc(a:dom,b:dom):dom
	local nr:DOM_INT,nc:DOM_INT,ra:DOM_LIST,rb:DOM_LIST,newrows:DOM_LIST,
	      newr:DOM_LIST,inds:DOM_LIST,i:C,j:DOM_INT,
	      ta:DOM_LIST,tb:DOM_LIST,c:DOM_LIST,t:R;
	begin
	    nr := NROWS(a);
	    nc := NCOLS(a);
	    inds := ALL_INDICES(a);
	    if nr<>nc or NROWS(b)<>NCOLS(b) then
		error("not a square matrix")
	    end_if;
	    if inds<>ALL_INDICES(b) then error("incompatible matrices") end_if;

	    b := dom::transpose(b);
	    newrows := [];
	    for ra in ROWS(a) do
		newr := [];
		j := 1;
		for rb in ROWS(b) do
		    c := [];
		    for i in inds do
			ta := select(ra, ((r,c) -> bool(r[1]=c)), i);
			tb := select(rb, ((r,c) -> bool(r[1]=c)), i);
			if ta<>[] and tb<>[] then
			    c := append(c,Rmult(op(ta,[1,2]),op(tb,[1,2])));
			end_if;
		    end_for;
		    t := Rplus(op(c));
		    if not(Riszero(t)) then
			newr := append(newr,[op(inds,j),t]);
		    end_if;
		    j := j+1;
		end_for;
		newrows := append(newrows,newr);
	    end_for;
	    new(dom,nr,nc,inds,newrows);
	end_proc;


    characteristic := R::characteristic;


begin
    if args(0)<2 or args(0)>3 then ARG_ERR end_if;
	if domtype(C)<>DOM_DOMAIN or C::hasProp=FAIL or
	   not(C::hasProp(Cat::OrderedSet)) then
	    error("first argument must be Cat::OrderedSet")
	end_if;
	if domtype(R)<>DOM_DOMAIN or R::hasProp=FAIL or
	   not(R::hasProp(Cat::Rng)) then
	    error("second argument must be Cat::Rng");
	end_if;

	if args(0)=2 then less := C::_less;
	else less := args(3) end_if;

	Rzero := R::zero;
	Riszero := R::iszero;
	Rcoerce := R::coerce;
	Rnegate := R::_negate;
	Rplus := R::_plus;
	Rmult := R::_mult;
	Rexpr := R::expr;
	Mat := Dom::/*Dense*/Matrix(R);
	if R::hasProp(Cat::IntegralDomain) then
	    Q := Dom::Fraction(R);
	    QMat := Dom::/*Dense*/Matrix(Q);
	    Qinvert := Q::_invert;
	    Qconvert := Q::convert;
	    Qmult := Q::_mult;
	    Qiszero := Q::iszero;
	    Rone := R::one;
	    Risone := R::isone;
	end_if;

	Ccoerce := C::coerce;
	remDup := listlib::removeDuplicates;
	sublist := listlib::sublist;
	TyLO := Type::ListOf;
	TyAT := Type::AnyType;

end_domain:


unalias(NROWS,NCOLS,ALL_INDICES,ROWS,ARG_ERR,COL_ERR,ROW_ERR,TYPE_ERR):

