//               
// stefanw, 12.02.1996 

/*++

Dom::SymmetricGroup(n) - 
symmetric group of order n, i.e. the group of bijective mappings
of { 1..n} onto itself

n - integer >= 1

An element is represented as a domain element with operands:

0 - domain Dom::SymmetricGroup(n)
1 - a list [a_1, ... ,a_n ] of n integers, representing the
    permutation that maps i to a_i for each i between 1 and n 

++*/

domain Dom::SymmetricGroup( n:Type::PosInt )
    inherits Dom::BaseDomain;
    category Cat::Group;
    axiom Ax::canonicalRep;

/* entries */

    size:= fact(n);

    one:= new(dom, [ $1..n ]);

    convert:= proc(x)

    begin
    case domtype(x)
     of dom do 
      return(x);
     of DOM_ARRAY do 
      return(dom::convert([op(x)]));
     of DOM_LIST do
      if {op(map(x,domtype)) } <> { DOM_INT } then
         error("List must consist of integers");
      elif has(map(x, y -> bool(y<= n and y > 0 ) ),
      FALSE) then
         error("List contains integers outside the range");
      elif nops(x) <> n then
         error("List has incorrect length");
      elif nops({ op(x) } ) <> n then
         error("No permutation (not bijective)");
      else
         return(new(dom, x));
      end_if;
    end_case;
        FAIL
    end_proc;

    convert_to:= proc(x, T)

    begin
	 case T
	   of DOM_LIST do	
		  return(extop(x,1));
	   of dom do 
       return(x);
	 end_case;
	 FAIL
    end_proc;

    cycles:= proc(a)
	/* returns a list of cycles c_1,..,c_k such that a=c_1*..*c_k, each
 	  cycle is written as a list */
	
	local l,i,k,cyc,result;
	// l[i] is TRUE iff i is already involved in a cycle 

	begin
	l:=[ FALSE $ n ];
	cyc:=[];
	result:=[];
	for i from 1 to n do	
		if not l[i] then
			k:=i; 
			repeat
				l[k]:=TRUE;
				cyc:=append(cyc,k);
				k:=op(extop(a,1),k);				
			until i=k end_repeat;
			result:=append(result,cyc);
			cyc:=[];
		end_if;
	end_for;
	result
	end_proc;

    // expr: returns permuatation as a list 
    expr:=  x -> extop(x,1);
					 
    print:= x -> extop(x,1);

    _mult:= proc()
	local i;
	begin
    	if args(0) = 2 then
	    if domtype(args(2)) <> dom or 
	       domtype(args(1)) <> dom then
	    	    FAIL
	    else
	        new(dom, [ op(extop(args(1),1),op(extop(args(2),1),i)) $ i=1..n])
	    end_if
    	elif args(0) = 1 then
    	    args(1)
    	else
    	    _mult(args(1..(args(0) div 2)));
    	    _mult(args(((args(0) div 2) + 1)..args(0)));
	    if (domtype(%2))::_mult <> FAIL then (domtype(%2))::_mult(%2, %1)
	    elif (domtype(%1))::_mult <> FAIL then (domtype(%1))::_mult(%2, %1)
	    else _mult(%2, %1)
	    end_if
    	end_if
        end_proc;

 //    "_power" : inherited from category constructor 

    _invert:= proc(x)
	local result,i;

	begin
	 for i from 1 to n do 
		result[op(extop(x,1),i)]:=i;
	 end_for;
 	 new(dom,[ result[i] $ i=1..n ])
	end_proc;

//    "_divide" : defined in Cat::Group  

    order:= proc(a)
	local c,i;

	begin
	 c:=dom::cycles(a);
	 ilcm(nops(op(c,i)) $ i=1..nops(c))
	end_proc;

    func_call:= proc(sigma:dom, ll)
	local cp,l;

	begin
	l:=context(ll);
	if type(l)=DOM_INT then
		if l<1 or l>n then
			error("Permutation is only defined for arguments between 1 and ".
				expr2text(n));
		else
			return(op(extop(sigma,1),l))
		end_if
	elif type(l)=DOM_LIST then
		if nops(l)<>n then
			error("List must have ".expr2text(n)." elements") 
		end_if;
		cp:=l;
		map([$1..n], (()->((cp[op(extop(sigma,1),args(1))]:=l[args(1)]))) );
		return (cp) 		
	else
		error("Second argument has wrong type")
	end_if
	end_proc;

   inversions:=proc(sigmad:dom)
   local i,result,sigma;

   begin
   sigma:=extop(sigmad,1);
   result:=0;
   for i from 1 to n-1 do
      result:=result+nops(select([$i+1..n], (()->(sigma[args(1)]<sigma[i]))))
   end_for;
   end_proc;
         
   sign:=proc(sigma:dom)
	begin
	map(dom::cycles(sigma),nops);
	map(%,_plus, -1);
	(-1)^_plus(op(%))
	end_proc;

   /*--
    descent(p:permutation):  Returns the descent set of p

    That is the list of the i for which p[i]>p[i+1]

    written by Nicolas Thiery
    --*/

    descent := proc(p:dom)
        local Desc, i;
        begin
            Desc:=[];
            for i from 1 to nops(extop(p,1))-1 do
                if p(i)>p(i+1) then
                    Desc:=Desc.[i];
                end_if
            end_for;
            Desc;
        end_proc;

    /*--
    descentPoly(p:permutation, z:DOM_IDENT) -

    Returns the descent polynomial of p.
    That is the product of all the (z[1]..z[i]), where i are descents of p.

    References: Garsia and Stanton 197? 

    written by Nicolas Thiery
    --*/

    descentPoly := proc(p:dom, z:DOM_IDENT)
        local DescPol,Pol,i;
        begin
            DescPol:=1;
            Pol:=1;
            for i from 1 to nops(extop(p,1))-1 do
                Pol:=Pol*z[p(i)];
                if p(i)>p(i+1) then
                    DescPol:=DescPol*Pol;
                end_if
            end_for;
            DescPol;
        end_proc;

    allElements:= () -> map(combinat::permute(n), dom::convert); 

    TeX:= x-> expr2text(extop(x,1));

    random:= proc()
	local beenthere,rnd,result,i,a;

	begin
	 beenthere:= [ FALSE $ n ];
         rnd:=random(n);    // random generator for 0..n-1 
	 result:=[];
	 for i from 1 to n do
		repeat a:=rnd()+1 until beenthere[a]=FALSE end_repeat;
		result:=append(result,a);
		beenthere[a]:=TRUE;
	 end_for;
	 new(dom,result);
	end_proc;

/* body of the domain */

begin
    if args(0) <> 1 then error("wrong no of args") end_if;
    userinfo(10, "Creating symmetric group of order ".expr2text(n))
end_domain:

// end of file 
