/*
Diese Routine wird in Dom::/*Sparse*/Matrix und Dom::DenseMatrix
fuer den Spezialfall Ring = Dom::IntegerMod(n) in einen
_divide bzw. _invert-Zweig aufgerufen. Z.B.  in 
Dom::/*Sparse*/Matrix::_invert:

>   elif R::hasProp( Cat::Ring ) then
>  
      if R::constructor = Dom::IntegerMod then
         return(Dom::Utilities::IntModInverse(...))
      end_if;

>     // Try to determine the inverse matrix via the formula
>     //
>     //          det(x)^-1 * adjoint(x)
*/
//--------------------------------------------------------------
// Walter Oevel + Kai Gehrs, 20.11.2002
//--------------------------------------------------------------
// Dom::Utilities::IntModInverse(A) -- 
//       modular inverse of the matrix A over a
//       non-integral coefficient domain IntegerMod(N)
//
// Calls:  IntModInverse(M)
//         IntModInverse(A)
//         IntModInverse(B, N)
//
// Parametres:  M -- a square matrix over the coefficient ring
//                   Dom::IntegerMod(N), where N is not 
//                   necessarily a prime (i.e. Dom::IntegerMod(N)
//                   is not an integral domain)
//              A -- a square array(1..n, 1..n) with entries
//                   of type Dom::IntegerMod(N)
//              B -- a square array(1..n, 1..n) with entries
//                   of type DOM_INT. In this case, the
//                   modulus N must be specified as well.
//              N -- an integer >= 2 (the modulus)
//
// Return Value: The modular inverse of A as a matrix of the
//               same domain type as A.
//               FAIL is returned if A is not invertible.
//
// Details:
//   -- This is a specialized routine for 'fast' 
//      (well, 'reasonably fast') inversion of
//      matrices of type
//          Dom::DenseMatrix(Dom::IntegerMod(N) or
//          Dom::/*Sparse*/Matrix(Dom::IntegerMod(N))
//      ** where N needs not be a prime **
//   -- It also works if N is a prime (i.e., IntegerMod(N)
//      is a field). 
//      Timings have shown that
//      1/ Dom::/*Sparse*/Matrix(..)(..) is a bit faster whilst
//      1/ Dom::DenseMatrix(..)(..) is significantly slower 
//      than IntModInverse (Dom::DenseMatrix is slower by 
//      a factor of 12). This should be double checked
//      (Probably this is caused by the fact that Dom::DenseMatrix
//      uses the **library domain** Dom::IntegerMod for its
//      arithmetic, whereas IntModInverse and Dom::/*Sparse*/Matrix
//      use the modular kernel arithmetic).
//   -- !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
//      !! IntModInverse accepts Dom::/*Sparse*/Matrices, but
//      !! it does not really work in a sparse way internally.
//      !! For the sparse case, further special tuning of the
//      !! routine is needed!
//      !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
//   -- IntModInverse over a non-integral domain
//      (i.e., N is not prime) is much faster than the
//      Berkowitz algorithm for computing the determinant!
//      A corresponding routine IntModDet should be
//      implemented for IntMod(N), N <> prime, to avoid
//      Berkowitz!
//--------------------------------------------------------------
//--------------------------------------------------------------
Dom::Utilities::IntModInverse:= proc(A)
local dtype, m, n, Ring, N, L, isunit, units, swap, 
      pivotFound, pivotCandidates, pivot, pivotIndex,
      i, j, k, x, f/*, det, s*/;
begin

  // we need the positive representation
  // {0,1,..,N-1} of IntegerMod(N), so 
  // we call modp(x, N) instead of x mod N
  // throughout this code

  dtype:= A::dom:
  if dtype::hasProp(Cat::Matrix) = TRUE then
     [m, n]:= dtype::matdim(A):
     if m <> n then
        error("expecting a square matrix"):
     end_if:
     Ring:= dtype::coeffRing:
     if Ring::constructor = Dom::IntegerMod then
          N:= Ring::characteristic(Ring):
     else N:= FAIL;
     end_if:
     if N = FAIL then 
        error("expecting a matrix over the ".
              "coefficient ring Dom::IntegerMod(N)"
             );
     end_if;
     A:= expr(A);
  elif dtype = DOM_ARRAY then
     m:= op(A, [0, 2, 2]):
     n:= op(A, [0, 3, 2]):
     if m <> n then
        error("expecting a square matrix"):
     end_if:
     Ring:= (A[1,1])::dom:
     if Ring::constructor = Dom::IntegerMod then
        // A is a matrix with Dom::IntegerMod entries.
        // Extract the modulus N from the first element
        N:= Ring::characteristic(Ring):
        A:= map(A, expr):
     else
        // A is a plain matrix with integers.
        // The modulus N must be passed by the user
        if args(0) = 1 then 
          error("expecting a second argument (the modulus)");
        end_if:
        N:= args(2):
        if domtype(N) <> DOM_INT or N < 2 then
           error("2nd argument: expecting an integer >= 2");
        end_if;
        // A:= map(A, modp, N); // This is done below, anyway
        Ring:= Expr;
     end_if;
  else
     error("expecting a matrix of category ".
           "'Cat::Matrix' or an array");
  end_if:

  // Initialize lower factor L of LR decomposition
  // Use the input matrix A to store the upper factor 
  L:= array(1..n, 1..n, [[0$n]$n]):
  for i from 1 to n do L[i,i]:= 1 end_for:

  //-----------------------------------------------
  // utilities: isunit and swap
  //-----------------------------------------------
  isunit:= x -> (igcd(x, N) = 1);
  //-----------------------------------------------
  swap:= proc(j, i) // swap the rows j and i in the
                    // jth step of the Gauss algorithm
  local k;
  begin
     if j = i then return() end_if;
     // s:= - s; // the sign of the determinant
     for k from j to n do
       [A[j, k], A[i, k]]:= [A[i, k], A[j, k]];
     end_for;
     for k from 1 to n do
       [L[j, k], L[i, k]]:= [L[i, k], L[j, k]];
     end_for:
  end_proc:
  //----------------------------------------------------------
  // At this point, A is an array of integers. Note, however, 
  // that if the user has set _mod = mods, these integers may 
  // be negative. The following algorithm is based on the
  // assumption that the input numbers are from the range
  // 0..N-1. Otherwise, the "minimal element" among the pivot
  // candidates, if negative, may not reduce the other
  // pivot candidates and the loop 'while TRUE do' below
  // may not terminate. So make sure, the integers in A
  // a positive by mapping modp: 
  //----------------------------------------------------------

  A:= map(A, modp, N);

  //----------------------------------------------------------
  // Begin of Gaussian elimination
  //----------------------------------------------------------

  // s:= 1; // the sign of the determinant

  for j from 1 to n do // eliminate in column j
    //-----------------------------
    // search for pivot in column j
    //-----------------------------
    pivotFound:= FALSE;
    while TRUE do
       pivotCandidates:= [[A[i, j], i] $ i = j..n]:
       pivotCandidates:= select(pivotCandidates, x -> _not(iszero(modp(x[1], N))));
       if nops(pivotCandidates) = 0 then
          break; // there are no pivot elements,
                 // the determinant is 0.
       end_if;
       units:= select(pivotCandidates, x -> isunit(x[1]));
       if nops(units) <> 0 then 
          // choose the first unit as the pivot element
          pivotFound:= TRUE;
          [pivot, pivotIndex]:= units[1];
          break; // a pivot was found
       end_if;
       if nops(pivotCandidates) = 1 then
          // there is just one pivotCandidate and it
          // is no unit --> the matrix is not invertible
          break;
       end_if;
       // initialize the pivot as the first element
       pivot:= pivotCandidates[1][1];
       pivotIndex:= pivotCandidates[1][2];
       // search for the smallest element in the candidates
       for i from 2 to nops(pivotCandidates) do
           if pivotCandidates[i][1] < pivot then
              [pivot, pivotIndex]:= pivotCandidates[i];
           end_if;
       end_for:
       for x in pivotCandidates do
           // Reduce all rows j .. n modulo
           // the smallest pivot candidate.
           // Note: x = [A[i, j], i]
           if x[2] = pivotIndex then
              next;
           end_if;
           f:= (A[x[2], j]) div pivot;
           // A[i, j] mod A[pivotIndex, j] = A[i,j] - f*A[pivotIndex, j]
           // This is the crucial reduction:
           //   pivotcandidate = pivotcandidate mod (smallest pivotcandidate)
           A[x[2], j]:= modp(modp(A[x[2], j], pivot), N);
           // the reduction in the last line correponds to
           // subtracting f times the pivot row from the 
           // current row:
           for k from j+1 to n do
               A[x[2], k]:= modp(A[x[2], k] - f*A[pivotIndex, k], N);
           end_for:
           for k from 1 to n do
               L[x[2], k]:= modp(L[x[2], k] - f*L[pivotIndex, k], N);
           end_for:
       end_for:
    end_while;
    //------------------------------------
    // end of search for pivot in column j
    //------------------------------------
    if not pivotFound then
         // warning("the matrix is not invertible");
         return(FAIL);
    else // swap the j-th row and the pivot row
         swap(j, pivotIndex);
    end_if;
    //------------------------------------
    // Eliminate below the j-th row 
    //------------------------------------
    for i from j+1 to n do
        if iszero(A[i, j]) then
           next;
        end_if;
        // the pivot A[j, j] is a unit, so the
        // following division mod N must work:
        f:= modp(A[i, j]/A[j, j], N);
        // reduce the element below the pivot element:
        A[i, j]:= 0; 
        // Elimination of the i-th row via the jth row
        // A[i, j] mod A[pivotIndex, j] = A[i,j] - f*A[j, j]
        for k from j+1 to n do
            A[i, k]:= modp(A[i, k] - f*A[j, k], N);
        end_for:
        for k from 1 to n do
            L[i, k]:= modp(L[i, k] - f*L[j, k], N);
        end_for:
     end_for;
    //--------------------------------------
    // end of elimination below the j-th row 
    //--------------------------------------
  end_for:

  // // If the determinant is to be computed:
  // det:= (s*_mult(A[j, j] $ j = 1 .. n)) mod N;
  // // Use det mod N instead of modp(det, N), because
  // // the user may have redefined _mod:= mods !
  // return(det);  // if only det is to be computed

  //-------------------------------------------------------
  // We have L*A_original = R. Solve for X = A_original^(-1)
  // by backsubstitution of R*X = L:
  //-------------------------------------------
  for j from 1 to n do 
    for i from n downto 1 do 
      L[i, j]:= modp((L[i,j]-_plus(A[i,k]*L[k,j] $ k=i+1..n)) 
                     / A[i,i] , N);
    end_for:
  end_for:

  //----------------------------------------------
  // Postprocessing:
  // The inverse is stored in L. Convert the array
  // L to the domain type of the input matrix
  //----------------------------------------------
  if dtype = DOM_ARRAY then
       if Ring = Expr then
            // convert the modp-Numbers in L
            // to the style set by the user:
            if _mod <> modp then
               L:= map(L, _mod, N);
            end_if;
            return(L)
       else // Let Ring = Dom::IntegerMod(N)
            // handle the conversion between
            // modp and mods.
            return(map(L, Ring))
       end_if;
  else // Let dtype = Dom::(Sparse)Matrix(Dom::IntegerMod(N))
       // handle the conversion between modp and mods
       return(dtype(L)); 
  end_if;
end_proc:
//---------------  end of code file --------------------------------
