/*++
generate::optimize -- optimize expression or list of equations

generate::optimize(e)

e - expression, equation or list of equations

'optimize' optimizes expressions or equations representing assignments
such that common subexpressions are computed only once.

- A single expression 'e' is interpreted as the lhs of an assignment.
- A single equation a=b represents an assignment a:=b.
- A list of equations represents a list of assignments.

The function returns a list of equations representing assignments. New
'temporary variables' t.i may be generated which are used as lhs of newly
generated auxiliary expressions.

Examples:

>> generate::optimize(ResetVarGen);
>> generate::optimize(x = f(a+b)-(a+b)*c^(a+b));

   [t3 = a + b, x = f(t3) - c^t3*t3]

>> generate::optimize(ResetVarGen);
>> generate::optimize([x=a+b, y=f(a+b) + f(x)^2]);

   [t1 = a + b, x = f(t1) - c^t1*t1]

   
>> generate::optimize(ResetVarGen);
>> // THIS IS AN INVALID USE OF 'generate::optimize'!!!
>> [t1=5,t2=42,x=t2^2].generate::optimize(sin(a^2)+a^2)

   [t1 = 5, t2 = 42, x = t2^2, t2 = a^2, t1 = t2 + sin(t2)]
++*/

generate::optimize:= proc(e, noRepSquaring=FALSE) 
    local p, f, r; 
begin
    // wrong number of arguments
    if args(0) < 1 or args(0) > 2 then 
       error("wrong number of arguments")
    end_if;
    
    // check the second value
    if noRepSquaring <> TRUE and noRepSquaring <> FALSE then
       error("second argument must be TRUE or FALSE");
    end_if;
   
    // initialize name generator for new variables
    if generate::OptimizeNewVar = FAIL or e = hold(ResetVarGen) then 
       sysassign(generate::OptimizeNewVar, prog::genidentgenerator("t", "", 1, 0));
       if e = hold(ResetVarGen) then 
          return(null());
       end_if; 
    end_if;

    // accept list of equations or convert to a list of an equation
    if domtype(e) = DOM_LIST then
       if e = [] then
          return([]);
       end_if;
       if {op(map(e, type))} <> {"_equal"} then
          error("list of equations expected")
       end_if
    elif type(e) = "_equal" then
       e:= [e]
    else
       e:= [generate::OptimizeNewVar(e)=e]
    end_if;
        
    // optimize each rhs of the assignments in turn
    p:= [];
    for f in e do
        r:= generate::opt_expr(op(f,2), p, [], FALSE, e);
        p:= append(r[2], op(f,1)=r[1]);
    end_for;

    p;
end_proc:

/*--
generate::opt_expr -- optimize subexpression

generate::opt_expr(e,p,s,noRepSquaring,doNotGenTheseIdents)

e - subexpression to optimize
p - list of already optimized assignments
s - list of already optimized siblings of e and of the parents of e
noRepSquaring       - TRUE or FALSE
doNotGenTheseIdents - holds the original expression to optimize

Returns a list [e1,p1,s1,c] with:-
e1 - optimized expression
p1 - list of optimized assignments
s1 - list of optimized siblings of e1 and of the parents of e1
c  - boolean indicating that e actually has changed or not
--*/

generate::opt_expr:= proc(e, p, s, noRepSquaring=FALSE, doNotGenTheseIdents=[]) 
    local i, r, g, c, l, m, f, j;
begin
    if contains({DOM_EXPR, DOM_LIST, DOM_SET, DOM_TABLE, DOM_ARRAY, Dom::Matrix()}, domtype(e)) then
        // optimize operands of e 
        c:= FALSE;
        l:= s;
        m:= nops(s);
        f:= [op(e)];
        for i from 1 to nops(f) do
            r:= generate::opt_expr(f[i], p, l, noRepSquaring, doNotGenTheseIdents);
            if r[4] then
                // replace operands by optimized expressions 
                f:= subsop(f, i=r[1], (j=r[3][j+m]) $ j=1..(i-1));
                p:= r[2];
                l:= append(r[3], r[1]);
                c:= TRUE
            else
                l:= append(l, f[i])
            end_if
        end_for;
        e:= subsop(e, (j=f[j]) $ j=1..nops(f));
        s:= [op(l, 1..m)];

        // don't optimize both x and -x 
        if type(e)="_mult" and (i:=select(e,testtype,Type::Union(DOM_INT,DOM_RAT)))<0 then
            g:= generate::opt_expr(-e, p, s, noRepSquaring, doNotGenTheseIdents);
            return(subsop(g, 1=-g[1], 4=TRUE));
            // don't optimize both x and 1/x 
            // ccr: commented this out, because optimize(1/x^2*exp(-1/x^2)) was broken
//      elif type(e)="_power" and contains({DOM_INT,DOM_RAT},type(op(e,2))) and op(e,2)<0 then
//          g:=generate::opt_expr(1/e, p, s, noRepSquaring, doNotGenTheseIdents);
//          return(subsop(g,1=1/g[1],4=TRUE))
        end_if;

        // is e subexpression of one of the optimized expressions? 
        for i from 1 to nops(p) do
            // has e already a variable with its value? 
            if op(p,[i,2]) = e then
                return([op(p,[i,1]), p, s, TRUE]);
            end_if;

            // is e a subexpression to be assigned to a new variable? 
            // g496498: e changed to [e] since 'has' behaves different on lists.
            // failing example was: generate::optimize(y=[cos(x^2),2,cos(x^2)]). 
            if has(op(p,[i,2]), [e]) then
                g:= generate::OptimizeNewVar(doNotGenTheseIdents);
                p:= append([op(p,1..(i-1))], g=e, subs(p[i], e=g)).[op(p,(i+1)..nops(p))];
                return([g, p, s, TRUE]);

            elif type(e)="_mult" and has(op(p,[i,2]), -e) then
                g:= generate::OptimizeNewVar(doNotGenTheseIdents);
                p:= append([op(p,1..(i-1))], g=e, subs(p[i], -e=-g)).[op(p,(i+1)..nops(p))];
                return([g, p, s, TRUE]);
            end_if
        end_for;

        // is e equal to a former sibling? 
        // g496498: e changed to [e] since 'has' behaves different on lists.
        if has(s, [e]) then
            // create new assignment (e is already optimized before) 
            g:= generate::OptimizeNewVar(doNotGenTheseIdents);
            return([g, append(p, g=e), subs(s, e=g), TRUE])
        end_if;

        // optimize integer powers further by repeated squaring 
        if not noRepSquaring and type(e) = "_power" then
            l:= op(e,2);
            if domtype(l) = DOM_INT then
                if l > 1 and l <= 1000 then
                    e:= op(e,1);
                    if domtype(e) = DOM_EXPR then
                        g:= generate::OptimizeNewVar(doNotGenTheseIdents);
                        p:= append(p, g=e);
                        e:= g;
                        c:= TRUE;
                    end_if;
                    if l = 2 then 
                        return([e^2, p, s, c])
                    end_if;
                    l:= numlib::g_adic(l, 2);
                    f:= (if l[1] = 1 then e else 1 end_if);
                    c:= e;
                    for i from 2 to nops(l) do
                        r:= generate::opt_expr(c^2, p, s, noRepSquaring, doNotGenTheseIdents);
                        c:= r[1];
                        p:= r[2];
                        s:= r[3];
                        if domtype(c) = DOM_EXPR then
                            if i < nops(l) or f <> 1 then
                                g:= generate::OptimizeNewVar(doNotGenTheseIdents);
                                p:= append(p, g=c);
                                c:= g;
                            end_if;
                        end_if;
                        if l[i] = 1 then 
                            f:= f*c;
                        end_if
                    end_for;
                    return([f, p, s, TRUE])
                end_if;  // l > 1 and l <= 1000
            end_if;  // DOM_INT...
        end_if;  // not noRepSquaring...
        [e, p, s, c];
    else  // elementary types
        [e, p, s, FALSE];
    end_if;
end_proc:

// end of file 
