/*--
generate::splitProduct  --  decomposes a product into a list
			    containing the sign, the numerator
			    and the denumerator

  generate::splitProduct(multExpr, separateUnits = FALSE)

  multExpr      --  an expression with type "_mult"
  separateUnits --  boolean:  should units and infinities be separated from other factors?

  splitProduct returns a list with the sign of the product
  (-1 or 1) and two lists with the numerator and the denum-
  erator. If the numerator is empty, a 1 is inserted.
  For the decision, if a factor has to be in the numerator 
  or not, exponents of powers are analysed. 
  Numerical factors (which are the last factors in the normal
  form) are splitted (rationals) and put in the first position.
  The result maybe used for output of "_mult" expressions.

  Examples:

>> generate::splitProduct( a / b);

                               [1, [a], [b]]

>> generate::splitProduct( - a / b);

                              [-1, [a], [b]]

>> generate::splitProduct( - a / b^(-x));

                                       x
                             [-1, [a, b ], []]

>> generate::splitProduct( - 2/3 * a / b^(x));

                                             x
                           [-1, [2, a], [3, b ]]


>> generate::splitProduct(9.80665 *unit::m *unit::s^-2, TRUE);

                                                 2
                       [1, [9.80665], [], [m], [s ]]


--*/

generate::splitProduct :=
   //  analyse a _mult-expr and return a list of a 1 number and 2 
   //  lists; the number is the sign of the product 1 or -1, the  
   //  1. list contains the numerator and the 2. the denominator. 
 
   proc(multExpr, separateUnits = FALSE)
     local numList, denumList, sigN, num, i, expo, normalForm, res,
           unitNumList, unitDenumList, base, iType, imPart,
           toNumList   /* what should be inserted in numList? */,
           toDenumList /* what should be inserted in denumList? */,
           insertFront : DOM_BOOL /* insert at the front or the end ? */;
   begin 
     if testargs() then
       if type( multExpr ) <> "_mult" then
         error("_mult-expr expected");
       end_if;
     end_if;

     
 
     normalForm := generate::hasNormalForm(multExpr);
     // if normalForm = FALSE don't change the order of terms
     if nops(multExpr) = 1 then  // _mult(a+b)
       if separateUnits then
         unitNumList := [];
         unitDenumList := [];
       else
         unitNumList := null();
         unitDenumList := null();
       end_if;
       if generate::isNeg(op(multExpr,1)) then
           return([-1, [generate::negate(op(multExpr,1))], [], unitNumList, unitDenumList]);
       else
         return([1, [op(multExpr,1)], [], unitNumList, unitDenumList]);
       end_if;
     end_if;
 
     sigN := 1;
     numList := [];
     denumList := [];
     unitNumList := [];
     unitDenumList := [];
     
     for i in multExpr do
       toNumList := [];  toDenumList := [];
       if normalForm and
         (testtype(i, Type::Real) or contains(Type::ConstantIdents, i)) then
         insertFront := TRUE;
       else
         insertFront := FALSE;
       end;
       
       iType := FAIL;
       traperror((iType := type(i)));

       if separateUnits and (iType = unit or 
         (iType = "_power" and testtype(op(i, 1), unit)) or
         i = infinity or i = hold(infinity)) then
         if iType = "_power" and testtype(op(i, 2), Type::NegInt) then
           unitDenumList := unitDenumList.[1/i];
         else
            unitNumList := unitNumList.[i];
        end_if
       elif iType <> FAIL and
            (testtype(i, Type::Real) or i::dom::isNeg <> FAIL) then
         if generate::isNeg(i) then
           i := generate::negate(i);
           sigN := sigN * (-1);
           if i = 1 then next end_if;
         end_if;
         if iType = DOM_RAT then  // fraction
           if FALSE and
              normalForm and denumList = [] then
             // We are in normal form and there is no denominator yet, then we don't want 
             // to see a fraction only for the denominator of the rational number.
             // Currently switched off by the FALSE in if above.
             
             // REMARK: 7^(2/3)/42*ln(x) should yield
             // (TeX) 7^{2/3}/42 ln(x) not (TeX) 1/42 7^{2/3} ln(x)
             // this has still to be implemented
             toNumList := [i];
           else
             num := op(i,1);
             if num <> 1 then  // ignore 1 ( a/2 -> [[a],[2]] not [[1,a],[2]] )
               toNumList := [num];
             end_if;
             toDenumList := [op(i,2)];
           end_if;
         else
           // real number, but no fraction
           toNumList := [i];
         end_if;
       elif iType = DOM_COMPLEX then  // insert at the beginning
         if iszero(op(i, 1)) then
           imPart := op(i, 2);
           if generate::isNeg(imPart) then
             i := generate::negate(i);
             imPart := -imPart;
             sigN := sigN * (-1);
           end_if;
           toNumList := [I];
           if testtype(imPart, DOM_RAT) then
             // add at the end but any number at front
             if op(imPart, 1) = 1 then
               denumList := [op(imPart, 2)].denumList;
             else
               numList := [op(imPart, 1)].numList;
               denumList := [op(imPart, 2)].denumList;
             end_if
           else
             if imPart <> 1 then
               numList := [imPart].numList;
             end_if
           end_if
         else
           toNumList := [i];
         end_if
       elif iType = "_invert" then
         denumList := denumList.[op(i)];
       elif iType = "_power" or iType = "exp" then
	 		 // no number, so insert at the end 
         // take a look at the exponent
         if  iType = "_power" then
           base := op(i, 1);
           expo := op(i ,2);
         else
           base := exp(1);
           expo := op(i ,1);
         end_if;
         if testtype(expo, "_mult") then
           if testtype(op(expo, nops(expo)), Type::Real) then
             if  op(expo, nops(expo)) < 0 then
               toDenumList := [1/i];
             else
               toNumList := [i];
             end_if; // product contains a real number 
           else
             // exponent is a product with no real number 
             toNumList := [i];
           end_if;
           // exponent is no product
           // if exponent is less zero, so insert 1/i in denumerator
           // but not if base and exponent are numbers
         elif testtype(expo, Type::Real) and
           not testtype(base, Type::Real) then
           if expo = -1 then
             // special case for -1 because of internal normal form
             toDenumList := [base];
           elif expo < 0 then
             toDenumList := [1/i];
           else
             toNumList := [i];
           end_if
         elif generate::isNeg(expo) and
           not testtype(base, Type::Real) then
           toDenumList := [1/i];
         else   // just insert
           toNumList := [i];
         end_if;
       else   // factor is no _power, just insert in numerator 
         toNumList := [i];
       end_if;
       if insertFront then
         numList := toNumList.numList;
         denumList :=toDenumList.denumList;
       else
         numList := numList.toNumList;
         denumList := denumList.toDenumList;
       end_if;
     end_for;
 
     if numList = [] then   // don't return empty numerator 
       numList := [1];
     end_if;

     if map({op(numList)}, l -> l::dom::hasProp(Cat::Matrix)) intersect {TRUE} <> {} and
       denumList <> [] then
       res := split(numList, l -> l::dom::hasProp(Cat::Matrix));
       return([sigN,
               [hold(_mult)(op(res[2]), 1/hold(_mult)(op(denumList))),
                op(res[1])], []] );
     end_if;
     
     if separateUnits then
       return([sigN, numList, denumList, unitNumList, unitDenumList]);
     else
       return([sigN, numList, denumList]);
     end_if;
   end_proc:


/* Utility function:  accepts 1 argument and returns TRUE if it is
  in MuPAD normal form and FALSE otherwise                         */

generate::hasNormalForm :=
proc(e)
  local normalForm, i;
begin
  normalForm := TRUE;
  if testtype(e, "_mult") then
    for i from 1 to nops(e) - 1 do
      if not normalForm then break; end;
      if traperror(type(op(e, i))) <> 0 then normalForm := FALSE; break; end;
      
      case type(op(e, i))
        of DOM_INT do
        of DOM_FLOAT do
        of DOM_INTERVAL do
        of DOM_RAT do
        of DOM_COMPLEX do
        of "_mult" do
          normalForm := FALSE;
          break;
          
        of "_power" do
          if not generate::hasNormalForm(op(e,i)) then
            normalForm := FALSE;
          end;
          break;
      end_case;
    end_for
  elif testtype(e, "_power") and
    testtype(op(e, 1), Type::Numeric) and
    testtype(op(e, 2), DOM_INT) then
    normalForm := FALSE;
    break;
  end;
  normalForm
end_proc:


