/*- 
 addEdges - adds one or several edges to a Graph
 addEdges(G, Edge)
 addEdges(G, Edge <,EdgeWeights=t> <,EdgeCosts=c> <,EdgeDescriptions=d> <,OnlySpecifiedEdges>)
 addEdges(G, [e1,..,en])
 addEdges(G, [e1,..,en] <,EdgeWeights=[t1,..,tn]> <,EdgeCosts=[c1,..,cn]> <,EdgeDescriptions=d>)
 Parameters:
   G               - Graph
   Edge            - a list of lists of one or more edges
   EdgeWeights    - a list of one or more edge weights
   EdgeCosts        - a list of one or more edge costs
   EdgeDescriptions - a list of one or more descriptions. 
                     If an empty string ("") is used, no table entry will be created for this edge.
                     Also any other object than DOM_STRING will be converted into one using expr2text.
-*/
Graph::addEdges := proc(G : Graph, Edge : Type::ListOf(DOM_LIST))
local _correctEdges, _edgeCosts, _edgeWeights, _edgesEntering, _edgesLeaving, _edgeDescriptions, _onlySE,
      arguments, leftOperator, rightOperator, i, j, _vertices, _edges, _directed, x, position, tmpList, edge;
begin
   _onlySE := FALSE;
   arguments := args();
   j := args(0);
   
   for i from 3 to j do
     if op(arguments[i],1) = hold(OnlySpecifiedEdges) then
       _onlySE := TRUE;
       delete arguments[i];
       j := j - 1;
       break;
     end_if;
   end_for;

   _directed := Graph::isDirected(G);
   _correctEdges := Graph::selectEdge(G, Edge, FALSE, hold(OnlySpecifiedEdges)=_onlySE);       

   // If graph is undirected put also reverted edges to it.
   if not _directed and not _onlySE then
     _correctEdges := _correctEdges . map(_correctEdges, revert);
   end_if;

   _vertices := Graph::getVertices(G);
   _edgeCosts := Graph::getEdgeCosts(G);
   _edgeDescriptions := Graph::getEdgeDescriptions(G);
   _edgeWeights := Graph::getEdgeWeights(G);
   _edgesLeaving := Graph::getEdgesLeaving(G);
   _edgesEntering := Graph::getEdgesEntering(G);

   // Check if vertices exist !!!
    tmpList := Graph::checkForVertices(_correctEdges, Graph::getVertices(G));
    if not tmpList = [] then
      error("One or more edges contain vertices that are not in list : " . expr2text(tmpList) . "\n" );
    end_if;

   // first parameter (Edge) is mandatory, whereas all the rest is optional
   for i from 3 to j do
      if not (type(arguments[i]) ="_equal") then
        error("Wrong argument type: equation expected!");
      end_if;
      // Store the description in leftOperator
      leftOperator := op(arguments[i],1); 
      // Store the edges in rightOperator
      rightOperator := op(arguments[i],2);
      
      if domtype(rightOperator) <> DOM_LIST then
        error("Wrong argument type for " . expr2text(leftOperator) . ": list expected!");
      end_if;
      if nops(rightOperator) < nops(Edge) then
        error("Number of " . expr2text(leftOperator) . " is smaller than number of edges.");
      elif nops(rightOperator) > nops(Edge) then
        error("Number of edges is smaller than number of " . expr2text(leftOperator) . "\n");
      end_if;

      case leftOperator
        of hold(EdgeWeights) do
          for x in _correctEdges do
            position := contains(Edge,x);  // 0 if edge 'x' not in Graph
            if position > 0 then
              _edgeWeights[x] := rightOperator[position];
            end_if;
          end_for;
          if _directed = FALSE then
            for x in _correctEdges do
              position := contains(Edge,x); // 0 if edge 'x' not in Graph
              if position > 0 then
                _edgeWeights[revert(x)] := rightOperator[position];
              end_if;
            end_for;
          end_if;
          _edgeWeights := select(_edgeWeights, x-> not op(x,2)=hold(None));
          if nops(_edgeWeights) = 0 then
            _edgeWeights := FAIL;
          end_if;
          break;
        of hold(EdgeCosts) do
          for x in _correctEdges do
            position := contains(Edge,x); // 0 if edge 'x' not in Graph
            if position > 0 then
              _edgeCosts[x] := rightOperator[position];
            end_if;
          end_for;
          if _directed = FALSE then
            for x in _correctEdges do
              position := contains(Edge,x); // 0 if edge 'x' not in Graph
              if position > 0 then
                _edgeCosts[revert(x)] := rightOperator[position];
              end_if;
            end_for;
          end_if;
          _edgeCosts := select(_edgeCosts, x-> not op(x,2)=hold(None));
          if nops(_edgeCosts) = 0 then
            _edgeCosts := FAIL;
          end_if;
          break;
        of hold(EdgeDescriptions) do
          for x in _correctEdges do
            position := contains(Edge,x); // 0 if edge 'x' not in Graph
            if position > 0 then
              case type(rightOperator[position])
              of DOM_STRING do
                if rightOperator[position] <> "" then
                  _edgeDescriptions[x] := rightOperator[position];                      
                  if _directed = FALSE then
                    _edgeDescriptions[revert(x)] := rightOperator[position];
                  end_if;
                end_if;
                break;
              otherwise
                _edgeDescriptions[x] := expr2text(rightOperator[position]);
                if _directed = FALSE then
                  _edgeDescriptions[revert(x)] := expr2text(rightOperator[position]);
                end_if;
                break;
              end_case;
            end_if;
          end_for;
          _edgeDescriptions := select(_edgeDescriptions, x-> not op(x,2)="None");
          if nops(_edgeDescriptions) = 0 then
            _edgeDescriptions := FAIL;
          end_if;
          break;
        otherwise
          error("Wrong argument type: " . expr2text(leftOperator) . ". Please consult help for more information.");
      end_case;
    end_for;
    
   /*-
    Generate the adjacency lists
   -*/

   _edges := _correctEdges;
   _edges := append(_edges, op(Graph::getEdges(G)));  

    _edgesEntering := table(_vertices[i] = table() $ i=1..nops(_vertices));
    _edgesLeaving  := table(_vertices[i] = table() $ i=1..nops(_vertices));
    for edge in _edges do
      _edgesEntering[edge[2]][edge] := edge[1];
      _edgesLeaving[edge[1]][edge] := edge[2];
    end_for;
    _edgesEntering := map(_edgesEntering,
			  x -> sort(map([op(x)], op, 2)));
    _edgesLeaving :=  map(_edgesLeaving,
			  x -> sort(map([op(x)], op, 2)));

   subsop(G, 2=_edges, 4=_edgeDescriptions, 5=_edgeWeights, 6=_edgeCosts, 7=_edgesEntering, 8=_edgesLeaving);
end_proc:

// End of file
null():

