/*
Graph::chromaticPolynomial(G, x)

Parameters:
G - Graph
x - Identifier

computes the chromatic polynomial of the graph G; 
x is used as the indeterminate of the polynomial.
G must be undirected, i._edge., for every edge [a, b], the edge [b, a] must
exist, too.
*/

Graph::chromaticPolynomial:= proc(G: Graph, x: DOM_IDENT): DOM_POLY
  local
  equ: "_equal", 
  delEdge: DOM_PROC,
  joinVertices: DOM_PROC,
  crpoly: DOM_PROC;
begin

  if Graph::isDirected(G) then
    error("Graph must be undirected!");
  end_if;

  // local method delVertex
  //
  // T - adjacency table
  // a - vertex
  // b - vertex
  //
  // returns a new graph T, with the edge between a and b removed
  
  delEdge:= proc(T: DOM_TABLE, a, b): DOM_TABLE
  begin
    T[a]:= subs(T[a], b = null());
    T[b]:= subs(T[b], a = null());
    T
  end_proc;

  // local method joinVertices
  //
  // T - adjacency table
  // a - vertex
  // b - vertex
  //
  // returns the equivalence graph, with a and b now being identified
  // the edge between a and b is *not* turned into a loop
  // a is used as name for the vertex ab
  
  joinVertices:= proc(T: DOM_TABLE, a, b)
  begin
    // join adjacency lists, remove duplicates
    T[a]:= [op({op(T[a].T[b])} minus {a, b})];
    delete T[b];
    subs(T, b=a);
  end_proc;
  
  // local method crpoly
  // 
  // crpoly(T)
  //
  // T - adjacency table

  //
  // returns chromatic polynomial of the graph reprsented
  // by T, an element of Z[x], where x is the arg. of the outer procedure

  crpoly :=  proc(T: DOM_TABLE): DOM_POLY
    local equ, a, b;
  begin
    if nops(T) = 1 then
      return(poly(x))
    end_if;

    equ:= op(T, 1);
    // now equ is an equation vertex = [adjacent vertices]
    a:= op(equ, 1);
    
    if op(equ, 2) = [] then
      // isolated vertex: remove
      delete T[a];
      poly(x) * crpoly(T)
    else
      // simply continue with first edge
      b:= op(equ, [2, 1]);
      crpoly(delEdge(T, a, b)) - crpoly(joinVertices(T, a, b))
    end_if;
      
  end_proc;

  for equ in Graph::getEdgesLeaving(G) do
    if contains(op(equ,2), op(equ, 1)) > 0 then
      // the graph G contains a loop and cannot be coloured
      return(poly(0, [x]))
    end_if
  end_for;
  
  crpoly(Graph::getEdgesLeaving(G))
  
end_proc:

// End of file
null():
