/*-
 Graph::convertSSQ - converts a Graph to a single source, single sink Graph
 convertSSQ(G,s,t)
 Parameters:
   G - Graph
   s - source (new node)
   t - sink (new node)
 Converts a Graph to a single source, single sink Graph. The given nodes s and t 
 are inserted in the Graph. The are connected to the other nodes in the following way:
   [s,i] for every i with Graph::getVertexWeight[i] > 0
   [i,t] for every i with Graph::getVertexWeight[i] < 0
   _edgeWeights[[s,i]] := getEdgeWeight[i]   getEdgeWeight[[s,i]] := 0
   _edgeWeights[[i,t]] := getEdgeWeight[i]   getEdgeWeight[[i,t]] := 0
-*/
Graph::convertSSQ := proc(G : Graph,s : DOM_LIST,t : DOM_LIST)
local _vertices,_vertexWeights,_edges,_edgeWeights,_edgeCosts,f,Q,S,i,new_Eds,new_Edt, G2;
begin
  _vertices := Graph::getVertices(G);
  if nops(s) > 1 then
    error("Only one source node allowed!");
  end_if;
  if nops(t) > 1 then
    error("Only one sink node allowed!");
  end_if;
  if contains(_vertices, op(s)) <> 0 or contains(_vertices, op(t)) <> 0 then
    error("Specified nodes are already nodes of the Graph!");
  end_if;

  // Shortcut if the graph has no edges at all.
  if Graph::getEdgeNumber(G) = 0 then
    _vertices := _vertices.[s,t];
  	 return (Graph::new(_vertices, [[]], hold(Directed)));
  end_if;

  _vertexWeights := Graph::getVertexWeights(G);
  _edges := Graph::getEdges(G);
  _edgeWeights := Graph::getEdgeWeights(G);
  _edgeCosts := Graph::getEdgeCosts(G);
  s := op(s);
  t := op(t);

  if _vertexWeights = FAIL then
    for i in _vertices do
      _vertexWeights[i] := 0;
    end_for;
  end_if;

  // Set to 1 because of predefinition
  if _edgeWeights = FAIL then
    for i in _edges do
      _edgeWeights[i] := 1;
    end_for;
  end_if;

  if _edgeCosts = FAIL then
    for i in _edges do
      _edgeCosts[i] := 0;
    end_for;
  end_if;

  f := proc() begin bool(_vertexWeights[args(1)] > 0) end_proc ;
  Q := select(_vertices, f);
  f := proc() begin bool(_vertexWeights[args(1)] < 0) end_proc ;
  S := select(_vertices, f);

  new_Eds := [[s,Q[i]] $ i=1..nops(Q)];
  new_Edt := [[S[i],t] $ i=1..nops(S)];
  _vertexWeights[s] := 0;
  _vertexWeights[t] := 0;
  for i in Q do 
    _edgeCosts[[s,i]] := 0;
    _edgeWeights[[s,i]] := _vertexWeights[i];
    _vertexWeights[s] := _vertexWeights[s] + _vertexWeights[i];
    _vertexWeights[i] := 0;
  end_for;
  for i in S do 
    _edgeCosts[[i,t]] := 0;
    _edgeWeights[[i,t]] := -_vertexWeights[i];
    _vertexWeights[t] := _vertexWeights[t] + _vertexWeights[i];
    _vertexWeights[i] := 0;
  end_for;
  _edges := _edges.new_Eds.new_Edt;
  _vertices := _vertices.[s,t];

  G2 := Graph(_vertices, _edges, Directed);
  subsop(G2, 3 = _vertexWeights, 5 = _edgeWeights, 6 = _edgeCosts);
end_proc:

// End of file
null():
