/*-
 plotBipartiteGraph - graphical plot where vertices are arranged left and right of a box and are connected respectively

 plotBipartiteGraph(G)
 plotBipartiteGraph(G <,PointSize=n> <,SpecialEdges=[e1,..,en]> <, EdgeColor=RGB> 
                      <, SpecialEdgeColor=RGB> <, VertexColor=RGB>, <Vertex1Color=RGB>, <Vertex2Color=RGB>)
 Parameters:
   G                - Graph
   PointSize       - n is a positive integer. Default = 1.5
   SpecialEdges     - a list of edges that have to be drawn in blue or the given color respectively.
   EdgeColor        - edges will be drawn in this color. Default = RGB::Red
   SpecialEdgeColor - If SpecialEdges is set then the selected edges will be drawn in this color. Default = RGB::Red
   VertexColor      - The color in which the vertices are drawn. Default = RGB::Red
   Vertex1Color     - The color in which the first vertex is drawn. Default = RGB::Blue
   Vertex2Color     - The color in which the second vertex is drawn. Default = RGB::Green
-*/
Graph::plotBipartiteGraph :=
proc(G : Graph)
  local i, pointList, edgeList, _edges, edge, numberEdges, v, w,
      _isDirected, _pointWidth, sets, setOne, setTwo, _vertices, leftOperator, rightOperator,
      _edgeColor, _specialEdgeColor, _vertexColor, _v1Color, _v2Color, _specialEdgeList, drawColor, 
      colorOpt, drawEdge, specialDrawColor, specialEdgeList;

begin
  _vertices := Graph::getVertices(G);
  _edges := Graph::getEdges(G);
  _isDirected := Graph::isDirected(G); // otherwise the function would have to be called n-times during the for loop.
  sets := Graph::bipartite(G, hold(Lists)); // To get either FAIL or the two sets in a list.
  if sets = FAIL then
    error("Graph is not bipartite!");
  else
    setOne := op(sets, 1);
    setTwo := op(sets, 2);
  end_if;
  assert(nops(_vertices) > 0); // expected to be checked by Graph::bipartite

  _pointWidth := 1.5;
  _specialEdgeList := [];
  _edgeColor := RGB::Red;
  _specialEdgeColor := RGB::Blue;
  _vertexColor := RGB::Red;
  _v1Color := RGB::Blue;
  _v2Color := RGB::Green;

  for i from 2 to args(0) do
    leftOperator := op(args(i),1);
    rightOperator := op(args(i),2);

    case leftOperator 
      of hold(PointSize) do
        if not testtype(rightOperator, Type::NonNegInt) then
          error("Wrong argument type for PointSize: positive integer expected!");
        end_if;
        _pointWidth := rightOperator;
        break;
      of hold(SpecialEdges) do
        if not testtype(rightOperator, Type::ListOf(DOM_LIST)) then
          error("Wrong argument type for SpecialEdges: List of edges expected!");
        end_if;
        _specialEdgeList := rightOperator;
        // make sure the edges are correct for undirected case.
        if not _isDirected then
          for v from 1 to nops(_specialEdgeList) do
            _specialEdgeList[v] := sort(_specialEdgeList[v]);
          end_for;
          _specialEdgeList:= [op({op(_specialEdgeList)})];
        end_if;
        break;
      of hold(EdgeColor) do
        if not testtype(rightOperator, Type::ListOf(Type::Real, 3, 3)) then
          error("Wrong argument type for EdgeColor: Color expected!");
        end_if;
        _edgeColor := rightOperator;
        break;
      of hold(SpecialEdgeColor) do
        if not testtype(rightOperator, Type::ListOf(Type::Real, 3, 3)) then
          error("Wrong argument type for SpecialEdgeColor: Color expected!");
        end_if;
        _specialEdgeColor := rightOperator;
        break;
      of hold(VertexColor) do
        if not testtype(rightOperator, Type::ListOf(Type::Real, 3, 3)) then
          error("Wrong argument type for VertexColor: Color expected!");
        end_if;
        _vertexColor := rightOperator;
        break;
      of hold(Vertex1Color) do
        if not testtype(rightOperator, Type::ListOf(Type::Real, 3, 3)) then
          error("Wrong argument type for Vertex1Color: Color expected!");
        end_if;
        _v1Color := rightOperator;
        break;
      of hold(Vertex2Color) do
        if not testtype(rightOperator, Type::ListOf(Type::Real, 3, 3)) then
          error("Wrong argument type for Vertex2Color: Color expected!");
        end_if;
        _v2Color := rightOperator;
        break;
      otherwise
        error("Wrong argument: " . expr2text(leftOperator));
    end_case;

  end_for;

  // Sorts out multiple edges (saves time during ploting !!!)
  if not _isDirected then
    for i from 1 to nops(_edges) do
      _edges[i] := sort(_edges[i]);
    end_for;
    _edges:= [op({op(_edges)})];
  end_if;

  numberEdges := nops(_edges);
  pointList := [];

  // Set 1
  if nops(setOne) > 0 then
    pointList := append(pointList, [0, 0, _v1Color]);
  end_if;
  if nops(setOne) > 1 then
    pointList := append(pointList, [0, (i/10), _vertexColor] $ i=1..(nops(setOne)-1));
  end_if;

  // Set 2
  if nops(setTwo) > 0 then
    pointList := append(pointList, [10, 0, _v2Color]);   
  end_if;
  if nops(setTwo) > 1 then
    pointList := append(pointList, [10, (i/10), _vertexColor] $ i=1..(nops(setTwo)-1));
  end_if;

  specialDrawColor := {};
  drawColor := {};
  edgeList := [];
  specialEdgeList := [];

  for i from 1 to numberEdges do
    edge := _edges[i];
    v := contains(setOne, edge[1]);
    if v = 0 then
      // For directed graphs the direction is important !
      v := contains(setTwo, edge[1]) + nops(setOne);
      w := contains(setOne, edge[2]);
      if w = 0 then
        error("Graph is not bipartite!");
      end_if;
    else
      w := contains(setTwo, edge[2]);
      if w = 0 then
        error("Graph is not bipartite!");
      end_if;
      w := w + nops(setOne);
    end_if;

    if _isDirected then
      drawEdge := plot::Arrow2d([op(pointList[v],1..2)], [op(pointList[w],1..2)]);
      colorOpt := plot::Arrow2d::LineColor;
    else
      drawEdge := plot::Line2d([op(pointList[v],1..2)], [op(pointList[w],1..2)]);
      colorOpt := plot::Line2d::LineColor;
    end_if;
    if (contains(_specialEdgeList, edge) > 0) then
      specialEdgeList := specialEdgeList.[drawEdge];
      specialDrawColor := specialDrawColor union {colorOpt = _specialEdgeColor};
    else
      edgeList := edgeList.[drawEdge];
      drawColor := drawColor union {colorOpt = _edgeColor};
    end_if;
  end_for;

  plot::Group2d(
       plot::Group2d(op(specialEdgeList),  op(specialDrawColor)),
       plot::Group2d(op(edgeList),  op(drawColor)),
       plot::PointList2d(pointList),
       Axes = None);
end_proc:

// End of file
null():
