/*-
 plotCircleGraph - graphical plot where vertices are arranged in a circle of a Graph
                   The first point is drawn in blue and the second in green (if not redefined) so the direction can be found out easily.
                   The vertices can be assigned easily too if after the plot(%) command getVertices Information is invoked.

 plotCircleGraph(G <, PointSize=n> <, SpecialEdges=[e1,..,en]> <, EdgeColor=RGB>  <, SpecialEdgeColor=RGB> 
                   <, VertexColor=RGB> <, Vertex1Color=RGB> <, Vertex2Color=RGB>)
 Parameters:
   G                - Graph
   PointSize       - n is a positive integer. Default = 1.5
   SpecialEdges     - a list of edges that have to be drawn in blue or the given color respectively.
   EdgeColor        - edges will be drawn in this color. Default = RGB::Red
   SpecialEdgeColor - If SpecialEdges is set then the selected edges will be drawn in this color. Default = RGB::Red
   VertexColor      - The color in which the vertices are drawn. Default = RGB::Red
   Vertex1Color     - The color in which the first vertex is drawn. Default = RGB::Blue
   Vertex2Color     - The color in which the second vertex is drawn. Default = RGB::Green
-*/
Graph::plotCircleGraph :=
proc(G : Graph)
  local alpha, i, pointList, edgeList, _vertices, _edges, edge, numberEdges,
      v, w, numberVertices, _isDirected, _pointWidth, tmpList, _specialEdgeList, _edgeColor, _specialEdgeColor, 
      leftOperator, rightOperator, _vertexColor, _v1Color, _v2Color, 
      drawColor, colorOpt, drawEdge, specialDrawColor, specialEdgeList, _arguments;

begin
  _pointWidth := 1.5;
  _edgeColor := RGB::Red;
  _specialEdgeColor := RGB::Blue;
  _vertexColor := RGB::Red;
  _v1Color := RGB::Blue;
  _v2Color := RGB::Green;

  _specialEdgeList := [];
  _vertices := Graph::getVertices(G);
  _edges := Graph::getEdges(G);
  _isDirected := Graph::isDirected(G); // otherwise the function would have to be called n-times during the for loop.
  _arguments := [args()];

  // First check for a vertex order...
  for i from 2 to nops(_arguments) do
    leftOperator := op(_arguments[i],1);
    rightOperator := op(_arguments[i],2);

    if leftOperator = hold(VertexOrder) then
      tmpList := {op(rightOperator)} minus {hold(None)};
      if nops(_vertices) <> nops(tmpList) then
        error("Number of defined vertices does not equal number of vertices in graph");
      end_if;
      _vertices := rightOperator;
      delete _arguments[i];
      break;
    end_if;
  end_for;

  for i from 2 to nops(_arguments) do

    leftOperator := op(_arguments[i],1);
    rightOperator := op(_arguments[i],2);

    case leftOperator 
      of hold(PointSize) do
        if not testtype(rightOperator, Type::NonNegInt) then
          error("Wrong argument type for PointSize: positive integer expected!");
        end_if;
        _pointWidth := rightOperator;
        break;
      of hold(SpecialEdges) do
        if not testtype(rightOperator, Type::ListOf(DOM_LIST)) then
          error("Wrong argument type for SpecialEdges: List of edges expected!");
        end_if;
        _specialEdgeList := rightOperator;
        // make sure the edges are correct for undirected case.
        if not _isDirected then
          for v from 1 to nops(_specialEdgeList) do
            _specialEdgeList[v] := sort(_specialEdgeList[v]);
          end_for;
          _specialEdgeList:= [op({op(_specialEdgeList)})];
        end_if;
        break;
      of hold(EdgeColor) do
        if not testtype(rightOperator, Type::ListOf(Type::Real, 3, 3)) then
          error("Wrong argument type for EdgeColor: Color expected!");
        end_if;
        _edgeColor := rightOperator;
        break;
      of hold(SpecialEdgeColor) do
        if not testtype(rightOperator, Type::ListOf(Type::Real, 3, 3)) then
          error("Wrong argument type for SpecialEdgeColor: Color expected!");
        end_if;
        _specialEdgeColor := rightOperator;
        break;
      of hold(VertexColor) do
        if not testtype(rightOperator, Type::ListOf(Type::Real, 3, 3)) then
          error("Wrong argument type for VertexColor: Color expected!");
        end_if;
        _vertexColor := rightOperator;
        break;
      of hold(Vertex1Color) do
        if not testtype(rightOperator, Type::ListOf(Type::Real, 3, 3)) then
          error("Wrong argument type for Vertex1Color: Color expected!");
        end_if;
        _v1Color := rightOperator;
        break;
      of hold(Vertex2Color) do
        if not testtype(rightOperator, Type::ListOf(Type::Real, 3, 3)) then
          error("Wrong argument type for Vertex2Color: Color expected!");
        end_if;
        _v2Color := rightOperator;
        break;
      otherwise
        error("Wrong argument: " . expr2text(leftOperator));
    end_case;

  end_for;

  if nops(_vertices) < 1 then
    error("Graph must contain at least one vertex!");
  end_if;

  // Sorts out multiple edges (saves time during ploting !!!)
  if not _isDirected then
    for i from 1 to nops(_edges) do
      _edges[i] := sort(_edges[i]);
    end_for;
    _edges:= [op({op(_edges)})];
  end_if;

  numberVertices := nops(_vertices);
  numberEdges := nops(_edges);
  alpha := float(2*PI / (numberVertices));
  // Starting point
  pointList := [[cos(0), sin(0), _v1Color]];
  if nops(_vertices) > 1 then
    pointList := append(pointList, [cos(alpha), sin(alpha), _v2Color]);
  end_if;
  if nops(_vertices) > 2 then
    pointList := append(pointList,
                        [cos(i*alpha), sin(i*alpha), _vertexColor] $
                        i=2..(numberVertices-1));
  end_if;

  specialDrawColor := {};
  drawColor := {};
  edgeList := [];
  specialEdgeList := [];

  for i from 1 to numberEdges do
    edge := _edges[i];
    v := contains(_vertices, edge[1]); 
    w := contains(_vertices, edge[2]);
      
    if v = w then
      drawEdge := plot::Ellipse2d(0.05, 0.05,
                                  map(pointList[v][1..2],
                                      _mult, 1.05));
      colorOpt := plot::Ellipse2d::LineColor;
    else
      if _isDirected then
        drawEdge := plot::Arrow2d([op(pointList[v],1..2)],
                                         [op(pointList[w],1..2)]);
        colorOpt := plot::Arrow2d::LineColor;
      else
        drawEdge := plot::Line2d([op(pointList[v],1..2)],
                                 [op(pointList[w],1..2)]);
        colorOpt := plot::Line2d::LineColor;
      end_if;
    end_if;
    if (contains(_specialEdgeList, edge) > 0) then
      specialEdgeList := specialEdgeList.[drawEdge];
      specialDrawColor := specialDrawColor union {colorOpt = _specialEdgeColor}
    else
      edgeList := edgeList.[drawEdge];
      drawColor := drawColor union {colorOpt = _edgeColor}
    end_if;
  end_for;

  plot::Group2d(
       plot::Group2d(op(specialEdgeList),  op(specialDrawColor)),
       plot::Group2d(op(edgeList),  op(drawColor)),
       plot::PointList2d(pointList),
       Axes = None)

end_proc:

// End of file
null():
