/*-
 plotGridGraph - graphical plot where vertices are arranged left and right of a box and are connected respectively

 plotGridGraph(G)
 plotGridGraph(G <, PointSize=n> <, SpecialEdges=[e1,..,en]> <, EdgeColor=RGB> <, VerticesPerLine=n|[n1..nm]> 
                 <, VertexOrder=[v1,..vn]> <, SpecialEdgeColor=RGB> <, SpecialVertices=[e1,..en]> <, SpecialVertexColor=RGB>
                 <, VertexColor=RGB> <, Vertex1Color=RGB> <, Vertex2Color=RGB>)
 Parameters:
   G                - Graph
   VerticesPerLine  - n is a positive integer. Default is the ceilling of the sqrt ot the number of vertices.
                      OR n1..nm define how many vertices are to be placed per line.
   VertexOrder      - defines in which order the vertices are to be arranged. None is defined as filler for 
                      vertices that should not be drawn.
   PointSize       - n is a positive integer. Default = 1.5
   SpecialEdges     - a list of edges that have to be drawn in blue or the given color respectively.
   SpecialVertices  - a list of vertices that have to be drawn in blue or the given color respectively.
   EdgeColor        - edges will be drawn in this color. Default = RGB::Red
   SpecialEdgeColor - If SpecialEdges is set then the selected edges will be drawn in this color. Default = RGB::Red
   VertexColor      - The color in which the vertices are drawn. Default = RGB::Red
   Vertex1Color     - The color in which the first vertex is drawn. Default = RGB::Blue
   Vertex2Color     - The color in which the second vertex is drawn. Default = RGB::Green
-*/
Graph::plotGridGraph :=
proc(G : Graph)
  local i, pointList, edgeList, specialEdgeList, _edges, edge, numberEdges, v, w, _vpl, rowCounter, _vplList,
      _isDirected, _pointWidth, _vertices, leftOperator, rightOperator, maxWidth, j, _height, tmpList, _arguments,
      _edgeColor, _specialEdgeColor, _specialVertexColor, _vertexColor, _v1Color, _v2Color, _specialEdgeList, _specialVertexList, 
      drawColor, alpha, colorOpt, drawEdge, specialDrawColor;
begin
  _vertices := Graph::getVertices(G);
  _edges := Graph::getEdges(G);
  _isDirected := Graph::isDirected(G); // otherwise the function would have to be called n-times during the for loop.

  _vpl := FAIL;
  _pointWidth := 1.5;
  _specialEdgeList := [];
  _specialVertexList := [];
  _edgeColor := RGB::Red;
  _specialEdgeColor := RGB::Blue;
  _specialVertexColor := RGB::Blue;
  _vertexColor := RGB::Red;
  _v1Color := RGB::Blue;
  _v2Color := RGB::Green;
  maxWidth := 0;
  rowCounter := 0;
  _arguments := [args()];     

  // Two parameters have to be checked first and then to be removed from the list.
  for i from 2 to nops(_arguments) do
    leftOperator := op(_arguments[i],1);
    rightOperator := op(_arguments[i],2);

    if leftOperator = hold(VertexOrder) then
      tmpList := {op(rightOperator)} minus {hold(None)};
      if nops(_vertices) <> nops(tmpList) then
        error("Number of defined vertices does not equal number of vertices in graph");
      end_if;
      _vertices := rightOperator;
      delete _arguments[i];
      break;
    end_if;
  end_for;

  for i from 2 to nops(_arguments) do
    leftOperator := op(_arguments[i],1);
    rightOperator := op(_arguments[i],2);
    if leftOperator = hold(VertexColor) then
      if not testtype(rightOperator, Type::ListOf(Type::Real, 3, 3)) then
        error("Wrong argument type for VertexColor: Color expected!");
      end_if;
      _vertexColor := rightOperator;
      _v1Color := rightOperator;
      _v2Color := rightOperator;
      delete _arguments[i];
      break;
    end_if;
  end_for;

  // Now the "usual" checking can be done.
  for i from 2 to nops(_arguments) do
    leftOperator := op(_arguments[i],1);
    rightOperator := op(_arguments[i],2);

    case leftOperator 
      of hold(VerticesPerLine) do
        if not domtype(rightOperator) = DOM_LIST and not testtype(rightOperator, Type::NonNegInt) then
          error("Wrong argument type for VerticesPerLine: List or positive integer expected!");
        end_if;
        _vpl := rightOperator;
        if domtype(rightOperator) = DOM_LIST then
          _vplList := rightOperator;
          for j from 1 to nops(_vplList) do
            rowCounter := rowCounter + op(_vplList,j);
            if maxWidth < op(_vplList,j) then
              maxWidth := op(_vplList,j);
            end_if;
          end_for;
          if rowCounter <> nops(_vertices) then
            error("Number of vertices in list VerticesPerLine (" . expr2text(rowCounter) . ") must equal number of vertices (" . expr2text(nops(_vertices)) . ")!");
          end_if;
        else
          if _vpl < 1 then
            error("VerticesPerLine must be greater than 0!");
          end_if;
          _vplList := [];
          maxWidth := _vpl;
          for j from 1 to floor(nops(_vertices) / _vpl) do
            _vplList := append(_vplList, _vpl);
          end_for;
          _vplList := append(_vplList, (nops(_vertices) - (_vpl*j)) );
        end_if;
        break;
      of hold(PointSize) do
        if not testtype(rightOperator, Type::NonNegInt) then
          error("Wrong argument type for PointSize: positive integer expected!");
        end_if;
        _pointWidth := rightOperator;
        break;
      of hold(SpecialVertices) do
        if not domtype(rightOperator) = DOM_LIST then
          error("Wrong argument type for SpecialVertices: List expected!");
        end_if;
        _specialVertexList := rightOperator;
        break;
      of hold(SpecialEdges) do
        if not testtype(rightOperator, Type::ListOf(DOM_LIST)) then
          error("Wrong argument type for SpecialEdges: List of edges expected!");
        end_if;
        _specialEdgeList := rightOperator;
        // make sure the edges are correct for undirected case.
        if not _isDirected then
          for v from 1 to nops(_specialEdgeList) do
            _specialEdgeList[v] := sort(_specialEdgeList[v]);
          end_for;
          _specialEdgeList:= [op({op(_specialEdgeList)})];
        end_if;
        break;
      of hold(EdgeColor) do
        if not testtype(rightOperator, Type::ListOf(Type::Real, 3, 3)) then
          error("Wrong argument type for EdgeColor: Color expected!");
        end_if;
        _edgeColor := rightOperator;
        break;
      of hold(SpecialEdgeColor) do
        if not testtype(rightOperator, Type::ListOf(Type::Real, 3, 3)) then
          error("Wrong argument type for SpecialEdgeColor: Color expected!");
        end_if;
        _specialEdgeColor := rightOperator;
        break;
      of hold(SpecialVertexColor) do
        if not testtype(rightOperator, Type::ListOf(Type::Real, 3, 3)) then
          error("Wrong argument type for SpecialVertexColor: Color expected!");
        end_if;
        _specialVertexColor := rightOperator;
        break;
      of hold(Vertex1Color) do
        if not testtype(rightOperator, Type::ListOf(Type::Real, 3, 3)) then
          error("Wrong argument type for Vertex1Color: Color expected!");
        end_if;
        _v1Color := rightOperator;
        break;
      of hold(Vertex2Color) do
        if not testtype(rightOperator, Type::ListOf(Type::Real, 3, 3)) then
          error("Wrong argument type for Vertex2Color: Color expected!");
        end_if;
        _v2Color := rightOperator;
        break;
      otherwise
        error("Wrong argument: " . expr2text(leftOperator));
    end_case;
  end_for;

  // No parameter for vpl was given...
  if _vpl = FAIL then
    _vpl := ceil(sqrt(nops(_vertices)));
    _vplList := [];
    maxWidth := _vpl;
    for i from 1 to _vpl-1 do
      _vplList := append(_vplList, _vpl);
    end_for;
    if nops(_vertices) - ((_vpl-1) * _vpl) > 0 then
      _vplList := append(_vplList, ( nops(_vertices) - ((_vpl-1) * _vpl) ) );
    end_if;
  end_if;

  if nops(_vertices) < 1 then
    error("Graph must contain at least one vertex!");
  end_if;

  // Sorts out multiple edges (saves time during ploting !!!)
  if not _isDirected then
    for i from 1 to nops(_edges) do
      _edges[i] := sort(_edges[i]);
    end_for;
    _edges:= [op({op(_edges)})];
    // JUST A TEST !!!!
    _edges := sort(_edges);
  end_if;

  if nops(_specialEdgeList) > 0  then
    // Sort out edges NOT existing in the graph
    numberEdges := Graph::selectEdge(G, _specialEdgeList, FALSE, Show=FALSE);
    if nops(numberEdges) > 0 then
      warning("The following edges in SpecialEdges do not exist in Graph: " . expr2text(numberEdges) . "\n");
    end_if;
  end_if;

  if nops(_specialVertexList) > 0 then
    numberEdges := Graph::selectVertex(G, _specialVertexList, FALSE, Show=FALSE);
    if nops(numberEdges) > 0 then
      warning("The following vertices in SpecialVertices do not exist in Graph: " . expr2text(numberEdges) . "\n");
    end_if;   
  end_if;

  numberEdges := nops(_edges);
  pointList := [];
  rowCounter := 0; //ceil(nops(_vertices) / _vpl);
  _height := nops(_vplList);

  // width between vertices = maxWidth / (VerticesPerLine - 2) + 1
  j := 0;
  _vpl := _vplList[rowCounter + 1];
  if _vpl = 1 then
    j := 1;
    alpha := maxWidth/2;
  else
    alpha := maxWidth / (_vpl - 1);
  end_if;
  if not _vertices[1] = hold(None) then
    if contains(_specialVertexList, _vertices[1]) > 0 then
      pointList := append(pointList,
                          [j * alpha, _height - rowCounter, _specialVertexColor]);
    else
      pointList := append(pointList,
                          [j * alpha, _height - rowCounter, _v1Color]);   
    end_if;
  end_if;

  // More than a single point ?
  j := j + 1;
  if nops(_vertices) > 1 then
    if _vpl = 1 then
      rowCounter := rowCounter + 1;
      j := 0;  // because here it is reduced to 0 and therefore the first point goes to 0,y
      _vpl := _vplList[rowCounter + 1];
      if _vpl = 1 then
        j := 1;
        alpha := maxWidth / 2;
      else
        alpha := maxWidth / (_vpl - 1);
      end_if;
    end_if;
    if not _vertices[2] = hold(None) then
      if contains(_specialVertexList, _vertices[2]) > 0 then
        pointList := append(pointList,
                            [j * alpha, _height - rowCounter, _specialVertexColor]);
      else
        pointList := append(pointList,
                            [j * alpha, _height - rowCounter, _v2Color]);
      end_if;
    end_if;
  end_if;

  // More than 2 points ? Then every vertex is drawn in same color from now on.
  j := j + 1;
  if nops(_vertices) > 2 then
    for i from 3 to nops(_vertices) do
      if j >= _vpl then
        j := 0;
        rowCounter := rowCounter + 1;
        _vpl := _vplList[rowCounter+1];
        if _vpl = 1 then
          j := 1;
          alpha := maxWidth/2;
        else
          alpha := maxWidth / (_vpl - 1);   
        end_if;
      end_if;
      if not _vertices[i] = hold(None) then
        if contains(_specialVertexList, _vertices[i]) > 0 then
          pointList := append(pointList,
                              [j * alpha, _height - rowCounter, _specialVertexColor]);
        else
          pointList := append(pointList,
                              [j * alpha, _height - rowCounter, _vertexColor]);
        end_if;
      end_if;
      j := j + 1;
    end_for;
  end_if;

  // Now kill all the None't...
  tmpList := listlib::removeDuplicates(_vertices, KeepOrder);
  j := contains(_vertices, hold(None));
  if j > 0 then
    delete tmpList[j];
  end_if;
  _vertices := tmpList;

  specialDrawColor := {};
  drawColor := {};
  edgeList := [];
  specialEdgeList := [];
  for i from 1 to numberEdges do
    edge := _edges[i];
    v := contains(_vertices, edge[1]); 
    w := contains(_vertices, edge[2]);
      
    if v = w then
      drawEdge := plot::Ellipse2d(_vpl/20, _vpl/40,
                                  map(pointList[v][1..2],
                                      _mult, 1 + _vpl/200));
      colorOpt := plot::Ellipse2d::LineColor;
    else
      if _isDirected then
        drawEdge := plot::Arrow2d([op(pointList[v],1..2)],
                                         [op(pointList[w],1..2)]);
        colorOpt := plot::Arrow2d::LineColor;
      else
        drawEdge := plot::Line2d([op(pointList[v],1..2)],
                                 [op(pointList[w],1..2)]);
        colorOpt := plot::Line2d::LineColor;
      end_if;
    end_if;
    if (contains(_specialEdgeList, edge) > 0) then
      specialEdgeList := specialEdgeList.[drawEdge];
      specialDrawColor := specialDrawColor union {colorOpt = _specialEdgeColor}
    else
      edgeList := edgeList.[drawEdge];
      drawColor := drawColor union {colorOpt = _edgeColor}
    end_if;
  end_for;

  plot::Group2d(
       plot::Group2d(op(specialEdgeList),  op(specialDrawColor)),
       plot::Group2d(op(edgeList),  op(drawColor)),
       plot::PointList2d(pointList),
       Axes = None)
end_proc:


// End of file
null():
