/*-
 Output of dom
-*/
Graph::print := proc()
begin 
   "Graph(...)"
end_proc:

Graph::expr2text :=
proc(G)
  local i, t, edges;
begin
  edges := dom::getEdges(G);
  if not dom::isDirected(G) then
    for i from nops(edges) downto 1 do
      if contains(edges, revert(edges[i])) > 0 then
        delete edges[i];
      end_if;
    end_for;
  end_if;
  "Graph(".expr2text(dom::getVertices(G)).", ".
      expr2text(edges).
      if (t := dom::getVertexWeights(G)) = FAIL then "" else
        ", VertexWeights = [".
        stringlib::join([if contains(t, i) then expr2text(t[i]) 
                         else "None" end_if $ i in dom::getVertices(G)],
                        ", ")."]"
      end_if.
      if (t := dom::getEdgeDescriptions(G)) = FAIL then "" else
        ", EdgeDescriptions = [".
        stringlib::join([if contains(t, i) then expr2text(t[i]) 
                         else "None" end_if $ i in edges],
                        ", ")."]"
      end_if.
      if (t := dom::getEdgeWeights(G)) = FAIL then "" else
        ", EdgeWeights = [".
        stringlib::join([if contains(t, i) then expr2text(t[i]) 
                         else "None" end_if $ i in edges],
                        ", ")."]"
      end_if.
      if (t := dom::getEdgeCosts(G)) = FAIL then "" else
        ", EdgeCosts = [".
        stringlib::join([if contains(t, i) then expr2text(t[i]) 
                         else "None" end_if $ i in edges],
                        ", ")."]"
      end_if.
      if dom::isDirected(G) then ", Directed" else "" end.")";
end_proc:

/*
auxiliary method
Graph::outputStrings

replaces former calls to asciiArt used up to now
*/

Graph::outputStrings:=
proc(l: Type::ListOf(DOM_STRING))
  local i;
begin
  l:= revert(l);
  print(Unquoted, _concat((l[i], "\n") $i=1..nops(l)))
end_proc:

/*-
 printGraphInformation - prints some information of a Graph
 Parameters:
   G - Graph
-*/
Graph::printGraphInformation :=
proc(G : Graph)
  local data;
begin

  data := [if Graph::getVertices(G) = FAIL then
	     "Vertices: no vertices."
	   else
	     "Vertices: ".expr2text(Graph::getVertices(G))
	   end_if,
 
       if Graph::getEdges(G) = [] then
         if nops(Graph::getVertices(G)) > 0 then
            "Edges: no edges."
         else
           "Edges: []"
         end_if;
       else
         "Edges: ".expr2text(Graph::getEdges(G))
       end_if,

	   if Graph::getVertexWeights(G) = FAIL then
	     if nops(Graph::getVertices(G)) > 0 then
	       "Vertex weights: no vertex weights."
	     else null()
	     end_if; 
	   else
	     "Vertex weights: ".expr2text(op(Graph::getVertexWeights(G))).
	     " (other existing vertices have no weight)"
	   end_if,

	   if Graph::getEdgeDescriptions(G) = FAIL then
	     if nops(Graph::getEdges(G)) > 0 then
	       "Edge descriptions: no edge descriptions."
	     else null()
	     end_if;
	   else
	     "Edge descriptions: ".expr2text(op(Graph::getEdgeDescriptions(G)))
	   end_if,

	   if Graph::getEdgeWeights(G) = FAIL then
	     if nops(Graph::getEdges(G)) > 0 then
	       "Edge weights: no edge weights."
	     else null()
	     end_if;
	   else
	     "Edge weights: ".expr2text(op(Graph::getEdgeWeights(G))).
	     " (other existing edges have no weight)"
	   end_if,

	   if Graph::getEdgeCosts(G) = FAIL then
	     if nops(Graph::getEdges(G)) > 0 then
	       "Edge costs: no edge costs."
	     else null()
	     end_if;
	   else
	     "Edge costs: " . expr2text(op(Graph::getEdgeCosts(G))) .
	     " (other existing edges have costs zero)"
	   end_if,

	   if Graph::getEdgesLeaving(G) = FAIL then
	     if nops(Graph::getEdges(G)) > 0 then
	       "Adjacency list (out): no edges."
	     else null()
	     end_if;
	   else
	     "Adjacency list (out): ".expr2text(op(Graph::getEdgesLeaving(G)))
	   end_if,

	   if Graph::getEdgesEntering(G) = FAIL then
	     if nops(Graph::getEdges(G)) > 0 then
	       "Adjacency list (in): no edges."
	     else null()
	     end_if;
	   else
	     "Adjacency list (in): ".expr2text(op(Graph::getEdgesEntering(G)))
	   end_if,

	   if Graph::isDirected(G) then
	     "Graph is directed."
	   else
	     "Graph is undirected."
	   end_if];

  Graph::outputStrings(revert(data))
end_proc:

/*-
 printVertexInformation - prints detailed information about the vertices of a Graph
 Parameters:
   G - Graph
-*/
Graph::printVertexInformation :=
proc(G : Graph)
  local vertices, vertexWeights, data;
begin
  if Graph::getVertices(G) = FAIL then
    Graph::outputStrings(["No vertices."])
  else
    data := ["Vertices existing in the graph:",
	     "-------------------------------"];
    vertices := Graph::getVertices(G);
    vertexWeights := Graph::getVertexWeights(G);
    data := data . map(vertices,
		       v ->
		       "Vertex " . expr2text(v) . " has weight ".
		       if (vertexWeights = FAIL) or (contains(vertexWeights,v) = FALSE)then
			 "None"
		       else
			 expr2text(vertexWeights[v])
		       end_if);
    Graph::outputStrings(revert(data));
  end_if;
end_proc:

/*-
 printEdgeInformation - prints detailed information about the edges of a Graph
 Parameters:
   G - Graph
-*/
Graph::printEdgeInformation :=
proc(G)
begin
  Graph::graphCheck(G);

  if Graph::getEdges(G) = FAIL then
    Graph::outputStrings(["No edges."]);
  else
    Graph::outputStrings([expr2text(op(Graph::getEdges(G))),
		      "----------------------------",
		      "Edges existing in the graph:"]);
  end_if;
end_proc:

/*-
 printEdgeDescriptionInformation - prints detailed information about the edge descriptions of a Graph
 Parameters:
   G - Graph
-*/
Graph::printEdgeDescInformation := proc(G)
local  edges, edgeDescriptions, data;
begin
  Graph::graphCheck(G);

  PRETTYPRINT := FALSE;
  edges := Graph::getEdges(G);
  edgeDescriptions := Graph::getEdgeDescriptions(G);
  if edgeDescriptions = FAIL then
    Graph::outputStrings(["No edge descriptions defined."]);
  else
    data := ["Edge descriptions existing in the graph:",
	     "----------------------------------------"].

	  map(edges, edge -> "Edge " . expr2text(edge) . " = " .
			    if edgeDescriptions = FAIL or
			       contains(edgeDescriptions, edge) = FALSE then
			      "no description."
			    else
			      expr2text(edgeDescriptions[edge])
			    end_if);
    Graph::outputStrings(revert(data));
  end_if;
end_proc:

/*-
 printEdgeCostInformation - prints detailed information about the edge costs of a Graph
 Parameters:
   G - Graph
-*/
Graph::printEdgeCostInformation := proc(G)
local edges, edgeCosts, data;
begin
  Graph::graphCheck(G);

  edges := Graph::getEdges(G);
  edgeCosts := Graph::getEdgeCosts(G);
  if edgeCosts = FAIL then
    Graph::outputStrings(["No edge costs defined."]);
  else
    data := ["Edge costs existing in the graph:",
	     "----------------------------------------"] .
	  map(edges,
	      edge -> "Edge " . expr2text(edge) . " has cost ".
		     if (edgeCosts = FAIL) or (contains(edgeCosts, edge) = FALSE)then
		       "0"
		     else
		       expr2text(edgeCosts[edge])
		     end_if);
    Graph::outputStrings(revert(data));
  end_if;
end_proc:

/*-
 printEdgeWeightInformation - prints detailed information about the edge weights of a Graph
 Parameters:
   G - Graph
-*/
Graph::printEdgeWeightInformation := proc(G)
local edges, edgeWeights, data;
begin
  Graph::graphCheck(G);

  edges := Graph::getEdges(G);
  edgeWeights := Graph::getEdgeWeights(G);

  if edgeWeights = FAIL then
    Graph::outputStrings(["No edge weights defined."])
  else
    data := ["Edge weights existing in the graph:",
	     "----------------------------------------"].
	  map(edges,
	      edge -> "Edge " . expr2text(edge) . " has weight ".
		     if edgeWeights = FAIL or contains(edgeWeights, edge) = FALSE then
		       "0"
		     else
		       expr2text(edgeWeights[edge])
		     end_if);
    Graph::outputStrings(revert(data));
  end_if;
end_proc:

// End of file
null():
