/*
  shortestPathAllPairs - finds a shortest path from every vertex to every other vertex.
  shortestPathSingleSource(G <, SearchFor = Weights|Costs>)
  Parameters:
  G          - Graph
  SearchFor  - either Weights or Costs. Default = Weights.
*/
Graph::shortestPathAllPairs := proc(G : Graph)
local searchFor, leftOperator, rightOperator, i;
begin 
  if Graph::isEmpty(G) then
  	 error("Graph must have at least one edge to make a search!");
  end_if;

  searchFor := hold(Weights);

  for i from 2 to args(0) do
    if not (type(args(i)) = "_equal") then
      error("Wrong type of " . expr2text(i) . ": equation expected!");
    end_if;
    leftOperator := op(args(i),1); 
    rightOperator := op(args(i),2);

    case leftOperator
      of hold(SearchFor) do
        if rightOperator <> hold(Weights) and rightOperator <> hold(Costs) then
          error("Wrong argument type for " . expr2text(leftOperator) . ": Weights or Costs expected!");
        end_if;
        searchFor := rightOperator;        
        break;
    end_case;
  end_for;

  Graph::apsp_Floyd(G, searchFor);

end_proc:

/*-
 Algorithm for determining all pair shortest paths.
 Algorithm of Floyd, "Algorithm 97, Shortest Path",
 Comm. ACM, Vol. 5, 1962
 Running time O(|Vertices|^3)
 Taken from: Ahuja, Magnanti, Orlin: Graph Flows, Prentice-Hall, 1993
       Section 5.6

 The first table returned shows the sum of the way
 The second table shows the predecessor of the second vertex (i._edge. [a,5] = 4 => 4 is the predecessor vertex of 5)
 Using a constant C was not possible, because it would be reduced and in the end not being detected anymore.
 apsp_Floyd(G)

 Parameters:
   G       - Graph
-*/
Graph::apsp_Floyd := proc(G : Graph)
save PRETTYPRINT;
local i, j, k, bestResults, actWeight, bestPredecessor, _vertices, _edges, _edgeWeights, searchWith;
begin
  searchWith := hold(Weights);

  if args(0) = 2 then    
    case args(2)
      of hold(Costs) do
        searchWith := hold(Costs);
        break;
      of hold(Weights) do
        break;
      otherwise
        error("Wrong argument definition (try: Weights or Costs)!");
    end_case;
  end_if;

  _vertices := Graph::getVertices(G);
  _edges := Graph::getEdges(G);

  if searchWith = hold(Weights) then
    _edgeWeights := Graph::getEdgeWeights(G);
  else
    _edgeWeights := Graph::getEdgeCosts(G);
  end_if;

  // Otherwise an error occurs in contains(...)
  if _edgeWeights = FAIL then
    _edgeWeights := table();
  end_if;

  for i in _vertices do
    for j in _vertices do 
      if i=j then 
        bestResults[i,j] := 0;
      elif contains(_edges, [i,j]) > 0 then
        if contains(_edgeWeights, [i,j]) then
          actWeight := _edgeWeights[[i,j]];
        else
          actWeight := 1;
        end_if;        
        bestResults[i,j] := actWeight;
        bestPredecessor[i,j] := i;
      else
        bestResults[i,j] := infinity;
        // It't the predecessor and it will not be used fopr calculations !
        // Therefore infinity means, no predecessor is available.
        bestPredecessor[i,j] := infinity; 
      end_if
    end_for;
  end_for;

  for k in _vertices do      
    for i in _vertices do
      if bestResults[i,k] + bestResults[k,i] < 0 then 
        error("Graph contains one or more negative cycles!");
      end_if;
      for j in _vertices do
        if bestResults[i,j] > bestResults[i,k] + bestResults[k,j] then
         bestResults[i,j] := bestResults[i,k] + bestResults[k,j];
         bestPredecessor[i,j] := bestPredecessor[k,j];
        end_if;
      end_for;
    end_for;
  end_for;

  // Return two tables
  [bestResults,bestPredecessor];
end_proc:

// End of file
null():
