/*
  shortestPathSingleSource - finds a shortest path from a given vertex to a given or every vertex in a graph.
  shortestPathSingleSource(G, startVertex <, EndVertex = [n]> <, SearchWith = Dijkstra|Bellman> <, SearchFor = Weights|Costs> <,ReturnAsTable>)
  Parameters:
  G             - Graph
  StartVertex   - List containing one vertex from G.
  EndVertex     - List containing one vertex from G.
  SearchWith    - either Bellman or Dijkstra. Default = Bellman.
  SearchFor     - either Weights or Costs. Default = Weights.
  ReturnAsGraph - if TRUE a Graph is returned (only possible, if EndVertex is given !) a TABLE otherwise. 
*/
Graph::shortestPathSingleSource := proc(G : Graph)
local searchWith, searchFor, leftOperator, rightOperator, endVertex, _output, _solution, _edges, tmpVertex,
      _edgeWeights, _edgeCosts, _edgeDescriptions, _edgesLeaving, _edgesEntering, _vertices, _vertexWeights,
      wayTable, predTable, i, dummyTable, arguments, startVertex;

begin 
    if (args(0) < 2) then  // StartVertex is needed!
      error("Wrong number of arguments! See help for more details.")
    end_if;

  if Graph::isEmpty(G) then
    error("Graph must have at least one edge to make a search!");
  end_if;

  searchWith := hold(Bellman);
  searchFor := hold(Weights);
  _output := hold(Tables);
  endVertex := FAIL;

  arguments := args();
  for i from 2 to args(0) do
    if arguments[i] = hold(ReturnAsGraph) then
      _output := hold(Graph);
      delete arguments[i];
      break;
    end_if;
  end_for;

  for i from 2 to nops(arguments) do
    if not (type(arguments[i]) = "_equal") then
      error("Wrong type of " . expr2text(arguments[i]) . ": equation expected!");
    end_if;
    leftOperator := op(arguments[i],1); 
    rightOperator := op(arguments[i],2);

    case leftOperator
      of hold(SearchWith) do
        if rightOperator <> hold(Bellman) and rightOperator <> hold(Dijkstra) then
          error("Wrong argument type for " . expr2text(leftOperator) . ": Bellman or Dijkstra expected!");
        end_if;
        searchWith := rightOperator;        
        break;
      of hold(SearchFor) do
        if rightOperator <> hold(Weights) and rightOperator <> hold(Costs) then
          error("Wrong argument type for " . expr2text(leftOperator) . ": Weights or Costs expected!");
        end_if;
        searchFor := rightOperator;        
        break;
      of hold(StartVertex) do
        if domtype(rightOperator) <> DOM_LIST then
          error("Wrong argument type for " . expr2text(leftOperator) . ": list expected!");
        end_if;
        startVertex := rightOperator;
        break;
      of hold(EndVertex) do
        if domtype(rightOperator) <> DOM_LIST then
          error("Wrong argument type for " . expr2text(leftOperator) . ": list expected!");
        end_if;
        endVertex := op(rightOperator);
        break;
    end_case;
  end_for;

  if endVertex = FAIL then
    if _output = hold(Graph) then
      _output := hold(Tables); // Switch back to tables for output. (Definition)
    end_if;
  end_if;

  if searchWith = hold(Bellman) then    
    _solution := Graph::sssp_Bellman(G, startVertex, searchFor);
  else
    _solution := Graph::sssp_Dijkstra(G, startVertex, searchFor);
  end_if;

  if endVertex = FAIL then
    // just return the solution.
    return (_solution);
  end_if;

  startVertex := op(startVertex);
  tmpVertex := op(endVertex);
  _vertices := [tmpVertex];
  _edges := [];
  // Patch by Nicolas M. Thiry
  // Make sure that those are initialized, even when endVertex is not
  // accessible from startVertex
  predTable := table();
  wayTable := table();

  if contains(op(_solution,2), tmpVertex) then
    // endVertex is accessible from startVertex
    // Gather information about vertices and edges existing in shortest path
    while not tmpVertex = startVertex do
      _edges := append(_edges, [op(_solution,2)[tmpVertex], tmpVertex]);
      wayTable[tmpVertex] := op(_solution,1)[tmpVertex];
      // For the table output create the way
      if not tmpVertex = startVertex then
        predTable[tmpVertex] := op(_solution,2)[tmpVertex];
      end_if;
      tmpVertex := op(_solution,2)[tmpVertex];
      _vertices := append(_vertices, tmpVertex);
    end_while;
    wayTable[tmpVertex] := op(_solution,1)[tmpVertex];
  end_if;
  // End of patch

  if _output = hold(Tables) then
    return([wayTable, predTable]);    
  end_if;

  dummyTable := Graph::getVertexWeights(G);
  if (not dummyTable = FAIL)  then
    for i in _vertices do
      _vertexWeights[i] := dummyTable[i];
    end_for;
  else
    _vertexWeights := FAIL;
  end_if;

  dummyTable := Graph::getEdgeCosts(G);
  if (not dummyTable = FAIL)  then
    for i in _edges do
      _edgeCosts[i] := dummyTable[i];
    end_for;
  else
    _edgeCosts := FAIL;
  end_if;

  dummyTable := Graph::getEdgeWeights(G);
  if (not dummyTable = FAIL)  then
    for i in _edges do
      _edgeWeights[i] := dummyTable[i];
    end_for;
  else
    _edgeWeights := FAIL;
  end_if;

  dummyTable := Graph::getEdgeDescriptions(G);
  if (not dummyTable = FAIL)  then
    for i in _edges do
      _edgeDescriptions[i] := dummyTable[i];
    end_for;
  else
    _edgeDescriptions := FAIL;
  end_if;

  _edgesEntering := table((_vertices[i]=sort( map(select(_edges, _edge->op(_edge,2)=_vertices[i]), op, 1) ) $ i=1..nops(_vertices))):
  _edgesLeaving  := table((_vertices[i]=sort( map(select(_edges, _edge->op(_edge,1)=_vertices[i]), op, 2) ) $ i=1..nops(_vertices))):

  subsop(G, 1=_vertices, 2=_edges, 3=_vertexWeights, 4=_edgeDescriptions, 5=_edgeWeights, 6=_edgeCosts, 7=_edgesEntering, 8=_edgesLeaving);

end_proc:

/*- 
 sssp_Dijkstra - returns the shortest path length (weights or costs are used) from one single node to every other
 Running time O(|Vertices|^2).
 Works correctly for non-negative weights only.
 Uses a subroutine 'choose' for choosing a vertex with minimum distance label.
 sssp_Dijkstra(G, startVertex <,Weights|Costs>)
 Parameters:
   G           - Graph
   startVertex - The vertex from which the shortest path to every other vertex is to be found
-*/
Graph::sssp_Dijkstra := proc(G : Graph, startVertex : DOM_LIST)
			    : Type::ListProduct(DOM_TABLE, DOM_TABLE)
save PRETTYPRINT;
local setToFill, startSet, C, bestResults, bestVertex, j, bestPredecessor, 
      actWeight, _edgeWeights, _edgesLeaving, _vertices, i, searchWith, extractMin;
begin 
  if (args(0) < 2) or (args(0) > 3) then
    error("Wrong number of arguments! See help for more details.")
  end_if;
  if args(0) = 3 then    
    if not args(3) = hold(Costs) and not args(3) = hold(Weights) then
      error("Wrong argument type: " . expr2text(args(3)) . ". Please consult help for more information.");
    end_if;
  end_if;

  searchWith := hold(Weights);

  if args(0) = 3 then    
    case args(3)
      of hold(Costs) do
        searchWith := hold(Costs);
        break;
      of hold(Weights) do
        break;
      otherwise
        error("Wrong argument definition (try: Weights or Costs)!");
    end_case;
  end_if;
  
  extractMin := proc(startSet, bestResults)
  local i, tempBestVertex, bestVertex;
  begin
    bestVertex := op(startSet,1);
    tempBestVertex := bestResults[bestVertex];
    for i in startSet do
      if bestResults[i] < tempBestVertex then
        bestVertex := i; 
        tempBestVertex := bestResults[i];
      end_if;
    end_for;
    bestVertex;
  end_proc:
  
  _vertices := Graph::getVertices(G);
  _edgesLeaving := Graph::getEdgesLeaving(G);
  if searchWith = hold(Weights) then
    _edgeWeights := Graph::getEdgeWeights(G);
  else
    _edgeWeights := Graph::getEdgeCosts(G);
  end_if;

  // Otherwise an error occurs in contains(...)
  if _edgeWeights = FAIL then
    _edgeWeights := table();
  end_if;

  // Patch by Nicolas M. Thiry:
  // Make sure that the return value is always a pair of table even
  // when there are no predecessors
  bestPredecessor := table();

  if contains(_vertices, op(startVertex)) = 0 then
     error("The starting vertex is not in the set of vertices!");
  end_if;

  /*- constant C was chosen, because calculations with infinity are quite expensive!!  -*/
  C := (nops(_vertices)+1) * max(op(_edgeWeights, [1..nops(_edgeWeights),2]));
  // If edges have no weight, weight 1 is assumed for every edge.
  if C = FAIL then
    C := (nops(_vertices)+1);
  end_if;

  setToFill := {}; 
  startSet := {op(_vertices)};

  bestResults := table((_vertices[i] = C $ i=1..nops(_vertices)));
  bestResults[op(startVertex)] := 0;

  while nops(setToFill) < nops(_vertices) do 
    bestVertex := extractMin(startSet, bestResults);
    setToFill := setToFill union {bestVertex}; 
    startSet := startSet minus {bestVertex};
    for j in _edgesLeaving[bestVertex] do
      if contains(_edgeWeights, [bestVertex,j]) then
        actWeight := _edgeWeights[[bestVertex,j]];
      else
        actWeight := 1;
      end_if;
      if bestResults[j] > bestResults[bestVertex] + actWeight then
        bestResults[j] := bestResults[bestVertex] + actWeight;
        if bestResults[bestVertex] + actWeight >= 0 then
          // Otherwise, the procedure would get the wrong predecessor !
          bestPredecessor[j] := bestVertex;         
        end_if;
      end_if;
    end_for;
  end_while;

  if C <> 0 then
    bestResults := subs(bestResults, C=infinity);
  end_if;

  userinfo(NoNL, 5, "Starting vertex was " . expr2text(op(startVertex)) ."\n" );
  for i in bestResults do
    if op(i,1) <> op(startVertex) then
      userinfo(NoNL, 5, "The shortest way to vertex " . expr2text(op(i,1)) . " had length " . expr2text(bestResults[op(i,1)]) . " and ");
      if contains(bestPredecessor, op(i,1)) then
        userinfo(NoNL, 5, "its predecessor was " . bestPredecessor[op(i,1)] . "\n");
      else
        userinfo(NoNL, 5, "no vertex as predecessor.\n");
      end_if;
    end_if;
  end_for;  
  // End of patch

  // Return two tables
  [bestResults,bestPredecessor];
end_proc:

/*-
 sssp_Bellman - shortest path algorithm of Bellman
 Gives shortest pathes from one single node
 "On a routing problem", Quart. Appl. Math. 16 (1958)
 Running time O(|Vertices| |Edge|) (FIFO implementation)
 Taken from: Ahuja, Magnanti, Orlin: Graph Flows, Prentice-Hall, 1993
             Section 5.4

 sssp_Bellman(G, startVertex <,Weights|Costs>)
 Parameters:
   G           - Graph
   startVertex - The vertex from which the shortest path to every other vertex is to be found
-*/
Graph::sssp_Bellman := proc(G : Graph, startVertex : DOM_LIST)
			    : Type::ListProduct(DOM_TABLE, DOM_TABLE)
save PRETTYPRINT;
local bestResults, j, C, LIST, i, bestPredecessor, actWeight, pass, _vertices, _edgesLeaving, _edgeWeights, bestVertex, searchWith;
begin
  if (args(0) < 2) or (args(0) > 3) then
    error("Wrong number of arguments (2 or 3 expected)! See help for more details.")
  end_if;
  if args(0) = 3 then    
    if not args(3) = hold(Costs) and not args(3) = hold(Weights) then
      error("Wrong argument type: " . expr2text(args(3)) . ". Please consult help for more information.");
    end_if;
  end_if;

  searchWith := hold(Weights);

  if args(0) = 3 then    
    case args(3)
      of hold(Costs) do
        searchWith := hold(Costs);
        break;
      of hold(Weights) do
        break;
      otherwise
        error("Wrong argument definition (try: Weights or Costs)!");
    end_case;
  end_if;

  _vertices := Graph::getVertices(G);
  _edgesLeaving := Graph::getEdgesLeaving(G);

  if searchWith = hold(Weights) then
    _edgeWeights := Graph::getEdgeWeights(G);
  else
    _edgeWeights := Graph::getEdgeCosts(G);
  end_if;

  // Otherwise an error occurs in contains(...)
  if _edgeWeights = FAIL then
    _edgeWeights := table();
  end_if;

  // Patch by Nicolas M. Thiry:
  // Make sure that the return value is always a pair of table even
  // in the trivial cases
  bestPredecessor := table();

  if contains(_vertices, op(startVertex)) = 0 then
    error("The starting vertex is not in the set of vertices!");
  end_if;   

  /*- constant C was chosen, because calculations with infinity are quite expensive!!  -*/
  if nops(_edgeWeights) = 0 then
    C := (nops(_vertices)+1);
  else
    C := (nops(_vertices)+1) * max(op(_edgeWeights, [1..nops(_edgeWeights),2]));  	
  end_if;

  bestResults := table((_vertices[i] = C $ i=1..nops(_vertices)));
  bestResults[op(startVertex)] := 0;
  LIST := {op(startVertex)};
  pass := table((_vertices[i] = 0 $ i=1..nops(_vertices)));
  while nops(LIST) > 0 do
    bestVertex := LIST[1];
    LIST := LIST minus {bestVertex};
    pass[bestVertex] := pass[bestVertex] + 1;
    if pass[bestVertex] > nops(_vertices) then
       error("Graph contains negative cycle(s)!");
    end_if;
    for j in _edgesLeaving[bestVertex] do
      if contains(_edgeWeights, [bestVertex,j]) then
        actWeight := _edgeWeights[[bestVertex,j]];
      else
        actWeight := 1;
      end_if;
      if bestResults[j] > bestResults[bestVertex] + actWeight then
        bestResults[j] := bestResults[bestVertex] + actWeight;
        bestPredecessor[j] := bestVertex;
        if contains(LIST, j) = FALSE then
          LIST := LIST union {j};
        end_if;
      end_if;
    end_for;
  end_while;

  if C <> 0 then
    bestResults := subs(bestResults, C=infinity);
  end_if;

  userinfo(NoNL, 5, "Starting vertex was " . expr2text(op(startVertex)) ."\n" );
  for i in bestResults do
    if op(i,1) <> op(startVertex) then
      userinfo(NoNL, 5, "The shortest way to vertex " . expr2text(op(i,1)) . " had length " . expr2text(bestResults[op(i,1)]) . " and ");
      if contains(bestPredecessor, op(i,1)) then
        userinfo(NoNL, 5, "its predecessor was " . bestPredecessor[op(i,1)] . "\n");
      else
        userinfo(NoNL, 5, "no vertex as predecessor.\n");
      end_if;
    end_if;
  end_for;
  // End of patch

  // Return two tables
  [bestResults,bestPredecessor];
end_proc:

// End of file
null():
