/*++
gbasis  - returns reduced normalized Groebner-basis for polynomials in list S

gbasis(S [,o] [,r])

S - set or list of polynomials
o - optional term ordering (LexOrder, DegInvLexOrder or DegreeOrder)
r - optional identifier 'Reorder'

If no term ordering is given, 'DegInvLexOrder' is used.

If the option 'Reorder' is given, the order of the indets may be changed
during the computation of the basis. (The order of the indets is changed
to be "heuristically optimal" in the sense of Boege, Gebauer and Kredel.)
In this case the result is returned with the re-arranged variables.
++*/

groebner::gbasis:=
proc(S: Type::Union(DOM_LIST, DOM_SET)) : DOM_LIST
  option escape;
  local reord, X, o, zero, monic, conv,
  options: DOM_TABLE;
  
begin
  if nops(S) = 0 then
    error("no polynomials in list")
  end_if;
  
  options:= groebner::getoptions(args(2..args(0)));

  o:= options["Order"];
  reord:= options["Reorder"];
  monic:= options["Monic"];


  if type(S)=DOM_SET then
    S:=[op(S)]
  end_if;
  
  if domtype(S[1]) = DOM_POLY then
    conv:= id
  elif (S[1])::dom::poly <> FAIL then
    conv:= (S[1])::dom::new;
    S:= map(S, poly)
  else  
    S:= gcdlib::expr2polys(S, Type::Rational);
    if S = FAIL then
      error("not a rational polynomial")
    end_if;
    conv:= expr;
    reord:= TRUE;	// the user didn't care 
  end_if;

  if testargs() then
    groebner::test_polyset(S)
  end_if;
  
  if stdlib::hasfloat(S) then
    S:=map(S, mapcoeffs, numeric::rationalize);
    userinfo(30, "Rationalizing coefficients gives ".
             expr2text(S));
    S:=groebner::gbasis(S, args(2..args(0)));
    return(map(S, mapcoeffs, float))
  end_if;
  
  zero:= S[1];
  S:= select(S, not iszero);
  if nops(S) = 0 then
    return([zero])
  end_if;

  if reord then
	// re-arrange variables 
    X:= groebner::opt_order(S);
    userinfo(1,"chosen order is",X);
    if X <> op(S[1],2) then
      S:= map(S, poly, X)
    end_if
  end_if;

  zero:=S[1]^0-S[1]^0;

  // install normalizing routine
  if monic = hold(Default) then
    if op(S[1],3) = hold(Expr) then
      monic:=groebner::primpart;
    else
      monic:=groebner::normalize;
    end_if;
  end_if;
  

  // normalize polynomials 
  S:= map(S, monic, o);

  // compute and reduce basis 
  S:= groebner::reduce_basis
  (groebner::basis(S, o, monic, _plus),o, monic, _plus);

  if nops(S) = 0 then
    S:= [ zero ]
  elif op(S[1],3) = hold(Expr) then
    // change into unit normal form 
    S:= map(S, (()->((
                      if sign(lcoeff(args(1), o)) < 0 then
                        multcoeffs(args(1), -1)
                      else
                        args(1)
                      end_if))))
  end_if;

  S:=map(S, conv);
  /* sort the basis according to the given order
            so that the polynomials with greatest term are first */
  sort(S,(()->(lmonomial(args(1)+args(2),o)=
               lmonomial(args(1),o))));
end_proc:

// end of file 
