/*----------------------------------------------------------------
import::readbitmap - reads bitmap files

Calling sequence: 

   import::readbitmap(file)

Parameter: file - non-empty string or a positive integer
                  (a file descriptor as returned by fopen)

Summary:

       - import::readbitmap is used to read bitmap data files
         produced by external programs. 

       - The internal module 'bitmap' claims to handle the 
         following bitmap formats:
 
            BMP, DCX, DDS, WAD, GIF, ICO, JPG, LIF,
            MDL, PCD, PCX, PIC, PIX, PNG, PNM, PSD,
            PSP, PXR, RAW, SGI, TGA, TIF, WAL, XPM.
 
         Experimentally, I found that
                 DCX, ICO, PCD, PCX
         do not seem to work. (--> double check)
   
      -  If the file is specified by a string, the corresponding
         file is opened and closed, automatically.

      -  If the user has opened a text file in Read mode and passes
         the file descriptor to import::readbitmap, the file remains
         open and needs to be closed by the user.

Return Value: the list [width , height, colorArray],
              where width and height are integers and
              colorArray = array(1..height, 1..width, [[RGB-Values,..], ..])

Examples:
>> [width, height, data]:= import::readbitmap("Norton.jpg");

         [360, 240, array(1..240, 1..360, [[..]])]

>> [width, height, data]:= import::readbitmap("Norton.png"):
>> [width, height, data]:= import::readbitmap("Norton.ppm"):
---------------------------------------------------------------------*/

import::readbitmap := proc(filename)
local isf, p, width, height, data, returntype, load;
begin
   if args(0) < 1 or args(0) > 2 then
      error("Expecting one or two arguments");
   end_if:
   //-----------------------------
   // check the file specification
   //-----------------------------
   case domtype(filename) 
   of DOM_STRING do
      if length(filename) = 0 then
         error("Empty filename")
      end_if:
      break;
   of DOM_INT do
      // under windows the underlying module 
      // cannot handle a file descriptor.
      // So we do not allow file descriptors as input.
      error("Expecting a file name as a string");
   otherwise
      error("Illegal file specification")
   end_case;
   //-----------------------------
   // check the module 
   //-----------------------------
   if domtype(stdlib::bitmap) <> DOM_DOMAIN then
      error("Cannot open the module for reading bitmap files");
   end_if;
   //--------------------------------
   // check return type specification
   //--------------------------------
   returntype := DOM_HFARRAY;
   if args(0) = 2 then
     if type(args(2)) <> "_equal" or lhs(args(2)) <> ReturnType then
       error("Second argument can only be ReturnType=...");
     end_if;
     returntype := rhs(args(2));
     if not returntype in {DOM_LIST, DOM_ARRAY, DOM_HFARRAY} then
       error("Supported return types: DOM_LIST, DOM_ARRAY, DOM_HFARRAY");
     end_if;
   end_if;

   load := if returntype = DOM_HFARRAY then
             stdlib::bitmap::loadRGB2hfarray
           else
             stdlib::bitmap::loadRGB
           end_if;

   //---------------------------------------
   // try to open the file and read the data 
   //---------------------------------------

   //---------------------------------------------------
   // search: 1) using READPATH
   //         2) as direct path
   //         3) using LIBPATH
   //---------------------------------------------------
   isf:= bool(filename[1] <> stdlib::PathSep);
   data:= FAIL;
   //--------------------------------
   // 1) try to open with READPATH
   //--------------------------------
   if domtype(READPATH) <> DOM_IDENT then
      for p in READPATH do
          if isf then
             if p[-1] <> stdlib::PathSep then
                p:= p.stdlib::PathSep
             end_if
          end_if;
          traperror((data:= load(p.filename)));
          if data <> FAIL then
             break;
          end_if;
      end_for;
   end_if;
   //--------------------------------
   // 2) try to open with direct path
   //--------------------------------
   if data = FAIL then
      traperror((data:= load(filename)));
   end_if;
   //--------------------------------
   // 3) try to open with LIBPATH
   //--------------------------------
   if data = FAIL then
      for p in LIBPATH do
          if isf then
              if p[-1] <> stdlib::PathSep then
                  p:= p.stdlib::PathSep
              end_if
          end_if;
          traperror((data:= load(p.filename)));
          if data <> FAIL then
             break;
          end_if;
      end_for;
   end_if;
   //--------------------------------
   // 4) give up
   //--------------------------------
   if data = FAIL then
      error("Cannot open file ".filename);
   end_if;

   //-----------------------------
   // check the data 
   //-----------------------------
   if domtype(data) <> DOM_LIST and domtype(data) <> DOM_HFARRAY then
      error("Cannot read he bitmap data");
   end_if;
   if nops(data) = 0 then 
      error("no bitmap data found");
   end_if;
   case returntype
   of DOM_LIST do
   of DOM_ARRAY do
     if domtype(data[1]) <> DOM_LIST then
        error("Unexpected bitmap format");
     end_if;
     if nops(data[1]) = 0 then 
        error("no bitmap data found");
     end_if;
     if domtype(data[1][1]) <> DOM_LIST or
        not nops(data[1][1]) in {3, 4} then
        error("Unexpected bitmap format");
     end_if;
     width:= nops(data[1]);
     height:= nops(data):
   end_case;

   case returntype
   of DOM_ARRAY do
     //-----------------------------------------
     // convert the nested list of RGB(a) values 
     // into an array and return
     //-----------------------------
     if traperror((data:= array(1..height, 1..width, data))) <> 0 then
        error("Unexpected bitmap format");
     end_if;
     break;
   of DOM_LIST do
     break;
   of DOM_HFARRAY do
     width := op(data, [0,3,2]);
     height := op(data, [0,2,2]);
   end_case;
   return([width, height, data]);
end_proc:

