/*----------------------------------------------------------------
import::readdata - reads data files

Calling sequence: 
        import::readdata(filename <, separator> <, NonNested> )

Parameter: filename  - non-empty string or a positive integer
                       (a file descriptor as returned by fopen)
           separator - optional string , length(separator) = 1
           NonNested - optional name

Summary: import::readdata (equivalent to import::readdata) is used 
         to read ascii data files produced by external programs. 

         In contrast to finput, the data must not be ended by a colon 
         or semicolon. Ascii data separated by 'separator' are interpreted 
         as single data items. The default separator is a white space.

         With NonNested, the result will be a list containing all data. 
         Otherwise, the result is a list of list, each 'inner' list
         representing a line of the ascii file.
         Empty lines are ignored.
         Data items that cannot be converted to a valid MuPAD object
         via text2expr are imported as MuPAD strings.
   
         If the file is specified by a string, the corresponding
         file is opened and closed, automatically.
         If the user has opened a text file in Read mode and passes
         the file descriptor to import::readdata, the file remains
         open and needs to be closed by the user.

Examples:
---------------------------------------------------
File data: 1 2 3
           4 5 6
           7 8 9
>> import::readdata("data");
              [ [1, 2, 3], [4, 5, 6], [7, 8, 9] ]
>> import::readdata("data", NonNested);
                 [ 1, 2, 3, 4, 5, 6, 7, 8, 9 ]
---------------------------------------------------
File data: 1|2|3
           4|5|6.65786

           7|8|9| 5 | "ahfjd" | ab
>> import::readdata("data","|");
          [[1, 2, 3], [4, 5, 6.65786], [7, 8, 9, 5, "ahfjd", ab]]

See also: read, finput, fread, ftextinput, text2expr, text2list
---------------------------------------------------------------------*/

import::readdata := proc(filename, separator=" ", opt)
local fd, isf, p, line, nested, r, t, i, j, tmp, extended_text2expr;
begin
   if args(0) = 0 then
      error("Expecting at least 1 argument");
   end_if:
   if args(0) > 3 then
      error("Expecting no more than 3 arguments");
   end_if:
   //-----------------------------
   // check the file specification
   //-----------------------------
   case domtype(filename) 
   of DOM_STRING do
      if length(filename) = 0 then
         error("Empty filename")
      end_if:
      break;
   of DOM_INT do
      if filename <= 0 then
         error("Expecting the file descriptor as a positive ".
               " integer, got ".expr2text(filename));
      end_if:
      break;
   otherwise
      error("Illegal file specification. Expecting a string or ".
            " a file descriptor (a positive integer). Got: ".expr2text(filename));
   end_case;
   //-----------------------------
   // check the options
   //-----------------------------
   nested := TRUE;
   case args(0)
     of 3 do 
        if opt = NonNested then
             nested := FALSE
        else error("Illegal 3rd argument. Expecting the flag 'NonNested', got :".
                   expr2text(args(3)))
        end_if;
     of 2 do 
        if separator = NonNested then
              nested := FALSE
        elif not (testtype(separator, DOM_STRING) and 
           length(separator) = 1) then
              error("Illegal separator");
        end_if
   end_case;
   //-----------------------------
   // try to open the file
   //-----------------------------
   if domtype(filename) = DOM_INT then
        // we have to assume that the user has opened
        // a file via fopen and passed the file descriptor
        // to import::readdata;
        fd := filename;
        // we rely on later calls to ftextinput to produce
        // appropriate error messages if the file descriptor
        // fd = filname specified by the user does not point
        // to a properly opened file
   else // The user passed a file name
        //---------------------------------------------------
        // search: 1) using READPATH
        //         2) as direct path
        //         3) using LIBPATH
        //---------------------------------------------------
        isf:= bool(filename[1] <> stdlib::PathSep);
        fd:= FAIL;
        //--------------------------------
        // 1) try to open with READPATH
        //--------------------------------
        if domtype(READPATH) <> DOM_IDENT then
           for p in READPATH do
               if isf then
                  if p[-1] <> stdlib::PathSep then
                     p:= p.stdlib::PathSep
                  end_if
               end_if;
               fd := fopen(p.filename, Read, Text);
               if fd <> FAIL then
                  break;
               end_if;
           end_for;
        end_if;
        //--------------------------------
        // 2) try to open with direct path
        //--------------------------------
        if fd = FAIL then
           fd := fopen(filename, Read, Text);
        end_if;
        //--------------------------------
        // 3) try to open with LIBPATH
        //--------------------------------
        if fd = FAIL then
           for p in LIBPATH do
               if isf then
                   if p[-1] <> stdlib::PathSep then
                       p:= p.stdlib::PathSep
                   end_if
               end_if;
               fd := fopen(p.filename, Read, Text);
               if fd <> FAIL then
                  break;
               end_if;
           end_for;
        end_if;
        //--------------------------------
        // 4) give up
        //--------------------------------
        if fd = FAIL then
           error("Cannot open file ".filename);
        end_if;
   end_if;

   //-------------------------------------------------------
   // Now, the file is opened and specified by the
   // file descriptor fd
   //-------------------------------------------------------

   // Try to get a number in cases like "139."
   extended_text2expr:= proc(x)
     local tmp;
   begin
     if x[-1] <> "." then
       return(x);
     end_if;
     if traperror((tmp:= eval(text2expr(x."0")))) = 0 then
       tmp;
     else
       x;
     end_if;
   end_proc;

   //-----------------------------
   // read the data
   //-----------------------------
   if not testtype(separator, DOM_STRING) then
      separator := " "
   end_if;
   r := table(): // a table of rows, indexed
   j := 1;       // via j = 1, 2, ...
   if traperror ((
     while type((line:=ftextinput(fd))) <> DOM_NULL do
       line:=     stringlib::subs(line, "\r"="");
       t := text2list(line, [separator]); 
       // now, t = [string1,separator,string2,separator,...]
       line := [t[1 + 2*i] $ i=0..(nops(t)-1) div 2];
       // now, l = [string1, string2,...]
       // If an item in the line can be converted to
       // a MuPAD expressions via text2expr, then do
       // convert! Otherwise, leave it as a string.
       line := map (line, x -> 
            (_if(traperror((tmp:= eval(text2expr(x)))) = 0, tmp, extended_text2expr(x))));
       if line <> [] then // skip empty lines
          r[j]:= line;    // add the line to the row table
          j:= j+1:        // increase the row counter
       end_if;
     end_while;
     r := [r[j] $ j = 1..nops(r)]:
     if (not nested) then 
        // flatten the list r
        r := map(r, op):
     end_if;
   )) <> 0 then // traperror ende
        if domtype(filename) = DOM_INT then 
             // the user passed a file descriptor.
             // Either the file descriptor is not valid,
             // or the file is not an ascii file.
             error("Cannot read the file with the descriptor ".  expr2text(fd).
                   ". Either the file was not opened via 'fopen', ".
                   "or it was not opened in 'Read' mode, ".
                   "or it is not a text file.");
        else // the user passed a file name. The file was
             // successfully opened above, but ftextinput
             // had problems.
             error("Cannot convert the data")
        end_if;
   end_if:
   if domtype(filename) = DOM_STRING then 
      fclose(fd);
   end_if:
   return(r);
end_proc:
