// 

// Evaluation of the gamma function, following
// C. Lanzcos, "A Precision Approximation of the Gamma Function",
// J. SIAM Numerical Analysis, Series B, Vol. 1, 1964, pp. 86-96

// Apart from the rd*-functions, the whole thing is valid for complex
// numbers with Re>0, too.  Reflection holds for complex arguments.

// Here, precomputed coefficients are stored.
// The first element is DIGITS used, the second one is
// the value of `&gamma;` for the coefficients and the
// third is a list of coefficients computed (in interval 
// arithmetics).  After that, an error estimate and a
// numerical approximation of the zero of psi(x) are stored.
DOM_INTERVAL::_precomputed_gamma_coeffs := [0, 0, [], 0...0, 1.46]:

// coefficients of chebycheff polynomials
DOM_INTERVAL::_cheby_coeffs:=[]:

DOM_INTERVAL::_compute_gamma_coeffs :=
proc(g)
  local k_max, i, rho, a, eta, min_of_gamma, x,
	sq2PI, sigma, e;
  save DIGITS;
begin
  if DIGITS <= DOM_INTERVAL::_precomputed_gamma_coeffs[1]
     and g = DOM_INTERVAL::_precomputed_gamma_coeffs[2] then return(); end;

  // TODO: compute required k_max.
  // This is only an upper bound and actually stopped by
  // 0 in rho[i] below.
  k_max := 250;
  x := genident("blabla");
  for i from nops(DOM_INTERVAL::_cheby_coeffs) to k_max do
    sysassign(DOM_INTERVAL::_cheby_coeffs, DOM_INTERVAL::_cheby_coeffs
      . [revert([coeff(orthpoly::chebyshev1(2*i,x))])]);
  end_for;

  rho := [0 $ k_max];
  
  sigma    := [0$k_max];
  sigma[1] := DOM_INTERVAL::exp(g+1/2);
  e := hull(exp(1));
  sq2PI := hull(sqrt(2/PI));
  rho[1] := sq2PI * DOM_INTERVAL::_cheby_coeffs[1][1]*sigma[1]/hull(sqrt(g+1/2));
  for i from 2 to k_max do
    sigma[i] := e*(i-3/2)*sigma[i-1];
    rho[i] := sq2PI*_plus((DOM_INTERVAL::_cheby_coeffs[i][a]
			   * sigma[a] * hull(a+g-1/2)^(1/2-a))
			  $a=1..i);
    // If a coefficient contains 0,
    // everything from there on is dominated by rounding errors
    if 0 in rho[i] then break; end;
  end_for;
  
  rho:=[op(rho,1..i)];

// The formula uses rho_0/2 only.  Simplify the code:
  rho[1] := rho[1]/2;

// claimed to be a quality measure
// note that in the paper, the factor PI/2 is
// not part of eta.
  eta := hull(exp(g)/sqrt(2));
  eta := (_plus(((-1)^(i+1)*rho[i])$i=1..nops(rho)) 
           - eta)/eta * hull(PI/2);

  eta := (-1...1) * eta;

  min_of_gamma := op(numeric::solve(psi(x)=0, x=1.4..1.5));

  sysassign(DOM_INTERVAL::_precomputed_gamma_coeffs,
        [DIGITS, g, rho, eta, min_of_gamma]);

end_proc:

// This function is used with x > 0.
DOM_INTERVAL::_internal_Gamma := proc(x, rounding, gamma)
local res, i, s1, s2, s, rho, rnd, ii;
begin
// this is a crude "seems to work" calculation
  if args(0) < 3 then gamma := 3+ceil(DIGITS/10); end_if;

  DOM_INTERVAL::_compute_gamma_coeffs(gamma);

  if iszero(x) then return(RD_INF); end_if;
  if x = RD_INF or x = infinity then return(RD_INF); end_if;

  // shifting the input down by 1 makes the formulas much less messy.
  x := rdplus(x, -1.0, rounding);

  if rounding = -1 then
    // lower bound
    res := lhs(DOM_INTERVAL::_precomputed_gamma_coeffs[3][1]);
    for i from 1 to nops(DOM_INTERVAL::_precomputed_gamma_coeffs[3])-1 do
      s1 := 1.0;
      s2 := 1.0;
      rho := DOM_INTERVAL::_precomputed_gamma_coeffs[3][i+1];
      if rho<0 then
        rnd := 1;
      else
        rnd := -1;
      end_if;
      for ii from 1 to i do
        s1 := rdmult(s1, x+ii, -rnd);
        s2 := rdmult(s2, x-ii+1, rnd);
      end_for;
      s := rho * rddivide(s2, s1, rnd);
      res := res +s;
    end_for;
// use the error bound
    res := res + (res * DOM_INTERVAL::_precomputed_gamma_coeffs[4]);
    res := (hull(sqrt(2*PI))
            *(x+gamma+1/2)^hull(x+1/2)
            *exp(hull(-x-gamma-1/2))
            *res);
  else
    // upper bound
    res := rhs(DOM_INTERVAL::_precomputed_gamma_coeffs[3][1]);
    for i from 1 to nops(DOM_INTERVAL::_precomputed_gamma_coeffs[3])-1 do
      s1 := 1.0;
      s2 := 1.0;
      rho := DOM_INTERVAL::_precomputed_gamma_coeffs[3][i+1];
      if rho<0 then
        rnd := -1;
      else
        rnd := 1;
      end_if;
      for ii from 1 to i do
        s1 := rdmult(s1, x+ii, -rnd);
        s2 := rdmult(s2, x-ii+1, rnd);
      end_for;
      s := rho * rddivide(s2, s1, rnd);
      res := res + s;
    end_for;
// use the error bound
    res := res + (res * DOM_INTERVAL::_precomputed_gamma_coeffs[4]);
    res := (hull(sqrt(2*PI))
            *(x+gamma+1/2)^hull(x+1/2)
            *exp(hull(-x-gamma-1/2))
            *res);
  end_if;

  return(res);

end_proc:

DOM_INTERVAL::gamma := proc(iv)
local t1, t2, l, r, b, t, psiiv;
begin
  if iv::dom <> DOM_INTERVAL then
     if iv = {} then return(iv); end_if;
     t1 := interval(iv);
     if type(t1) = DOM_INTERVAL then
        iv := t1;
     else
        return(FAIL);
     end_if;
  end_if;

  if op(iv,0) = hold(_union) then
     return(_union(map(op(iv), DOM_INTERVAL::gamma)));
  end_if;

  if op(iv,0) = FAIL then // complex interval
    // look for monotonicity
    psiiv := DOM_INTERVAL::psi(iv);
    if not 0 in DOM_INTERVAL::Re(psiiv) and
       not 0 in DOM_INTERVAL::Im(psiiv) then
      [l, r, b, t] := map([op(iv)], op);
      // evaluate at the corners
      return(hull(DOM_INTERVAL::exp(DOM_INTERVAL::ln_gamma(subsop(iv, [1,1]=r, [2,1]=t))),
                  DOM_INTERVAL::exp(DOM_INTERVAL::ln_gamma(subsop(iv, [1,1]=r, [2,2]=b))),
                  DOM_INTERVAL::exp(DOM_INTERVAL::ln_gamma(subsop(iv, [1,2]=l, [2,1]=t))),
                  DOM_INTERVAL::exp(DOM_INTERVAL::ln_gamma(subsop(iv, [1,2]=l, [2,2]=b)))));
    else
      return(DOM_INTERVAL::exp(DOM_INTERVAL::ln_gamma(iv)));
    end_if;
  end_if;

  // Is one of the poles inside the interval?
  if lhs(iv) < 0 then
     if lhs(iv) = RD_NINF or lhs(iv) = -infinity
       or rhs(iv) >= ceil(lhs(iv)) then
        return(RD_NINF...RD_INF);
     end_if;
     // No, but the interval contains (only) negative numbers.
     // use reflection formula
     t1 := DOM_INTERVAL::sin(hull(PI)*iv);
     t2 := DOM_INTERVAL::gamma(1-iv);
     return ((PI/rdmult(rhs(t1), rhs(t2), 1))
         ... (PI/rdmult(lhs(t1), lhs(t2), -1)));
  end_if;

  if rhs(iv) < DOM_INTERVAL::_precomputed_gamma_coeffs[5] then
    t1 := DOM_INTERVAL::_internal_Gamma(lhs(iv),  1);
    t2 := DOM_INTERVAL::_internal_Gamma(rhs(iv), -1);
  elif lhs(iv) > DOM_INTERVAL::_precomputed_gamma_coeffs[5] then
    t1 := DOM_INTERVAL::_internal_Gamma(lhs(iv), -1);
    t2 := DOM_INTERVAL::_internal_Gamma(rhs(iv),  1);
  else // DOM_INTERVAL::_precomputed_gamma_coeffs[5] in iv
    t1 := DOM_INTERVAL::_internal_Gamma(lhs(iv),  1);
    t2 := DOM_INTERVAL::_internal_Gamma(rhs(iv),  1);
    // minimum of gamma >= 0.885601.
    // TODO: Implement better approximation
    t1 := t1...0.885601;
  end_if;

  t1 ... t2;

end_proc:


//////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////
//
// gamma for complex arguments
//
// following the discussion in Edward W. Ng: "A Comparison of 
// Computational Methods and Algorithms for the Complex Gamma Function",
// ACM Transactions on Mathematical Software, Vol. 1, No. 1,
// March 1975 Pages 56-70.
// Note that on the 9th page, the analysis suggests Kuki's or Spira's
// algorithm.  That of Spira is much easier to implement in general.

// utility: Enclosures of Bernoulli numbers
DOM_INTERVAL::_bernoulli :=
  proc(n, digits)
    option remember;
    save DIGITS;
  begin
    DIGITS := digits;
    hull(bernoulli(n));
  end_proc:

DOM_INTERVAL::_ln_gamma_rem_spira :=
  proc(z, n, digits)
    option remember;
    save DIGITS;
  begin
    DIGITS := digits;
    DOM_INTERVAL::abs(DOM_INTERVAL::_bernoulli(2*n, digits)/
                      (2*n-1))
    * DOM_INTERVAL::abs(z)^(1-2*n);
  end_proc:

// ln(gamma(iv)), where iv is not a union and 0<=Re(iv)
// Note that this routine works best when abs(iv) is large

// TODO: Lots of optimization!
// - if rhs(abs(Im(iv))) < rhs(abs(Re(iv))), p. 59
//   of Ngs paper lists an error bound due to Lucas
//   and Terril which is perfect for a dynamic stopping
//   criterion
// - the coefficients of the _plus call should be calculated
//   iteratively.
// - Check the formulas for removable dependency and wrapping
//   problems.
// - precompute constants such as ln(2*PI)
// - For some specified values of DIGITS, we might want Kuki's
//   error bound and use the recursion formula.  Then again, this
//   introduces more dependency issues.
DOM_INTERVAL::_ln_gamma :=
  proc(iv)
    local n, k, res, s, term, lucas;
  begin
    n := 10;
    res := (RD_NINF...RD_INF) + (RD_NINF...RD_INF)*I;
    s := (iv-1/2)*DOM_INTERVAL::ln(iv)-iv+DOM_INTERVAL::ln(2*hull(PI))/2
       + DOM_INTERVAL::_bernoulli(2, DIGITS)/(2*iv);
    if rhs(abs(Im(iv))) <= rhs(abs(Re(iv))) then
      lucas := () -> (res := res intersect (s + (((-1...1)+(-I...I)) * term)));
    else
      lucas := FALSE;
    end_if;
    for k from 2 to n do
      term := DOM_INTERVAL::_bernoulli(2*k, DIGITS)/(2*k*(2*k-1)*iv^(2*k-1));
      lucas();
      s := s + term;
      res := res intersect
           (s + (-1...1+(-I...I)) * DOM_INTERVAL::_ln_gamma_rem_spira(iv, k, DIGITS));
    end_for;
  end_proc:
  
DOM_INTERVAL::ln_gamma :=
  proc(iv)
    local p, n;
  begin
    if op(iv, 0) = hold(_union) then
      return(_union(op(map([op(iv)], DOM_INTERVAL::ln_gamma))));
    end_if;
    
    p := iv intersect ((0...RD_INF)+(RD_NINF...RD_INF)*I);
    n := iv intersect ((RD_NINF...0)+(RD_NINF...RD_INF)*I);
    
    if p <> {} then
      if op(DOM_INTERVAL::abs(p), 1) < 4 then
        // use the recursion formula to get a larger absolute value
        // for the computation converges only then.
        p := DOM_INTERVAL::_ln_gamma(p+4)-
            DOM_INTERVAL::ln(p  )-DOM_INTERVAL::ln(p+1)-
            DOM_INTERVAL::ln(p+2)-DOM_INTERVAL::ln(p+3);
      else
        p := DOM_INTERVAL::_ln_gamma(p);
      end_if;
    end_if;
    
    if n <> {} then
      n := DOM_INTERVAL::ln(hull(PI)*DOM_INTERVAL::csc(hull(PI)*n))
          -DOM_INTERVAL::ln_gamma(1-n);
    end_if;
    
    p union n;
  end_proc:
