// 

/*
  interval::gldata -- compute verified inclusions for the evaluation
                      points and weights for Gauss-Legendre quadrature.

*/


/*

  utility: evaluate Legendre polynomials, returning L_n(x) and L'_n(x).


  With direct evaluation and DIGITS=10, the sign of p_54(0.9), e.g.,
  is undetermined:

>> orthpoly::legendre(54, hull(9/10))

                      -0.1020817817 ... 0.1763593624

 The thesis of Wedner contains an algorithm claimed to give better
 evaluations of Legendre polynomials.  So far, I have not been able to
 reproduce his findings.

*/

/*

p(-1, x) = 0
p(0, x) = 1
p(n, x) = (2*n-1)/n*x*p(n-1, x) - (n-1)/n*p(n-2, x)

p'(n, x) = n*(x*p(n, x) - p(n-1, x))/(x^2 - 1)

*/

// first, trivial implementation.  As long as this suffices
// to get good decisions on signs, use it -- it is much faster.
// As said above, Wedner's thesis might contain an alternative.
interval::legendre_direct_ :=
  proc(n, x)
    local v1, v2, t, i;
  begin
    assert(n mod 2 = 0);
    v1 := x;
    v2 := (3*x^2-1)/2;
    for i from 3 to n step 2 do
      t  := ((2*i-1)*x*v2 - (i-1)*v1)/i;
      v2 := ((2*i+1)*x*t - (i)*v2)/(i+1);
      v1 := t;
    end_for;
    [v2, n*(x*v2-v1)/(x^2-1)];
  end_proc:

// First implementation: Trivial, slow, but works.

interval::legendre_newton_ :=
  proc(x0, n, digits, lo, hi)
    local iv, L, Lprime, w, m, iv1, eps, s;
    save DIGITS;
  begin
    if args(0) < 5 then
      // find a starting region that is both safe (L'<>0)
      // and known to contain a zero
  
      // idea behind the following: we need a small region known
      // to contain exactly one of the roots, which are known to be
      // simple and real.  For positive (negative) first derivative
      // at x0, decrease (increase) lo, starting from x0, by linear
      // interpolation until L_n(hull(lo)) < 0 (> 0) and likewise
      // for hi, such that the intermediate value theorem guarantees
      // a zero.  We use an "overshooting" Newton step.
      proc()
        name `enclosure step`;
      begin
      lo := hull(x0);
      [L, Lprime] := interval::legendre_direct_(n, lo);
      s := sign(Lprime);
      if 0 in Lprime then
        error("initial approx. is too bad");
      end_if;
      case s
        of hull(1) do
          while not L < 0 do // NOTE: this is *not* equiv. to L>=0!!
            lo := hull(op(lo, 1) - 1.3*op(L, 2)/op(Lprime, 1));
            [L, Lprime] := interval::legendre_direct_(n, lo);
          end_while;
          break;
        of hull(-1) do
          while not L > 0 do
            lo := hull(op(lo, 1) - 1.3*op(L, 1)/op(Lprime, 2));
            [L, Lprime] := interval::legendre_direct_(n, lo);
          end_while;
          break;
        otherwise
          assert(FALSE);
      end_case;
      
      hi := hull(x0);
      [L, Lprime] := interval::legendre_direct_(n, hi);
      case s
        of hull(1) do
          while not L > 0 do // NOTE: this is *not* equiv. to L<=0!!
            hi := hull(op(hi, 2) - 1.3*op(L, 1)/op(Lprime, 2));
            [L, Lprime] := interval::legendre_direct_(n, hi);
          end_while;
          break;
        of hull(-1) do
          while not L < 0 do
            hi := hull(op(hi, 1) - 1.3*op(L, 2)/op(Lprime, 1));
            [L, Lprime] := interval::legendre_direct_(n, hi);
          end_while;
          break;
        otherwise
          assert(FALSE);
      end_case;
      end_proc();
    end_if;
    
    iv := (lo...hi);
    assert(hi > lo);
    assert(x0 in iv);
    
    eps := 10.0^(2-digits);
    w := op(iv, 2) - op(iv, 1); // width
    while w > eps do
      m := hull((op(iv,1)+op(iv,2))/2);
      L := interval::legendre_direct_(n, m)[1];
      Lprime := interval::legendre_direct_(n, iv)[2];
      iv := hull(iv intersect m - L/Lprime);
      if iv = {} then
        return({});
      end_if;
      if 0 in Lprime then
        // insufficient convergence, bisect
        // actually, this code will never be reached
        // if the inclusion above is worth its bytes...
//      print("bisect");
        DIGITS := ceil(1.5*DIGITS);
        m := (op(iv,1)+op(iv,2))/2;
        if x0 < m then
          iv1 := interval::legendre_newton_(x0, n, digits, lo, m);
          if iv1 = {} then
//          print("bisect2");
            return(interval::legendre_newton_((m+hi)/2, n, digits, m, hi));
          else
            return(iv1);
          end_if;
        else
          iv1 := interval::legendre_newton_(x0, n, digits, m, hi);
          if iv1 = {} then
//          print("bisect2");
            return(interval::legendre_newton_((lo+m)/2, n, digits, lo, m));
          else
            return(iv1);
          end_if;
        end_if;
      end_if;
      w := op(iv, 2) - op(iv, 1);
    end_while;
    assert(0 in interval::legendre_direct_(n, iv)[1]);
    return(iv);
  end_proc:

interval::legendre_weight_ :=
  proc(iv, n)
    local Lprime;
  begin
    Lprime := interval::legendre_direct_(n, hull(1)*iv)[2];
    2/(1-iv^2)/Lprime^2;
  end_proc:

// question 1: zeroes of L_n(x) and L_{n+1}(x) separate one another,
// and cos((i-1/4)*PI/(n+1/2)) are good approximations to the zeroes
// of L_n(x).  Do they provably separate the zeroes of L_{n+1}(x)?

// note that, unlike numeric::gldata, this routine returns abscissas
// and weights for integration over -1..1.

interval::gldata := 
  proc(n)
    local digits;
    save DIGITS;
  begin
    digits := DIGITS;
    DIGITS := ceil(1.3*DIGITS)+10;
    interval::gldata_(n, digits):
  end_proc:

interval::gldata_ :=
  proc(n, digits)
    save DIGITS;
    option remember;
    local l, w;
  begin
    if digits <= 200 then
      if contains(interval::gldata_saved_, n) then
        return(interval::gldata_saved_[n]);
      end_if;
    end_if;
    if modp(n, 2) = 1 then
      error("interval::gldata not implemented yet for odd n");
    end_if;
    
    DIGITS := max(DIGITS, ceil(n/50+10)); // heuristic

    l := map((numeric::gldata(n, digits))[2], x->2*x-1);
    l := l[n/2+1..-1];
    // Newton iteration
    l := map(l, interval::legendre_newton_, n, digits);
    assert(not has(l, {}));

    // weights
    w := map(l, interval::legendre_weight_, n);
    
    l := map(revert(l), _negate).l;
    w := revert(w).w;
    
    assert(nops(l) = n);
    return([w, l]);
  end_proc:
  
interval::gldata_saved_ := table():

null():

