// 

// validated integration using Gauss-Legendre quadrature

interval::int :=
  proc(fx, xiv, goal_width)
    local f, x, iv, a, b, xm, xr, gldata, i, s, df, dfx, n, Cn,
	  subdiv, width_sum, wid, sum, int_1, int_n;
  begin
    [x, iv] := [op(xiv)];
    [a, b] := map([op(iv)], hull);

    if args(0) < 3 then
      goal_width := 10^(1-DIGITS);
    end_if;
    
    if iszero(a-b) then return(hull(0)); end_if;

    s := 1;
    
    if a > b then
      [a, b] := [b, a];
      s := -1;
    end_if;

    // TODO: a and b are supposed to be *thin* intervals.
    // what should happen if they are not?

    if has([a, b], [RD_NINF, RD_INF]) then
      return(FAIL);
    end_if;
    
    fx := interval(fx);
    f := fp::unapply(fx, x);

    userinfo(5, "Integrating ".expr2text(fx)." over ".expr2text(a...b));
    
    n := 20;
    gldata := interval::gldata(n);
    df := interval::autodiff(fx, 2*n);

    // we store the intervals to be considered in a heap,
    // which is ordered by the width of the remainder term,
    // negative to get the widest one first.
    subdiv := adt::Heap();
    
    dfx := df(a...b);
    /*
       In the remainder formulae, note that dfx[-1] already contains a factor
       1/(2*n)!, which is listed extra in the text books!
     */
    // error term formula:
    // USED AGAIN BELOW!
    
    // listed in Oevel, Einfhrung in die numerische Mathematik:
//    Cn := n!^4/(2*n)!/(2*n+1)!
//	 * (b-a)^(2*n+1) * dfx[-1];
    // for n = 20:
    userinfo(30, "dfx[1] = ".expr2text(dfx[1]).", dfx[-1] = ".expr2text(dfx[-1]));
    Cn := 1/779068285816646714288400 * (b-a)^41 * dfx[-1];

    // TODO: The errors must be relative, so we probably need
    // a running estimate.  Division by op(abs(dfx[1]), 2)/(b-a)
    // does not work, see the difference between 
    // int(exp(-1000/(x^2+1)), x=-1..1) and int(x+exp(-1000/(x^2+1)), x=-1..1)
    width_sum := min(DOM_INTERVAL::width(Cn), 
		     DOM_INTERVAL::width(dfx[1]*(b - a)/2));
    subdiv::insert(-width_sum, [a, b, hull((a+b)/2), dfx, Cn]);
    
    width_sum := hull(width_sum);
    
    while subdiv::nops() < 1000 and not width_sum < goal_width do

      width_sum := width_sum + subdiv::min_pair()[1];
      [a, b, xm, dfx, Cn] := subdiv::delete_min();
      
      userinfo(30, "Looking at ".expr2text(a...b));
      
      if has(width_sum, RD_NAN) or not 0 < width_sum then
	// this operation can be expansive.  OTOH, we have
	// severe cancellation in width_sum
	width_sum := -(_plus(op(map(select(subdiv::H(), _not@has, FAIL), hull@op, 1))));
      end_if;
      
      dfx := df(a...xm);
      userinfo(30, "dfx[1] = ".expr2text(dfx[1]).", dfx[-1] = ".expr2text(dfx[-1]));
      Cn := 1/779068285816646714288400 * (xm-a)^41 * dfx[-1];
      wid := min(DOM_INTERVAL::width(Cn), 
		 DOM_INTERVAL::width(dfx[1]*(xm - a)));
      width_sum := width_sum + hull(wid);
      subdiv::insert(-wid, [a, xm, hull((a+xm)/2), dfx, Cn]);

      dfx := df(xm...b);
      Cn := 1/779068285816646714288400 * (b-xm)^41 * dfx[-1];
      wid := min(DOM_INTERVAL::width(Cn),
		 DOM_INTERVAL::width(dfx[1]*(b-xm)));
      width_sum := width_sum + wid;
      subdiv::insert(-wid, [xm, b, hull((xm+b)/2), dfx, Cn]);
    end_while;

    userinfo(10, expr2text(subdiv::nops())." subintervals used");
    sum := 0;
    while subdiv::nops() > 0 do
      [a, b, xm, dfx, Cn] := subdiv::delete_min();
      assert(xm in a...b);
      xr := hull((b-a)/2);
      int_n := (xr * _plus(gldata[1][i]*
			   (f(xm+xr*gldata[2][i])+
			    f(xm-xr*gldata[2][i]))
			   $ i = 1..n/2)
		+ Cn);
      int_1 := 2 * xr * f(a...b);
//      if DOM_INTERVAL::width(int_1) < DOM_INTERVAL::width(int_n) then
//	print("Deg 1 better than deg n over ".expr2text(a...b));
//	print(int_1, int_n, int_1 intersect int_n);
//      end_if;
      assert(int_1 intersect int_n <> {});
      sum := sum + (int_n intersect int_1);
    end_while;
    s * sum;
  end_proc:
    
/*

 An example where choosing a lower derivative yields a better error
 bound: x*sqrt(x).

 Also note that this is an example where autodiff yields suboptimal results:

>> interval::autodiff(x*sqrt(x), 40)(0.001...1)[-3..-1]                    

[-2.086560026e104 ... 2.086558228e104,

   -8.636990182e106 ... 8.636993198e106,

   -3.549600678e109 ... 3.549601171e109]
>> [interval::evalBiCentered(simplify(diff(x*sqrt(x), x$n)),
                             x=0.001...1)$n=38..40] 
[2.614341441e40 ... 8.267273538e149, -3.017554842e154 ... -9.542346263e41,

   3.578379848e43 ... 1.131583066e159]

*/

/*
  Substitutions from NR:

  If f is known to behave like sqrt(x-a), substitute
>> % = intlib::changevar(%, x=a+t^2, t)

                b            (b - a)^(1/2)
                 /                  /
                |                  |                  2
                |  f(x) dx =       |       2 t f(a + t ) dt
               /                  /
                a                  0

  If f is know to behave like sqrt(b-x), substitute
>> int(f(x), x=a..b) = intlib::changevar(int(f(x), x=a..b), x=b-t^2, t)

               b            (b - a)^(1/2)
                /                  /
               |                  |                    2
               |  f(x) dx =       |       2 t f(b - t ) dt
              /                  /
               a                  0

 If f is known to behave like (x-a)^c, 0<=c<1, substitute t=(x-a)^(1-c) to get

        b                  (b - a)^(1 - c)    c    /    1       \
         /                         /        -----  |  -----     |
        |              1          |         1 - c  |  1 - c     |
        |  f(x) dx = -----        |        t      f\ t      + a / dt
       /             1 - c       /
        a                         0

 Likewise, for f like (b-x)^c, we have

        b                  (b - a)^(1 - c)    c    /        1   \
         /                         /        -----  |      ----- |
        |              1          |         1 - c  |      1 - c |
        |  f(x) dx = -----        |        t      f\ b - t      / dt
       /             1 - c       /
        a                         0


 For f decreasing exponentially, we have

>> I_f = intlib::changevar(I_f, x=-ln(t))

                  infinity           exp(-a)
                       /                 /
                      |                 |    f(-ln(t))
                      |    f(x) dx =    |    --------- dt
                     /                 /         t
                      a                 0
which requires (semi-)symbolic simplification for the (hopefully)
removable singularity at t=0.

 
 From Desktop/doku/Numerics/sinh.pdf, we find

>> I_f = intlib::changevar(I_f, x=ln((1+t)/(1-t)))               

              infinity            1       /   /   t + 1 \ \
                  /                /   2 f| ln| - ----- | |
                 |                |       \   \   t - 1 / /
                 |     f(x) dx =  |  - -------------------- dt
                /                /             2
             -infinity            -1          t  - 1

>> I_f = intlib::changevar(intlib::changevar(I_f, t2=arcsinh(x)),
&>                         t2 = ln((1+t)/(1-t)))

 infinity          
     /             
    |              
    |     f(x) dx =
   /               
-infinity          

     1	        /   /   t + 1 \ \  /     /   /   t + 1 \ \ \
      /	  2 cosh| ln| - ----- | | f| sinh| ln| - ----- | | |
     |	        \   \   t - 1 / /  \     \   \   t - 1 / / /
     |  - -------------------------------------------------- dt
    /	                         2
     -1                         t  - 1

>> I_f = intlib::changevar(intlib::changevar(I_f, t2=sinh(x)),   
&>                         t2 = ln((1+t)/(1-t)))

   infinity            1            /        /   /   t + 1 \ \ \
       /                /        2 f| arcsinh| ln| - ----- | | |
      |                |            \        \   \   t - 1 / / /
      |     f(x) dx =  |  - ----------------------------------------- dt
     /                /                     /   /   t + 1 \2     \1/2
  -infinity            -1   (t - 1) (t + 1) | ln| - ----- |  + 1 |
                                            \   \   t - 1 /      /

But there are very, very many t on the righthand side, leading to
severe dependency effects.

 */

//////////////////////////////////////////////////////////////////////
// samples against numeric::int

/*

>> numeric::int(exp(-t^2), t=-infinity..0)

                               0.8862269255
>> interval::int(exp(-t^2), t=-100000..0) 

                       0.8862269254 ... 0.8862269255
>> numeric::int(exp(-t^2), t=-100000..0)

                                      2
                  numeric::int(exp(- t ), t = -100000..0)
*/

/*

>> numeric::int(1/x^2, x=1e-40..1)

                                /  1                 \
                    numeric::int| --, x = 1.0e-40..1 |
                                |  2                 |
                                \ x                  /
>> interval::int(1/x^2, x=1e-40..1)

                     9.999999999e39 ... 1.000000001e40
*/


/*

semi-symbolic pre-analysis?  |exp(-X*sin(X))| < |exp(-X^2)| for pos. X

better formulas for improper integration!

esp., interval::int(1/x^2, x=0..1) should work, since the transformation
of infinite intervals relies on this!

Better detection of diverging integrals!

Note: numeric::gtdata(n) returns *exact* abscissae and weights, which
we can use directly here -- after scaling, since numeric::gtdata
expects to work over 0..1.

*/
