// 

// utility function: Create a thin interval of lhs/rhs
// of *real* interval
DOM_INTERVAL::_ilhs :=
proc(iv)
begin
  assert(iv::dom = DOM_INTERVAL and op(iv, 0)=hold(hull));
  subsop(iv, 2=op(iv, 1));
end:

DOM_INTERVAL::_irhs :=
proc(iv)
begin
  assert(iv::dom = DOM_INTERVAL and op(iv, 0)=hold(hull));
  subsop(iv, 1=op(iv, 2));
end:

alias(ilhs = DOM_INTERVAL::_ilhs):
alias(irhs = DOM_INTERVAL::_irhs):


// trigonometric functions on DOM_INTERVALs.  These all go
// down to DOM_INTERVAL::internal_sin.

DOM_INTERVAL::sin :=
proc(a)
begin
  a := interval(a);
  if a::dom <> DOM_INTERVAL then
     if a = {} then return(a); end_if;
     return(FAIL);
  end_if;
  if op(a, 0) = hold(_union) then
     _union(map(op(a), DOM_INTERVAL::sin));
  elif op(a, 0) = hold(hull) then // a is real
     DOM_INTERVAL::sin_internal(a);
  else // a is a complex simple interval
     DOM_INTERVAL::sin_internal(Re(a)) * DOM_INTERVAL::cosh(Im(a))
        + I*(DOM_INTERVAL::cos_internal(Re(a)) * DOM_INTERVAL::sinh(Im(a)));
  end_if;
end:

DOM_INTERVAL::cos :=
proc(a)
begin
  a := interval(a);
  if a::dom <> DOM_INTERVAL then
     if a = {} then return(a); end_if;
     return(FAIL);
  end_if;
  if op(a, 0) = hold(_union) then
     _union(map(op(a), DOM_INTERVAL::cos));
  elif op(a, 0) = hold(hull) then // a is real
     DOM_INTERVAL::cos_internal(a);
  else // a is a complex simple interval
     DOM_INTERVAL::cos_internal(Re(a)) * DOM_INTERVAL::cosh(Im(a))
        - I*(DOM_INTERVAL::sin_internal(Re(a)) * DOM_INTERVAL::sinh(Im(a)));
  end_if;
end:

DOM_INTERVAL::cot :=
proc(a)
  local sina, cosa;
begin
  a := interval(a);
  if a::dom <> DOM_INTERVAL then
     if a = {} then return(a); end_if;
     return(FAIL);
  end_if;
  if op(a, 0) = hold(_union) then
     _union(map(op(a), DOM_INTERVAL::cot));
  else
     sina := DOM_INTERVAL::sin(a);
     cosa := DOM_INTERVAL::cos(a);
     cosa/sina;
  end_if;
end_proc:


DOM_INTERVAL::tan :=
proc(a)
  local rea, ima, l, r, x, y, xx, k, critpts;
begin
  a := interval(a);
  if a::dom <> DOM_INTERVAL then
     if a = {} then return(a); end_if;
     return(FAIL);
  end_if;
  if op(a, 0) = hold(_union) then
     _union(map(op(a), DOM_INTERVAL::tan));
  else
    // real?
    if op(a, 0) = hold(hull) then
      // reduce mod PI
      a := a - interval(PI)*ceil(lhs(interval(a/PI-1/2)));
      // check if it encompasses any poles
      if rhs(a) >= float(PI/2) then
        if rhs(a)=RD_INF or lhs(a)=RD_NINF or 
           rhs(a)-lhs(a) > float(PI) then
          return(RD_NINF...RD_INF);
        end;
        // monotonicity on two intervals
        l := ilhs(a);
        r := irhs(a);
        DOM_INTERVAL::sin(a)/DOM_INTERVAL::cos(a) intersect
          (DOM_INTERVAL::sin(l)/DOM_INTERVAL::cos(l)...RD_INF union
          RD_NINF...DOM_INTERVAL::sin(r)/DOM_INTERVAL::cos(r));
      else
        // use monotonicity
        l := ilhs(a);
        r := irhs(a);
        DOM_INTERVAL::sin(l)/DOM_INTERVAL::cos(l)...
        DOM_INTERVAL::sin(r)/DOM_INTERVAL::cos(r);
      end;
    else
      // complex interval.
      rea := Re(a);
      ima := Im(a);
      if has(rea, {RD_NINF, RD_INF}) then
        return(DOM_INTERVAL::sin(a)/DOM_INTERVAL::cos(a));
      end;
      // reduce mod PI
      rea := rea - interval(PI)*ceil(lhs(interval(rea/PI-1/2)));
      assert(-PI/2<=lhs(rea));
      // due to rounding errors, we may not have nailed it on the first try:
      while 3/2*PI < lhs(rea) do
        rea := rea - interval(PI)*ceil(lhs(interval(rea/PI-1/2)));
      end_while;
      assert(-PI/2<=lhs(rea));
      // check if it encompasses any poles
      if 0 in ima and rhs(rea) > PI/2 then
        return((RD_NINF+RD_NINF*I)...(RD_INF+RD_INF*I));
      end;
      // no poles.
      // reduce wide real parts, using periodicity
      x := ceil(rhs(interval(rea/PI))) - 2;
      while x <> FAIL and x>0 do
        x := subsop(rea, 2=op(rea, 2)-interval(x*PI));
        if x <> FAIL then
          rea := x;
          x := ceil(rhs(interval(rea/PI))) - 2;
        end_if;
      end_while;
      x := FAIL;
      // Fall back to our implementations for real intervals:
      critpts := {ilhs(rea)+I*ilhs(ima),
                  ilhs(rea)+I*irhs(ima),
                  irhs(rea)+I*ilhs(ima),
                  irhs(rea)+I*irhs(ima)};
      // The extremal values are on the
      // curves where tan(x)^2=coth(y)^2
      // or x=k*PI/2.
      y := ilhs(ima);
      x := DOM_INTERVAL::arctan(DOM_INTERVAL::coth(y));
      for xx in [x-PI, x, x+PI,
                 -x, PI-x] do // these are sufficient by periodicity
        if xx in rea then critpts := critpts union {xx+I*y}; end_if;
      end_for;
      y := irhs(ima);
      x := DOM_INTERVAL::arctan(DOM_INTERVAL::coth(y));
      for xx in [x-PI, x, x+PI,
                 -x, PI-x] do
        if xx in rea then critpts := critpts union {xx+I*y}; end_if;
      end_for;
      x := ilhs(rea);
      y := DOM_INTERVAL::arccoth(DOM_INTERVAL::tan(x));
      if y in ima then critpts := critpts union {x+I*y}; end_if;
      if -y in ima then critpts := critpts union {x-I*y}; end_if;
      x := irhs(rea);
      y := DOM_INTERVAL::arccoth(DOM_INTERVAL::tan(x));
      if y in ima then critpts := critpts union {x+I*y}; end_if;
      if -y in ima then critpts := critpts union {x-I*y}; end_if;
      critpts := critpts union
               {(interval(k*PI/2+I*ilhs(ima)),
                 interval(k*PI/2+I*irhs(ima)))
                $ k = ceil(2/PI*lhs(rea)) .. floor(2/PI*rhs(rea))};
      if 0 in ima then
        critpts := critpts union {ilhs(rea), irhs(rea)};
      end_if;
      hull(map(critpts,
               DOM_INTERVAL::sin/DOM_INTERVAL::cos));
    end_if;
  end_if;
end_proc:

// The inverse functions all use the kernel routine DOM_INTERVAL::arctan_internal

DOM_INTERVAL::arctan :=
proc(a)
begin
  a := interval(a);
  if a::dom <> DOM_INTERVAL then
     if a = {} then return(a); end_if;
     return(FAIL);
  end_if;

  if op(a, 0) = hold(_union) then
     _union(map(op(a), DOM_INTERVAL::arctan));
  elif op(a,0) = FAIL then // complex interval
     // TODO: Improve the overestimation
     I/2 * DOM_INTERVAL::ln(1-I*a)
      - I/2 * DOM_INTERVAL::ln(1+I*a);
  else // real interval
     DOM_INTERVAL::arctan_internal(a);
  end_if;
end_proc:

DOM_INTERVAL::arccot :=
proc(a)
begin
  a := interval(a);
  if a::dom <> DOM_INTERVAL then
     if a = {} then return(a); end_if;
     return(FAIL);
  end_if;
  DOM_INTERVAL::arctan(1/a);
end_proc:


DOM_INTERVAL::arcsin :=
proc(a)
  local sqra;
begin
  a := interval(a);
  if a::dom <> DOM_INTERVAL then
     if a = {} then return(a); end_if;
     return(FAIL);
  end_if;

  if op(a, 0) = hold(_union) then
     _union(map(op(a), DOM_INTERVAL::arcsin));
  elif op(Re(a), 1) < 0 then
    DOM_INTERVAL::arcsin(a intersect ((0...RD_INF)+(RD_NINF...RD_INF)*I))
    union
    -DOM_INTERVAL::arcsin(-a intersect ((0...RD_INF)+(RD_NINF...RD_INF)*I))
  else // Re(a) >= 0
    sqra := sqrt(1-a^2);
//    ((hull(PI/2) -
//      DOM_INTERVAL::arctan(sqra/a))
//     union if 0 in a then hull(0) else {} end_if)
//    intersect
    (2*DOM_INTERVAL::arctan((1-sqra)/a))
    intersect -I*DOM_INTERVAL::ln(I*a + sqra);
  end_if;
end_proc:

DOM_INTERVAL::arccos := x -> (hull(PI/2) - DOM_INTERVAL::arcsin(x)):


// To avoid errors for intervals containing zero:
DOM_INTERVAL::arccsc := x -> DOM_INTERVAL::arcsin(1/x):
DOM_INTERVAL::arcsec := x -> DOM_INTERVAL::arccos(1/x):

DOM_INTERVAL::csc := x -> 1/DOM_INTERVAL::sin(x):
DOM_INTERVAL::sec := x -> 1/DOM_INTERVAL::cos(x):
