// 

// interval::newton performs a simple Newton iteration
// given expressions and initial starting conditions

// Actually, this code is implemented using the Hansen-Sengupta operator

/* TODO: merge intervals caused by bisecting.  Example:

>> interval::newton([x^(1/3)], [x=-1...1])

[x in -0.007812500001 ... 1.875958165e-2525222],

   [x in -1.875958165e-2525222 ... 0.007812500001]

*/

interval::newton_rand := frandom(1234):

interval::newton :=
proc(fs, start, eps=1e-2)
  local jacobian, vars, regions;
begin
  if domtype(fs) <> DOM_LIST and domtype(start) <> DOM_LIST then
    fs := [fs];
    start := [start];
  end_if;
  
  if testargs() then
    if not testtype(fs, DOM_LIST) or 
       not testtype(start, DOM_LIST) or
       not nops(fs) = nops(start) then
      error("expecting two lists of the same size");
    end_if;
    
    if not testtype(float(eps), DOM_FLOAT) or
       not float(eps)>0 then
      error("Precision must be >0");
    end_if;
    
    if map({op(start)}, type) minus {"_equal", "_in"} <> {} then
      error("Starting regions must be given as [var=iv, var=iv, ...]");
    end_if;
    regions := map(start, op, 2);

    if map({op(regions)}, domtype) <> {DOM_INTERVAL} then
      error("starting regions must be intervals");
    end_if;

    if min(op(map(regions, DOM_INTERVAL::width))) = RD_INF then
      error("all starting intervals are infinite");
    end_if;
  end_if;

  vars := map(start, op, 1);
  jacobian := linalg::jacobian(fs, vars);
  if has(jacobian, [hold(diff), hold(D)]) then
    return(FAIL);
  end_if;
  regions := map(start, op, 2);
  
  sysassign(interval::newton_rand, frandom(1234)):

  map([interval::newton_int(fs, regions, vars, jacobian, eps)],
      s -> zip(vars, s, _in));
end_proc:
  
interval::newton_int :=
proc(fs, regions, vars, jacobian, eps)
  local last_regions, cur_f, cur_jacobian,
        midpoint, union_index, subproblems, widths, i, 
        bisected, last_widths, bisect_at, gs;
  save DIGITS;
begin

  last_regions := FAIL;
  widths := FAIL;
  bisect_at := FAIL;
  
  if not _lazy_and(op(zip([0$nops(fs)],
                          interval::evalSimple(fs,
                                               zip(vars, regions, `=`)),
                          _in))) then
    return();
  end_if;
  
  while contains(regions, {}) = 0 and
	regions <> last_regions do
    last_regions := regions;
    cur_jacobian := evalAt(jacobian, zip(vars, regions, `=`));
    midpoint := map(regions, DOM_INTERVAL::center);
    cur_f := evalAt(fs, zip(vars, interval(midpoint), `=`));
    gs := interval::GaussSeidel(cur_jacobian,
                                map(cur_f, eval@`-`),
                                zip(regions, midpoint,
                                    _subtract));
    if gs=FAIL then 
      return(null());
    end_if;
    regions := zip(eval(zip(interval(midpoint), gs, `+`)),
                   regions, _intersect);
    if ((union_index :=
         contains(map(regions, op, 0), hold(_union)))) > 0 then
      subproblems := map([op(regions[union_index])],
                         x->[op(regions, 1..(union_index-1)),
                             x,
                             op(regions, (union_index+1)..nops(regions))]);
      subproblems := map(subproblems,
                         x->interval::newton_int(fs, x, vars, jacobian, eps));
      subproblems := select(subproblems,
                            x->nops(x)>0);
      return(op(subproblems));
    end;
    
    last_widths := widths;
    widths := map(regions, DOM_INTERVAL::width);
    if max(op(widths)) < eps then
      break;
    end;
    
  end_while;

  if contains(regions, {}) = 0 and
     max(op(widths)) >= eps then
    // bisect
    if bisect_at = FAIL then
      bisect_at := 1;
      for i from 1 to nops(regions) do
        if DOM_INTERVAL::width(regions[i]) >
           DOM_INTERVAL::width(regions[bisect_at]) then
          bisect_at := i;
        end_if;
      end_for;
    end_if;
    userinfo(2, "bisecting in direction ".expr2text(bisect_at));
    bisected := interval::_newton_bisect(regions[bisect_at]);
    regions := [interval::newton_int(fs,
                                     [op(regions, 1..bisect_at-1),
                                      bisected[1],
                                      op(regions, bisect_at+1..nops(regions))],
                                     vars, jacobian, eps),
                interval::newton_int(fs,
                                     [op(regions, 1..bisect_at-1),
                                      bisected[2],
                                      op(regions, bisect_at+1..nops(regions))],
                                     vars, jacobian, eps)];
    return(op(select(regions, `<>`, [])));
  end_if;
  
  if contains(regions, {}) > 0 then
    null();
  else
    regions;
  end_if;
end_proc:

interval::_newton_bisect :=
proc(iv)
  local Re_iv, Im_iv, c, r;
begin
  Re_iv := Re(iv);
  Im_iv := Im(iv);
  r := interval::newton_rand();
  if iszero(Re_iv) then
    c := r*lhs(Im_iv) + (1-r)*rhs(Im_iv);
    return([subsop(iv, [2,2]=c), subsop(iv, [2,1]=c)]);
  elif iszero(Im_iv) then
    c := r*lhs(Re_iv) + (1-r)*rhs(Re_iv);
    return([subsop(iv, 2=c), subsop(iv, 1=c)]);
  elif DOM_INTERVAL::width(Re_iv) > DOM_INTERVAL::width(Im_iv) then
    c := r*lhs(Re_iv) + (1-r)*rhs(Re_iv);
    return([subsop(iv, [1,2]=c), subsop(iv, [1,1]=c)]);
  else
    c := r*lhs(Im_iv) + (1-r)*rhs(Im_iv);
    return([subsop(iv, [2,2]=c), subsop(iv, [2,1]=c)]);
  end_if;
end_proc:
