// psi, the logarithmic derivative of gamma.


// formulae based on Spouge, "Computation of the Gamma, Digamma,
// and Trigamma Functions", SIAM Journal of Numerical Analysis,
// Vol 31, No. 3 (June 1994), 931-944

DOM_INTERVAL::_psi_spouge_coeffs :=
proc(a, prec)
  option remember;
  local N, k, fac, c;
begin
  fac := hull(1/sqrt(2*PI));
  N := ceil(a)-1;
  c := [1$k=0..N];
  for k from 1 to N do
    c[k+1] := fac * hull(a-k)^(k-1/2)*DOM_INTERVAL::exp(a-k);
    fac := -fac/k;
  end_for;
  c;
end_proc:

DOM_INTERVAL::_psi_spouge :=
proc(z)
  local a, d, N, c, f1, f2, k, eps, errorfact;
begin
  a := 1.3*DIGITS;
  d := interval(1/(1-sqrt(2/3)/(2*PI)^2));
  while (eps := interval(d*ln(2*a)/((2*PI)^(a+1/2)*(Re(z)+a))))
	> 10.0^(3-DIGITS) do
    a := 2*a+1;
  end_while;
  N := ceil(a)-1;
  c := DOM_INTERVAL::_psi_spouge_coeffs(a, DIGITS);
  
  eps := interval(1/(1-sqrt(2/3)/(2*PI)^2)*ln(2*a)*
		  1/((2*PI)^(a+1/2)*(Re(z)+a)));
  if iszero(Im(z)) then
    errorfact := -1...1;
  else
    errorfact := -1-I...1+I;
  end_if;
  f1 := c[1] + _plus(c[k+1]/(z+k)$k=1..N) +
      // error bound
      errorfact*interval(1.022*ln(2*a)/sqrt(a)*eps);
  f2 := -_plus(c[k+1]/(z+k)^2$k=1..N) +
      // error bound
      errorfact*interval(sqrt(a)*eps);
  
  interval(ln(z+a)-(a-1/2)/(z+a)+f2/f1);
end_proc:

// for real arguments, psi is monotoneous,
// with simple poles at nonpositive integers:

// for real intervals, use monotonicity
DOM_INTERVAL::_psi_real :=
proc(iv)
  local k, l, r, psi_neg, psi_pos;
begin
  psi_pos :=
  (s -> DOM_INTERVAL::_psi_spouge(s+5)-_plus(1/(s+k) $ k = 0..5));
  // psi(z) = psi(1-z)+PI*cot(PI*(1-z))
  psi_neg :=
  (s -> psi_pos(1-s) +
       interval(PI)*cot(interval(PI)*(1-s)));
  
  [l, r] := [op(iv)];
  if r <= 0 then
    if r = l and iszero(frac(r)) then
      // only a singular point (nonpos. integer)
      return({});
    end_if;
    if r > ceil(l) and l < ceil(l) then
      // negative integer inside the interval
      if r >= l+1 then
        RD_NINF...RD_INF
      else
        psi_neg(subsop(iv, 2=l)) ... RD_INF
        union
        RD_NINF...psi_neg(subsop(iv, 1=r));
      end_if;
    elif iszero(r - ceil(l)) then
      psi_neg(subsop(iv, 2=l)) ... RD_INF
    elif iszero(l - floor(r)) then
      RD_NINF...psi_neg(1-subsop(iv, 1=r));
    else
      // r and l negative, but no integer between them
      psi_neg(subsop(iv, 2=l))
      ...
      psi_neg(subsop(iv, 1=r));
    end_if;
  else // r >0
    if l < 0 then
      psi_neg(subsop(iv, 2=l)) ... RD_INF
      union
      RD_NINF...psi_pos(subsop(iv, 1=r), DIGITS);
    elif iszero(l) then
      RD_NINF...psi_pos(subsop(iv, 1=r), DIGITS);
    else
      psi_pos(subsop(iv, 2=l), DIGITS)
      ...
      psi_pos(subsop(iv, 1=r), DIGITS);
    end_if
  end_if
end_proc:

//////////////////////////////////////////////////


DOM_INTERVAL::psi :=
proc(iv)
  local k;
begin
  iv := interval(iv);
  if iv::dom <> DOM_INTERVAL then
    if iv = {} then return(iv); end_if;
    return(FAIL);
  end_if;
  
  if op(iv,0) = hold(_union) then
    return(_union(map(op(iv), DOM_INTERVAL::psi)));
  elif op(iv, 0) = hold(hull) then // real interval
    return(DOM_INTERVAL::_psi_real(iv));
  end_if;
  
  if Re(iv) intersect RD_NINF...0 <> {} and
     not iszero(op(Re(iv), 1)) then
    return(DOM_INTERVAL::psi((1-iv) intersect ((RD_NINF*hull(I)) ... RD_INF+RD_INF*I))
	   + interval(PI)*cot(interval(PI*(1-iv)))
	   union
	   DOM_INTERVAL::psi(iv intersect ((RD_NINF*hull(I)) ... RD_INF+RD_INF*I)));
  end_if;
  
  // Spouge formulae
  // use recurrence formula to *increase* the argument
  // for better error bounds, esp. in [0,2]
  // Note that _psi_spouge gets z-1 to calculate psi(z).
  DOM_INTERVAL::_psi_spouge(iv+5)-_plus(1/(iv+k) $ k = 0..5);
end_proc:
