// Rothstein's SPDE algorithm

// Given a derivation d on k[t], an integer n and a,b,c in k[t] with a<>0,
// return either [FAIL,FAIL,FAIL,FAIL,FAIL], in which case the equation
// a*d(q)+b*q=c has no solution of degree at most n in k[t], or the tuple
// [b',c',m,aa,bb] such that a',b',aa,bb in k[t], m in Z_, and any solution 
// q in k[t] of degree at most n of a*d(q)+b*q=c is of the form
// q=aa*h+b, where h in k[t], degree(h)<=m, and d(h)+b'*h=c'.

// Follows Bronstein, Symbolic Integration I, p.203

intlib::algebraic::rde::SPDE :=
proc(a, b, c, ts, diffs, algs, n)
  local t, g, r, z, bbar, cbar, m, aa, bb, d;
begin
  t := ts[-1];
  if n<0 then
    if iszero(c) then
      return([poly(0,[t]), poly(0,[t]), 0, poly(0, [t]), poly(0, [t])]);
    else
      return([FAIL$5]);
    end_if;
  end_if;
  g := gcd(a, b);
  if not iszero(mapcoeffs(divide(c, g, Rem), normal)) then
    return([FAIL$5])
  end_if;
  a := intlib::algebraic::polyDivide(a, g);
  b := intlib::algebraic::polyDivide(b, g);
  c := intlib::algebraic::polyDivide(c, g);
  if iszero(degree(a)) then
    return([multcoeffs(b, 1/tcoeff(a)), multcoeffs(c, 1/tcoeff(a)), 
            n, poly(1,[t]), poly(0,[t])]);
  end_if;
  // b*r+a*z=c, degree(r) < degree(a):
  [r, z] := intlib::algebraic::extendedEuclidean(b, a, c);
  r := mapcoeffs(r, normal);
  z := mapcoeffs(z, normal);
  d := intlib::algebraic::diff(ts, diffs, algs);
  [bbar, cbar, m, aa, bb] :=
   intlib::algebraic::rde::SPDE(a, b+d(a), z-d(r), ts, diffs, algs, n-degree(a));
  if bbar = FAIL then
    return([FAIL$5]);
  end_if;
  return([bbar, cbar, m, a*aa,a*bb+r]);
end:
