// Normal part of the denominator of the solution of an RDE.

// Given a derivation D on k[t] and f,g in k(t) with f weakly
// normalized w.r.t. t, return either FAIL, in which case Dy+f*y=g
// has no solution in k(t), or the list [a,b,c,h] such that
// a, h, in k[t], b, c, in k<t>, and for any solution y in k(t)
// of Dy+f*y=g, q=y*h in k<t> satisfies a*Dq+b*q=c.
//
// Follows Bronstein, Symbolic Integration I, p. 185.

intlib::algebraic::rde::normalDenominator :=
proc(fNum, fDen, gNum, gDen, ts, diffs, algs)
  local dn, ds, en, es, p, h, bNum, bDen, cNum, cDen, tmp;
begin
  [dn, ds] := intlib::algebraic::splitFactor(fDen, ts, diffs, algs);
  [en, es] := intlib::algebraic::splitFactor(gDen, ts, diffs, algs);
  p := gcd(dn, en);
  h := divide(gcd(en, en'), gcd(p, p'), Quo); // Exact

  // this was:
  //  if iszero(mapcoeffs(divide(dn*h^2, en, Rem), normal, Rationalize=None)) then
  // but we aim to avoid the normal and to bail out early:
  tmp := TRUE;
  proc()
    local ex, exv, env, vars, i, rem, c;
    save SEED;
  begin
    ex := dn*h^2;
    vars := freeIndets({coeff(ex), coeff(en)});
    SEED := 76543;
    if nops(vars) > 0 then
      for i from 1 to 10 + max(map({coeff(ex), coeff(en)}, e -> degree(numer(e)))) do
        traperror((
          [exv, env] := subs([ex, en],
            map(vars, v -> v=random()));
          rem := divide(exv, env, Rem);
          rem := [coeff(rem)];
          for c in rem do
            if testeq(c, 0, Steps=1) = FALSE then
              tmp := FAIL;
              break;
            end_if;
          end_for;
        ));
        if tmp = FAIL then return(); end_if;
      end_for;
    end_if;
    if not iszero(mapcoeffs(divide(dn*h^2, en, Rem), normal)) then
      tmp := FAIL;
    end_if;
  end_proc();

  if tmp = FAIL then
    return([FAIL$4]);
  end_if;
  
  // b = dn*h*f-dn*d(h) = (h*fNum/ds - fDen/ds*d(h)) = (h*fNum - fDen*d(h))/ds
  bNum := (h*fNum - intlib::algebraic::diff(ts, diffs, algs)(h)*fDen);
  bDen := ds;
  tmp := gcd(bNum, bDen);
  bNum := mapcoeffs(divide(bNum, tmp, Quo), normal); // Exact
  bDen := mapcoeffs(divide(bDen, tmp, Quo), normal); // Exact
  
  // c = dn*h^2*g
  tmp := gcd(dn*h^2, gDen);
///  cNum := divide(dn*h^2, tmp, Quo)*gNum; // Exact
/// We've observed large intermediate expression swell here,
/// which seems to be kept under control by temporarily using
/// excessive normalization.
  cNum := divide(poly(dn*h^2, [ts[-1]], Dom::ExpressionField(normal, iszero)),
    poly(tmp, [ts[-1]], Dom::ExpressionField(normal, iszero)), Quo);
  cNum := poly(cNum, [ts[-1]], Expr)*gNum;
  cDen := divide(gDen, tmp, Quo); // Exact
  if degree(cDen) > 0 then
    [cNum, cDen] := intlib::algebraic::normal(expr(cNum)/expr(cDen), [ts[-1]]);
  end_if;
  [dn*h, [bNum, bDen], [cNum, cDen], h];
end:
