// parametric logarithmic derivative **heuristic**

// Given a derivation d on k[t], f in k(t) and a hyperexponential
// monomial theta over k(t), solve n*f = d(v)/v + m*d(th)/th
// for n, m, and v. Returns [n,m,v] if found, [FAIL, TRUE, FAIL]
// if there is definitely no solution and [FAIL, FAIL, FAIL] if it
// cannot decide whether there is one.

// Follows Bronstein, Symbolic Integration I, p. 253

// Note: theta is given implicitly by being ts[-1],
// fNum and fDen are expected to be polynomials in ts[-2]!

intlib::algebraic::rde::parametricLogarithmicDerivative :=
proc(fNum, fDen, ts, diffs, algs)
  local theta, w, p, a, q, b, B, C, s, M, N, Q, l, ln, ls, z,
        u1, r1, u2, r2, v, c, gc, lf, lw, i, m;
begin
  assert(op(fNum, 2)=[ts[-2]]);
  assert(op(fDen, 2)=[ts[-2]]);
  theta := ts[-1];
  assert(not has(fNum, theta));
  assert(not has(fDen, theta));
  w := intlib::algebraic::normal(diffs[-1]/theta, [theta]);
  assert(degree(w[1])=0);
  assert(degree(w[2])=0);
  w := intlib::algebraic::normal(expr(w[1])/expr(w[2]), [ts[-2]]);
  assert(op(fNum, 2)=op(w[1], 2));
  [p, a] := [divide(fNum, fDen)];
  [q, b] := [divide(w[1], w[2])];
  B := max(0, degree(diffs[-2], [ts[-2]])-1);
  C := max(degree(p), degree(q));
  
  if degree(q)>B then
    c := solvelib::getIdent(Any, indets([ts, diffs, algs, op(fNum), op(fDen)]));
    s := p-multcoeffs(q, c);
    s := intlib::algebraic::ratlinsolve([coeff(s, i)$i=B+1..C], [c]);
    s := op(s, [1,2]);
    if iszero(s) or s=FAIL then return([FAIL, TRUE, FAIL]); end_if;
    assert(testtype(s, Type::Rational));
    M := numer(s);
    N := denom(s);
    // w := N*f-M*w
    gc := gcd(fNum, w[2]);
    fNum := intlib::algebraic::polyDivide(multcoeffs(fNum, N), gc);
    w[2] := intlib::algebraic::polyDivide(w[2], gc);
    gc := gcd(fDen, w[1]);
    fDen := intlib::algebraic::polyDivide(fDen, gc);
    w[2] := intlib::algebraic::polyDivide(multcoeffs(w[2], -M), gc);
    w := [fNum*w[2]+fDen*w[1], fDen*w[2]];
    [v, Q] := intlib::algebraic::radicalLogarithmicDerivative(w[1], w[2], 
      ts[1..-2], diffs[1..-2], algs[1..-2]);
    if Q=FAIL then
      return([FAIL, TRUE, FAIL]);
    else
      return([Q*N, Q*M, v]);
    end_if;
  end_if;
  
  if degree(p) > B then return([FAIL, TRUE, FAIL]); end_if;
  
  l := lcm(fDen, w[2]);
  [ln, ls] := intlib::algebraic::splitFactor(l, ts[1..-2], diffs[1..-2], algs[1..-2]);
  assert(op(ln, 2)=[ts[-2]]);
  z := ls * gcd(ln, ln');
  if iszero(intlib::algebraic::diff(ts, diffs, algs)(z)) then
    // The code in this branch is not in Bronstein's book,
    // but follows a strategy sketched very briefly on p. 255.
    
    if degree(fNum)=0 and degree(fDen)=0 and
      degree(w[1])=0 and degree(w[2])=0 and
      not has(diffs[-2], ts[-2]) then
      // ts[-2] is a primitive monomial not appearing in the equation.
      // Thus, the equation has a solution in k(t) iff it has a solution
      // in k. We just have to make sure we don't drop our initial
      // primitive from the list:
      if nops(ts)=2 then
        // this implies d(v)=0.
        m := expr(fNum)/expr(fDen)*expr(w[2])/expr(w[1]);
        if testtype(m, Type::Rational) then
          return([denom(m), numer(m), 0]);
        end_if;
      else
        delete ts[-2], diffs[-2], algs[-2];
        return(intlib::algebraic::rde::parametricLogarithmicDerivative
          (op(intlib::algebraic::normal(expr(fNum)/expr(fDen), [ts[-2]])),
           ts, diffs, algs));
      end_if;
    end_if;
    // heuristic failed.
    // before giving up, try some small values for m and n
    s := expr(fNum)/expr(fDen);
    w := expr(w[1])/expr(w[2]);
    v := FAIL;
    traperror((
      for M in [0, 1, -1, 2, -2/* , 1/2, -1/2 */] do
        v := intlib::algebraic::logarithmicDerivative(
          op(intlib::algebraic::normal(s-M*w, [ts[-2]])),
          ts[1..-2], diffs[1..-2], algs[1..-2]);
        if v <> FAIL then
          break;
        end_if;
      end_for;
    ));
    if v <> FAIL then
      return([denom(M), numer(M), v]);
    end_if;
    return([FAIL, FAIL, FAIL]);
  end_if;
  
  // lf := l*f, lw := l*w
  lf := intlib::algebraic::polyDivide(l, fDen) * fNum;
  lw := intlib::algebraic::polyDivide(l, w[2]) * w[1];
  [u1, r1] := [divide(lf, z)];
  [u2, r2] := [divide(lw, z)];
  c := solvelib::getIdent(Any, indets([ts, diffs, algs, expr(r1), expr(r2)]));
  s := r1-multcoeffs(r2, c);
  s := intlib::algebraic::ratlinsolve([coeff(s, i)$i=0..degree(z)], [c]);
  if s = FAIL then
    return([FAIL, TRUE, FAIL]);
  end_if;
  s := op(s, [1,2]);
  M := numer(s);
  N := denom(s);
  // w := N*f-M*w
///  gc := gcd(fNum, w[2]);
///  fNum := intlib::algebraic::polyDivide(multcoeffs(fNum, N), gc);
///  w[2] := intlib::algebraic::polyDivide(w[2], gc);
///  gc := gcd(fDen, w[1]);
///  fDen := intlib::algebraic::polyDivide(fDen, gc);
///  w[2] := intlib::algebraic::polyDivide(multcoeffs(w[2], -M), gc);
///  w := [fNum*w[2]+fDen*w[1], fDen*w[2]];
///  gc := gcd(w[1], w[2]);
///  w := map(w, intlib::algebraic::polyDivide, gc);
  w := intlib::algebraic::normal(N*expr(fNum)/expr(fDen)-M*expr(w[1])/expr(w[2]), [ts[-2]]);
  [v, Q] := intlib::algebraic::radicalLogarithmicDerivative(w[1], w[2], 
    ts[1..-2], diffs[1..-2], algs[1..-2]);
  if Q=FAIL then
    return([FAIL, TRUE, FAIL]);
  else
    return([Q*N, Q*M, v]);
  end_if;
end:
