// Parametric SPDE algorithm

// Given a derivation d on k[t], an integer n and a,b,q1,...,qm in k[t], 
// a <> 0, return the tuple [a', b',[q1',...,qm'],[r1,...,rm],M,n'] 
// such that for any solution c1,...,cm in Const(k) and q in k[t], 
// degree(q)<=n, of a*D(q)+b*q=sum(c[i]*q[i],i=1..m), M*vector(c)=0 and
// p=(q-sum(c[i]*r[i]))/a' is in k[t], has degree at most n' and satisfies 
// D(p)+b'*p = sum(c[i]*q'[i])

// Follows Bronstein, Symbolic Integration I, p.230 ff.

intlib::algebraic::rde::parametricSPDE :=
proc(a : DOM_POLY, b : DOM_POLY, q : DOM_LIST, ts : DOM_LIST, diffs : DOM_LIST, algs : DOM_LIST, n : DOM_INT)
  local M, M2, t, g, r, z, bbar, cbar, m, aa, bb, d, v, i;
begin
  t := ts[-1];
  M := table(0);
  M["m"] := nops(q);
  M["cols"] := nops(q);
  M["rows"] := 0;
  if n<0 then
    // ensure c[i]=0
    M["rows"] := nops(q);
    for i from 1 to nops(q) do
      M[i, i] := 1;
    end_for;
    return([poly(1, [t]), poly(0, [t]), q, [poly(0, [t])$nops(q)], M, 0]);
  end_if;
  g := gcd(a, b);
  if degree(g) > 0 then
    a := intlib::algebraic::polyDivide(a, g);
    b := intlib::algebraic::polyDivide(b, g);
    q := map(q, qi -> intlib::algebraic::normal(expr(qi)/expr(g), [ts[-1]]));
    [q, M2] := intlib::algebraic::rde::linearConstraints(a, b, ts, diffs, algs, q);
    [M2, v] := intlib::algebraic::rde::constantSystem(M2, [0$M2["rows"]], ts, diffs, algs);
    M := intlib::algebraic::rde::combineHashMatrices(M, M2);
    q := map(q, mapcoeffs, normal, Rationalize=(x -> (x, {})));
  end_if;
  if iszero(degree(a)) then
    return([poly(1, [t]), multcoeffs(b, 1/tcoeff(a)), 
            map(q, multcoeffs, 1/tcoeff(a)),
            [poly(0, [t])$nops(q)], M, n]);
  end_if;
  // b*r[i]+a*z[i]=q[i], degree(r[i]) < degree(a):
  r := map(q, qi -> intlib::algebraic::extendedEuclidean(b, a, qi));
  z := map(r, op, 2);
  r := map(r, op, 1);
  r := map(r, mapcoeffs, normal, Rationalize=(x -> (x, {})));
  z := map(z, mapcoeffs, normal, Rationalize=(x -> (x, {})));
  d := intlib::algebraic::diff(ts, diffs, algs);
  [aa, bbar, cbar, bb, M2, m] :=
    intlib::algebraic::rde::parametricSPDE(a, b+d(a),
      zip(z, r, (zi, ri) -> zi-d(ri)), ts, diffs, algs, n-degree(a));
  if bbar = FAIL then
    return([FAIL$6]);
  end_if;
  M := intlib::algebraic::rde::combineHashMatrices(M, M2);
  return([a*aa, bbar, cbar, zip(bb, r, (bi, ri) -> a*bi+ri), M, m]);
end:
