// polynomial RDE, cancellation -- hyperexponential case

// Given a derivation D on k[t] with D(t)/t in k, 
// n either an integer or infinity,
// b in k and c in k[t],
// return either FAIL, in which case the equation
// D(q)+b*q=c has no solution of degree at most n in k[t],
// or a solution q in k[t] of this equation with degree(q) <= n.

// Follows Bronstein, Symbolic Integration I, p. 213

intlib::algebraic::rde::polyRDECancelExp :=
proc(b, c, ts, diffs, algs, n)
  local z, m, p, q, s, mm, czt, gc, dtt, bmdt, lcc, bb;
begin
  assert(nops(ts)>1);
  assert(not has(expr(b), [ts[-1]]));
  if iszero(c) then return(poly(0, [ts[-1]])); end_if;
  if iszero(b) then
    q := intlib::algebraic::inField(c, poly(1, [ts[-1]]), ts, diffs, algs);
    q := intlib::algebraic::normal(q, [ts[-1]]);
    if q[1]=FAIL or degree(q[2])>0 or degree(q[1])>n then
      return(FAIL);
    end_if;
    return(q[1]/lcoeff(q[2]));
  end_if;
  
  // check for constant solutions
  if iszero(degree(c)) then
    bb := intlib::algebraic::normal(expr(b)/expr(c), ts);
    if iszero(degree(bb[1])) and iszero(degree(bb[2])) then
      return(poly(expr(bb[1])/expr(bb[2]), [ts[-1]]));
    end_if;
  end_if;

  // check whether there are z in k* and m in Z_ such that
  // b=D(z)/z+m*D(t)/t
  bb := intlib::algebraic::normal(expr(b), [ts[-2]]);
  [mm, m, z] := intlib::algebraic::rde::parametricLogarithmicDerivative(bb[1], bb[2], ts, diffs, algs);
  if mm=1 and not iszero(z) then
    // k<t> = k[t,1/t]
    czt := intlib::algebraic::normal(expr(c)*z*ts[-1]^m, [ts[-1]]);
    p := intlib::algebraic::inField(czt[1], czt[2], ts, diffs, algs);
    p := intlib::algebraic::normal(p, [ts[-1]]);
    if p[1] <> FAIL and degree(p[1])+degree(p[2])-m>=0 then
      // q := p/(z*t^m) in k[t]?
      q := intlib::algebraic::normal(expr(p[1])/expr(p[2])/(z*ts[-1]^m), [ts[-1]]);
      if degree(q[2])>0 then return(FAIL); end_if;
      // deg(q)<=n?
      q := multcoeffs(q[1], 1/lcoeff(q[2]));
      if degree(q) <= n then
        return(q);
      else
        return(FAIL);
      end_if;
    end_if;
    return(FAIL);
  end_if;
  
  // no. Keep reducing degree bound and degree(c) until 0 or non-solvability proved.
  q := poly(0, [ts[-1]]);
  dtt := intlib::algebraic::normal(normal(diffs[-1]/ts[-1]), [ts[-2]]);
  assert(not has(dtt, [ts[-1]]));
  while not iszero(c) do
    m := degree(c);
    if n<m then return(FAIL); end_if;
    // bmdt := b+m*D(t)/t
    bmdt := [bb[1]*dtt[2]+multcoeffs(dtt[1]*bb[2], m), bb[2]*dtt[2]];
    gc := gcd(bmdt[1], bmdt[2]);
    bmdt := map(bmdt, intlib::algebraic::polyDivide, gc);
    lcc := intlib::algebraic::normal(lcoeff(c), [ts[-2]]);
    s := intlib::algebraic::rde(bmdt[1], bmdt[2], lcc[1], lcc[2], ts[1..-2], diffs[1..-2], algs[1..-2]);
    if has(s, FAIL) then return(FAIL); end_if;
    // s := s*t^m
    s := poly([[expr(s[1])/expr(s[2]), m]], [ts[-1]]);
    q := q+s;
    n := m-1;
    c := c - s*b - intlib::algebraic::diff(ts, diffs, algs)(s);
    c := mapcoeffs(c, normal);
  end_while;
  return(q);
end:
