// polynomial RDE, cancellation -- hypertangent case

// Given a derivation D on k[t] with D(t)/(t^2+1) in k, 
// not I in k, n either an integer or infinity,
// b0 in k and c in k[t],
// return either FAIL, in which case the equation
// D(q)+(b-n*t*D(t)/(t^2+1))*q=c has no solution of degree at most n in k[t],
// or a solution q in k[t] of this equation with degree(q) <= n.

// Follows Bronstein, Symbolic Integration I, p. 215

intlib::algebraic::rde::polyRDECancelTan :=
proc(b0, c, ts, diffs, algs, n)
  local cc, bb0, t, q, p, eta, cbar, c1, c0, u, v, r, h;
begin
  t := ts[-1];
  assert(nops(ts)>1);
  if iszero(n) then
    if iszero(degree(c)) then
      cc := intlib::algebraic::normal(lcoeff(c), [ts[-2]]);
      if iszero(b0) then
        return(poly(intlib::algebraic::inField(cc[1], cc[2], ts[1..-2], diffs[1..-2], algs[1..-2]), [t]))
      else
        bb0 := intlib::algebraic::normal(expr(b0), [ts[-2]]);
        q := intlib::algebraic::rde
          (cc[1], cc[2], bb0[1], bb0[2], ts[1..-2], diffs[1..-2], algs[1..-2]);
        if q = FAIL then return(FAIL); end_if;
        return(poly([[expr(q[1])/expr(q[2]), 0]], [t]));
      end_if;
    else return(FAIL); end_if;
  end_if;
  
  p := poly(t^2+1, [t]);
  eta := intlib::algebraic::normal(diffs[-1]/(t^2+1), [t]);   // t = tan(int(eta))
  assert(iszero(degree(eta[1])) and iszero(degree(eta[2])));
  eta := intlib::algebraic::normal(expr(eta[1])/expr(eta[2]), [ts[-2]]);
  [cbar, c1] := [divide(c, p)];
  assert(degree(c1)<2);
  c0 := coeff(c1, 0);
  c1 := coeff(c1, 1);
  // D(u)+b0*u+n*eta*v=c0, D(v)-n*eta*u+b0*v=c1
  bb0 := intlib::algebraic::normal(expr(b0), [ts[-2]]);
  [u, v] := intlib::algebraic::rde::coupledDE(bb0[1], bb0[2], -n*eta[1], eta[2], c0, c1, 
    ts[1..-2], diffs[1..-2], algs[1..-2]);
  if u=FAIL then return(FAIL); end_if;
  if n=1 then
    return(poly([[expr(u), 0], [expr(v), 1]], [t])); // return(u+v*t)
  end_if;
  r := poly([[u, 0], [v, 1]], [t]); // u+v*t
  c := c - intlib::algebraic::diff(ts, diffs, algs)(r) 
   - (poly([[n*expr(eta[1])/expr(eta[2]), 1]], [t])*r);
  c := intlib::algebraic::polyDivide(c, p);
  h := intlib::algebraic::rde::polyRDECancelTan(b0, c, ts, diffs, algs, n-2);
  return(p*h+r);
end:
