/*

  intlib::algebraic::fieldTower is supposed to find all the expressions needed for a
  simple differential field extension in which the given expression and its
  derivative are rational expressions.

  Input: expression and the indeterminate w.r.t. which to integrate
  
  Output: A list of
  - a list of new indeterminates to use
  - the list of original expressions these stand for
  - the list of derivatives, expressed in the new indeterminates
  - the list of algebraic dependencies,
    for transcendentals, as 
       an empty list
    for y^n-a=0, as
       "simpleradical", (verbatim string)
       n, a, the list of basis elements,
       another list with the derivatives of the basis elements,
       the list of what Bronstein[1] calls D_i,
       and what he refers to as H(=z^n)
        (the latter two are used in the Hermite reduction)

  - the expression, expressed in the new indets
  - a list of backsubstitutions to perform after
    the integration. All those terms are piecewise constant,
    such as ln(x)-ln(x+1)-ln(x/(x+1))

[1] M. Bronstein, Symbolic Integration Tutorial, ISSAC '98

*/

proc()
  local transExprs, hideNonIntegralPowers, hideNonRationalPowers;
  option escape;
begin

  // utility function: transExprs returns all the 
  // transcendental and simple radical expressions
  // in its argument which contain x.
  transExprs :=
  proc(f, x, options)
    local a, expo, n;
  begin
    if not has(f, x) then return({}); end_if;
    case type(f)
    of DOM_LIST do
    of DOM_SET do
    of DOM_TABLE do
    of "_mult" do
    of "_plus" do
      return(_union(op(map([op(f)], transExprs, x, options))));
    of "_power" do
      if testtype(op(f, 2), DOM_INT) then
        return(transExprs(op(f, 1), x, options));
      elif testtype(op(f, 2), DOM_RAT) and options["TranscendentalsOnly"]<>TRUE then
        return({op(f, 1)^(1/denom(op(f, 2)))} union transExprs(op(f, 1), x, options));
      else
        // normalize exponent: additive constant should be in [0,1)
        expo := op(f, 2);
        if testtype(expo, Type::Constant) then
          expo := expo - floor(expo);
        elif testtype(expo, "_plus") then
          expo := expo - floor(select(expo, testtype, Type::Constant));
        end_if;
        return({op(f, 1)^expo} union transExprs(op(f, 1), x, options));
      end_if;
      break;
//    // special case: don't add exp(2*x), exp(-x) etc.
//    of "exp" do
//      a := op(f);
//      if type(a) = "_mult" then
//        a := select(a, _not@testtype, Type::Integer);
//      end_if;
//      return({exp(a)} union transExprs(a, x, options));
    end_case;
    if domtype(f)=DOM_EXPR then
      return(_union({f}, op(map([op(f)], transExprs, x, options))));
    else
      return({f});
    end_if;
  end_proc;

// to store t^n, which is left alone even for t=exp(...):
intlib::algebraic::frozenPower := 
funcenv((base, expo) -> if expo=0 then 1 elif expo=1 then base
                           elif indets([args()]) minus Type::ConstantIdents = {} then base ^ expo
                           else procname(args()); end_if);
intlib::algebraic::frozenPower::diff :=
proc(ex, x)
  local f, g;
begin
  [f, g] := [op(ex)];
  ln(f)*ex*diff(g, x)+ex*g/f*diff(f, x);
end_proc;

  hideNonIntegralPowers :=
  proc(f, x)
    local i, powers, eq, base, expo;
  begin
    powers := subsop(table(), 0={});
    misc::maprec(f,
      {"_power"} = (ex -> (
        if has(ex, x) and not testtype(op(ex, 2), DOM_INT) then
          powers[op(ex, 1)] := powers[op(ex, 1)] union {op(ex, 2)};
        end_if;
        ex)));
    // TODO: find rational dependencies
    // substitute fractional powers by robust expressions:
    powers := [op(powers)];
    // for treating nested powers like sqrt(x^n), make sure we
    // substitute from the outermost to the innermost:
    powers := prog::sort(powers, ex -> -length(op(ex, 1)));
    for i from 1 to nops(powers) do
      eq := powers[i];
      base := op(eq, 1);
      for expo in op(eq, 2) do
        if testtype(expo, DOM_RAT) then
          f := subs(f, base^expo = 
            intlib::algebraic::frozenPower(base, 1/denom(expo))^numer(expo));
          powers := subs(powers, base^expo = 
            intlib::algebraic::frozenPower(base, 1/denom(expo))^numer(expo));
        else
          // TODO: Extract integer for 3/2*PI etc.
          f := subs(f, base^expo =
            intlib::algebraic::frozenPower(base, expo));
          powers := subs(powers, base^expo =
            intlib::algebraic::frozenPower(base, expo));
        end_if;
      end_for;
    end_for;
    f;
  end_proc:

  hideNonRationalPowers :=
  proc(f, x)
    local i, powers, eq, base, expo;
  begin
    powers := subsop(table(), 0={});
    misc::maprec(f,
      {"_power"} = (ex -> (
        if has(ex, x) and not testtype(op(ex, 2), DOM_RAT) and not testtype(op(ex, 2), DOM_INT) then
          powers[op(ex, 1)] := powers[op(ex, 1)] union {op(ex, 2)};
        end_if;
        ex)));
    // TODO: find rational dependencies
    // substitute fractional powers by robust expressions:
    powers := [op(powers)];
    // for treating nested powers like sqrt(x^n), make sure we
    // substitute from the outermost to the innermost:
    powers := prog::sort(powers, ex -> -length(op(ex, 1)));
    for i from 1 to nops(powers) do
      eq := powers[i];
      base := op(eq, 1);
      for expo in op(eq, 2) do
        if testtype(expo, DOM_RAT) then
          f := subs(f, base^expo = 
            intlib::algebraic::frozenPower(base, 1/denom(expo))^numer(expo));
          powers := subs(powers, base^expo = 
            intlib::algebraic::frozenPower(base, 1/denom(expo))^numer(expo));
        else
          // TODO: Extract integer for 3/2*PI etc.
          f := subs(f, base^expo =
            intlib::algebraic::frozenPower(base, expo));
          powers := subs(powers, base^expo =
            intlib::algebraic::frozenPower(base, expo));
        end_if;
      end_for;
    end_for;
    f;
  end_proc:

intlib::algebraic::fieldTower :=
proc(f, x, options)
  local vars, Dex, n, diffs, i, j, ts, valuation, seen, simpl,
        substs, exps, lns, algs,
        ex, ex2, eqs, eq, r, constants,
        offset, k, const, sol, simplificationSubsts,
        ret;
begin
  // not really differentiable:
  ret := FALSE;
  misc::maprec(f,
    {"abs", "Re", "Im", "sign",
    // differentiable, but don't yield useful towers:
    "sum", "product"} =
      (ex -> (if has(ex, [x]) then ret := FAIL; misc::breakmap(); end; ex)),
//    {"sum"} =
//      (s -> (if testtype(op(s, [2, 2]), RootOf) and has(s, [x]) then ret := FAIL; misc::breakmap(); end_if; args())),
    Unsimplified);
  if ret = FAIL then
    return([FAIL $ 6]);
  end_if;
  
  // simplify logs: Extract roots
  f := misc::maprec(f,
    {"ln"} = proc(lnx)
             begin
               if has(lnx, x) and type(op(lnx)) = "_power" and
                 testtype(op(lnx, [1, 2]), DOM_RAT) and
                 (abs(op(lnx, [1, 2])) < 1 or is(op(lnx, [1, 1]) in R_, Goal = TRUE))
               then
                 ln(op(lnx, [1, 1]))*op(lnx, [1, 2]);
               else
                 lnx
               end_if;
             end_proc,
    PostMap);
  
  if options["TreatAlgebraicsAsTranscendentals"] = TRUE then
    f := hideNonIntegralPowers(f, x);
  else
    f := hideNonRationalPowers(f, x);
  end_if;
  
  vars := indets(f, All) union {x};
  ts := [];
  algs := [];

  // for new vars, #t1, #t2, etc. are used.
  // Find the first one not appearing in the input:
  n := select(map(vars, expr2text),
              x -> _lazy_and(length(x) > 2, x[1..2] = "#t" or x[1..2] = "#c"));
  if n = {} then n := 1
  else
    n := map(n, s -> text2expr(strmatch(s."x0", "\\d+", ReturnMatches)[1]));
    n := max(n)+1;
  end_if;

  // to avoid non-transcendental extensions, try rewriting in terms
  // of what we already had:
  valuation := ex -> intlib::Simplify::defaultValuation(ex) +
                      1500 * nops(_union(transExprs(ex, x, options),
                        transExprs(ex, i, options) $ i in ts) minus {op(vars)});

  // to get consistent arguments, and, hence lookups
  /*
  simpl := proc(t) option remember;
           begin
             intlib::Simplify(t, options,
                     Valuation=valuation,
                     Strategy="intlib::algebraic::fieldTower",
                     Steps=max(1, min(100,
                       ceil(intlib::Simplify::defaultValuation(t)/5))));
           end_proc;
  */
  simpl := t -> t:

  vars := transExprs(f, x, options);
  if FAIL in vars then // non-algebraic powers (x^n and things like that)
    return([FAIL $ 6]);
  end_if;
  diffs := map(vars, diff, x);
  
  seen := {}; // handled these expressions already
  i := 1;
  repeat
    i := i+1;
    if i > 5 then
      return([FAIL $ 6]); 
    end_if;
    Dex  := map(diffs, ex -> op(transExprs(ex, x, options))) minus vars minus seen;
    seen := seen union Dex;
    Dex := map(Dex, t -> op(transExprs(simpl(t), x, options)));
    if has(Dex, [FAIL]) then
      return([FAIL $ 6]);
    end_if;
    
    vars := vars union Dex;
    diffs := map(vars, diff, x);
  until Dex = {} end_repeat;
  
  // ensure consistent forms in substitution and input etc.
  if op(simpl, 5) = NIL then
    simplificationSubsts := []
  else
    simplificationSubsts := prog::sort([op(op(simpl, 5))],
      l -> -length(l[1]));
    f := subsex(f, simplificationSubsts);
  end_if;
  
  // remove those algebraically dependent one some other term
  // The following sorting will place subexpressions in front of
  // expressions they appear in:
  vars := prog::sort([op(vars)], intlib::Simplify::defaultValuation);
  diffs := table(op(map(vars, t -> t=diff(t, x))));
  diffs := map(diffs, subsex, simplificationSubsts);
  substs := table();

  constants := table():
  for i in diffs do
    if iszero(op(i, 2)) then
      const := op(i, 1);
      delete vars[contains(vars, const)];
      delete diffs[const];
      f := subs(f, const = `#c`.(nops(constants)+1));
      diffs := subs(diffs, const = `#c`.(nops(constants)+1));
      constants[`#c`.(nops(constants)+1)] := const;
    end;
  end_for;
    
  i := 2;
  ts := [`#t`.(i+n) $ i = 0..nops(vars)-1];
  r := solvelib::getIdent(Any, indets([args(), vars, ts]));

  while i <= nops(vars) do
    // use the Risch structure theorem
    // to check whether vars[i] is indeed transcendental
    // over K[vars[1],...,vars[i-1]]
    ex := vars[i];
    j := 1;
    while j < i do
      if iszero((ex - vars[j]) | hold(intlib::algebraic::frozenPower) = _power) then
        break;
      end_if;
      j := j + 1;
    end_while;
    if j < i then
      // found genuine duplicate, vars[j] is simpler by sort order
      f := subsex(f, vars[i]=vars[j]);
      delete vars[i];
      next;
    end_if;
    case type(ex)
    of "exp" do
      exps := select([$1..i-1], j -> testtype(vars[j], "exp"));
      lns := select([$1..i-1], j -> testtype(vars[j], "ln"));
      if exps=[] and lns=[] then break; end_if;
      eq := subsex((_plus(r[j]*diffs[vars[lns[j]]] $ j = 1..nops(lns),
                     r[j+nops(lns)]*diffs[vars[exps[j]]]/vars[exps[j]] $ j = 1..nops(exps),
                     -diffs[ex]/ex)), revert(zip(vars[1..i], ts, _equal)),
                     hold(intlib::algebraic::frozenPower) = _power);
      eqs := intlib::algebraic::normal(eq, ts);
      eqs := [coeff(eqs[1])];
      sol := intlib::algebraic::ratlinsolve(eqs, [r[j]$j=1..nops(exps)+nops(lns)]);
      if sol=FAIL then break; end_if;
      // there is a rational solution, i.e.,
      // vars[i] is algebraic over K[vars[1],...,vars[i-1]].
      // To be more precise,
      // vars[i] = constant*_mult(op(vars[lns[i]])^r[i]$i=1..nops(lns))
      //                   *_mult(vars[exps[i]]^r[i+nops(lns)]$i=1..nops(exps))
      //
      // In the ideal case, all the right hand sides are integers:
      if map({op(map(sol, op, 2))}, domtype) = {DOM_INT} then
        // find the constant
        const := simplify(subs(op(ex)-
                               _plus(r[j]*vars[lns[j]] $ j = 1..nops(lns),
                                     r[j+nops(lns)]*op(vars[exps[j]]) $ j = 1..nops(exps)),
                               sol, hold(intlib::algebraic::frozenPower)=_power));
        assert(_lazy_or(not has(const, x), testeq(diff(const, x), 0)));
        f := subs(f, ex = subs(_mult(exp(const),
             intlib::algebraic::frozenPower(op(vars[lns[j]]), r[j]) $ j = 1..nops(lns),
             intlib::algebraic::frozenPower(vars[exps[j]], r[j+nops(lns)]) 
               $ j = 1..nops(exps)), sol));
        vars := subs(vars, ex = subs(_mult(exp(const),
             intlib::algebraic::frozenPower(op(vars[lns[j]]), r[j]) $ j = 1..nops(lns),
             intlib::algebraic::frozenPower(vars[exps[j]], r[j+nops(lns)]) 
               $ j = 1..nops(exps)), sol));
        diffs := table(op(subs([op(diffs)], ex = subs(_mult(exp(const),
             intlib::algebraic::frozenPower(op(vars[lns[j]]), r[j]) $ j = 1..nops(lns),
             intlib::algebraic::frozenPower(vars[exps[j]], r[j+nops(lns)]) 
               $ j = 1..nops(exps)), sol))));
        delete vars[i];
        i := i-1; // counteract the increasement below.
        break;
      end_if;
        
      // Otherwise, we must build the tower differently
      // e.g. [x, exp(x), exp(3/2*x)] -> [x, exp(x/2)]
      sol := subs([r[j]$j=1..nops(lns)+nops(exps)],sol);
      for j from 1 to nops(sol) do
        if testtype(sol[j], DOM_RAT) then
          break;
        end_if;
      end_for;
      // If the first entry in vars which has a non-integral
      // coefficient in the solution is an exponential,
      if j > nops(lns) then
        // we insert the common base in front of it and restart from there:
        i := exps[j-nops(lns)];
        ex2 := exp(1/denom(sol[j])*op(vars[i]));
        k := contains(vars, ex2);
        if k > 0 then
          if k <> i then
            [vars[i],vars[k]] := [vars[k], vars[i]];
          end_if;
        else
          vars[i..i-1] := [ex2];
          diffs[ex2] := diff(ex2, x);
        end_if;
        // unneccessary:
        // i := i-1;
      else
        // otherwise, we have some form of exp(1/2*ln(...))
        // TODO: it may be possible to do this,
        //       e.g. for exp(ln(x^2+2*x+1)/2)
        return([FAIL $ 6]);
      end_if;
      break; // end of handling exp
    of "ln" do
      exps := select([$1..i-1], j -> testtype(vars[j], "exp"));
      lns := select([$1..i-1], j -> testtype(vars[j], "ln"));
      if exps=[] and lns=[] then break; end_if;
      eq := subsex((_plus(r[j]*diffs[vars[lns[j]]] $ j = 1..nops(lns),
                     r[j+nops(lns)]*diffs[vars[exps[j]]]/vars[exps[j]] $ j = 1..nops(exps),
                     -diffs[ex])), revert(zip(vars[1..i], ts, _equal)),
                     hold(intlib::algebraic::frozenPower) = _power);
      eqs := intlib::algebraic::normal(eq, ts);
      eqs := [coeff(eqs[1])];
      sol := intlib::algebraic::ratlinsolve(eqs, [r[j]$j=1..nops(exps)+nops(lns)]);
      if sol=FAIL then break; end_if;
      // there is a rational solution, i.e.,
      // vars[i] is algebraic over K[vars[1],...,vars[i-1]].
      // To be more precise,
      // vars[i] = constant+_plus(r[i]*vars[lns[i]]$i=1..nops(lns))
      //                   *_plus(r[i+nops(lns)]*op(vars[exps[i]])$i=1..nops(exps))
      //
      // In the ideal case, all the right hand sides are integers:
      if map({op(map(sol, op, 2))}, domtype) = {DOM_INT} then
        // find the constant
        const := simpl(ex-subs(_plus(r[j]*vars[lns[j]] $ j = 1..nops(lns),
                                     r[j+nops(lns)]*op(vars[exps[j]]) $ j = 1..nops(exps)), sol));
        // problem: the constant may well contain x,
        // just be piecewise constant, such as ln(x/(x+1))-ln(x)+ln(x+1).
        if has(const, x) then
          f := subs(f, ex = subs(_plus(`#c`.(nops(constants)+n),
                                       r[j]*vars[lns[j]] $ j = 1..nops(lns),
                                       r[j+nops(lns)]*op(vars[exps[j]]) $ j = 1..nops(exps)), sol));
          constants[`#c`.(nops(constants)+n)] := const;
        else
          f := subs(f, ex = subs(_plus(const,
                                       r[j]*vars[lns[j]] $ j = 1..nops(lns),
                                       r[j+nops(lns)]*op(vars[exps[j]]) $ j = 1..nops(exps)), sol));
        end_if;
        delete vars[i];
        i := i-1; // counteract the increasement below.
        break;
      end_if;
        
      // Otherwise, we give up. It may be possible to
      // reorder the terms appropriately, but there is
      // currently no code to recognize a square in k(x,y,z,...).
      return([FAIL $ 6]);
      break; // end of handling ln
    of "tan" do
      break;
    of "arctan" do
      break;
    end_case;
    i := i+1;
  end_while;

  vars := map(vars, subsex, simplificationSubsts);
  
  diffs := map(vars, diff, x);

  // this uses remembered value from above, so make sure
  // the arguments remain synchronized!
  diffs := map(diffs,
               ex -> if _union(transExprs(ex, x, options),
                               transExprs(ex, i, options) $ i in ts)
                        minus {op(vars)} = {} then
                      ex
                    else
                      simpl(ex)
                    end_if);

  diffs := map(diffs, subsex, simplificationSubsts);

  ts := [#t.(i+n) $ i = 0..nops(vars)-1];
  
  // subsex(exp(-x), exp(x)=t) does not work as expected. :-(
  diffs := subsex(diffs, map(revert(zip(vars, ts, _equal)),
    eq -> if type(op(eq, 1))="exp" then eq, exp(-op(op(eq, 1))) = 1/op(eq, 2)
          else eq end_if));

  f := subsex(f, revert(zip(vars, ts, _equal)));
  f := subsex(simpl(f), revert(zip(vars, ts, _equal)),
     hold(intlib::algebraic::frozenPower)=hold(_power));
  diffs := subs(diffs, hold(intlib::algebraic::frozenPower)=hold(_power));
  vars := subs(vars, hold(intlib::algebraic::frozenPower)=hold(_power));
     
  // remove monomials that appear neither in f nor in diffs nor in bases of algebraics,
  // except for `#t1`=x
  offset := 0;
  for i from 2 to nops(ts) do
    if not has(diffs, ts[i]) and not has(f, ts[i]) and 
      not has(select(vars[1..i-offset-1].vars[i-offset+1..-1],
          v -> testtype(v, "_power") and testtype(op(v, 2), DOM_RAT)),
        vars[i-offset]) then
      delete vars[i-offset];
      delete diffs[i-offset];
      offset := offset+1;
    else
      assert(iszero(offset) or not has([diffs, f], ts[i-offset]));
      [diffs, f] := subs([diffs, f], ts[i]=ts[i-offset], Unsimplified);
    end_if;
  end_for;

  ts := ts[1..-1-offset];
  if options["TreatAlgebraicsAsTranscendentals"]=TRUE then
    algs := [[] $ nops(ts)];
  else
    algs := [[]];
    for i from 2 to nops(ts) do
      if type(vars[i])="_power" and testtype(op(vars[i], 2), DOM_RAT) then
//        assert(testtype(1/op(vars[i], 2), Type::PosInt));
        ret := intlib::algebraic::simpleRadicalIntegralBasis(
          ts[i], 1/op(vars[i], 2),
          // subsex(exp(-x), exp(x)=t) does not work as expected. :-(
          subsex(op(vars[i], 1), map(revert(zip(vars[1..i-1], ts[1..i-1], _equal)),
            eq -> if type(op(eq, 1))="exp" then eq, exp(-op(op(eq, 1))) = 1/op(eq, 2)
                  else eq end_if)),
          ts[1..i-1], diffs[1..i-1], algs);
        if ret = FAIL then return([FAIL $ 6]); end_if;
        diffs[i] := ret[2][i];
        algs := ret[3];
        delete ret;
      else
        algs := algs.[[]];
      end_if;
    end_for;
  end_if;

  ret := [ts, subs(vars, hold(intlib::algebraic::frozenPower)=hold(_power)),
    diffs, algs, f, [op(constants)]];
  ret := [rationalize(ret, StopOn = (ex -> has(ex, ts)), FindRelations = ["exp", "_power"])];
  ret[1][-1] := ret[1][-1].[op(ret[2])];
  ret := ret[1];
  ret;
end_proc;

end_proc():
