// In-Field Integration
// 
// Given a derivation d on k(t) and f in k(t), check if there is
// g in k(t) such that d(g)=f. Returns g if it exists, FAIL if
// no such g exists.

// Basically follows Bronstein, Symbolic Integration I, p. 175f.

intlib::algebraic::inField :=
proc(fNum, fDen, ts, diffs, algs)
  local g, g1, dg1, h, r, t, gc, good, v, c;
begin
  [g,h,r] := intlib::algebraic::Hermite(fNum, fDen, ts, diffs, algs);
  if g = FAIL or degree(h[2])>0 then
    // no such g
    return(FAIL);
  end_if;
  
  if not iszero(h[1]) then
    // r := r+h
    r[1] := h[2]*r[1]+h[1]*r[2];
    r[2] := r[2]*h[2];
    gc := gcd(r[1], r[2]);
    r := map(r, intlib::algebraic::polyDivide, gc);
  end_if;

  if iszero(r[1]) then return(g); end_if;
  
  t := ts[-1];

  dg1 := degree(diffs[-1], [t]);
  if dg1 <> FAIL and dg1 > 0 then
    // write r=r[1]/r[2] as p+a/d, reduce polynomial part
    g1 := divide(r[1], r[2], Quo);
    g1 := intlib::algebraic::polynomialReduce(g1, ts, diffs, algs)[1];
    g := g+g1;

    if not iszero(g1) then
      // r := r - d(g1)
      dg1 := intlib::algebraic::normal(intlib::algebraic::diff(ts, diffs, algs)(g1), [ts[-1]]);
      r := [dg1[2]*r[1]-dg1[1]*r[2], dg1[2]*r[2]];
      gc := gcd(r[1], r[2]);
      r := map(r, intlib::algebraic::polyDivide, gc);
      r := map(r, intlib::algebraic::reduceModAlgebraics, ts, diffs, algs);
      r := map(r, mapcoeffs, normal, Expand = FALSE);
    end_if;

    if iszero(r[1]) then return(g); end_if;
  end_if;
  
  if not has(diffs[-1], t) then
    // t is a primitive
    [g1, good] := intlib::algebraic::integratePrimitive(r[1], r[2], ts, diffs, algs);
    if good=FALSE then return(FAIL); end_if;
    
    g := g+g1;
    
    r := intlib::algebraic::normal(expr(r[1])/expr(r[2])
           -intlib::algebraic::diff(ts, diffs, algs)(g1), [t]);
    if nops(ts)=1 then
      return(g+expr(r[1])/expr(r[2])*t);
    else
      // solve the limited integration problem r=d(v)+c*d(t)
      [v, c] := intlib::algebraic::limitedIntegrate(r[1], r[2],
        [intlib::algebraic::normal(diffs[-1], [ts[-1]])], ts, diffs, algs);
      if v=FAIL then return(FAIL); end_if;

      g := g+c[1]*t;
      return(g + intlib::algebraic::inField(op(intlib::algebraic::normal(v, [ts[-2]])),
                         ts[1..-2], diffs[1..-2], algs[1..-2]));
    end_if;
  else
    // degree(d(t), [t])>=1. Now, r in k has an elementary
    // integral over k(t) iff it has one over k. Try that:
    if degree(r[1]) > 0 or degree(r[2]) > 0 then
      return(FAIL);
    end_if;
    assert(nops(ts)>1); // x is a primitive over Q_.
    t := ts[-2];
    r := intlib::algebraic::normal(expr(r[1])/expr(r[2]), [t]);
    return(g + intlib::algebraic::inField(r[1], r[2], ts[1..-2], diffs[1..-2], algs[1..-2]));
  end_if;
end:
