// limited integration -- reduction to a polynomial problem

// Given a derivation d on k(t) and f,w1,...,wn in k(t)
// (each given as a list of [numerator, denominator]),
// return [v, [c1,...,cm]], v in k(t), c1,...,cm in Const(k)
// such that f=d(v)+sum(c[i]*w[i]), if such a v exists,
// [FAIL, FAIL] otherwise.

// Follows Bronstein, Symbolic Integration I, ch. 7.2 and 7.1

intlib::algebraic::limitedIntegrate :=
proc(fNum, fDen, w, ts, diffs, algs)
  local a, b, h, N, g, v, f, A, p, q, r, z, M, M2, dummy, i, m;
begin
  // step 1: reduce to polynomial problem
  [a,b,h,N,g,v] := 
    intlib::algebraic::limitedIntegrateReduce(fNum, fDen, w, ts, diffs, algs);
  if a=FAIL then
    // failed to reduce to polynomial problem
    return([FAIL, FAIL]);
  end_if;
  if iszero(degree(g[2])) then
    g := multcoeffs(g[1], 1/expr(g[2]));
  else
    // multiply the equation a*d(q)+b*q=g+sum(c[i]*v[i]) 
    // with the denominator of g - does not change q.
    a := a*g[2];
    b := b*g[2];
    v := map(v,
      proc(vi)
        local a;
      begin
        a := gcd(vi[2], g[2]);
        vi[2] := intlib::algebraic::polyDivide(vi[2], a);
        vi[1] := vi[1] * intlib::algebraic::polyDivide(g[2], a);
        vi
      end_proc):
    g := g[1];
  end_if;

  // reduce a*d(q)+b*q=g+sum(c[i]*v[i])
  // to the same problem with v[i] in k[t], plus M*c=0:
  [v, M] := intlib::algebraic::rde::linearConstraints(a, b, ts, diffs, algs, v);
  [M, dummy] := intlib::algebraic::rde::constantSystem(M, [0$nops(M)], ts, diffs, algs);

  // problem now: a*d(q)+b*q=g+sum(c[i]*v[i]).
//  // if d = d/dt and b is constant, c[i]=0 is a solution
//  if (iszero(b) or iszero(degree(b))) and nops(ts)=1 then
//    // base case
//    // equating coefficients leads to linear system
//    M := array(0..N, 0..N+1,
//      [[0, j*coeff(a, i-j+1) $ j=1..i+1, 0 $ N-i] $ i=0..N]);
//    for i from 0 to N do
//      M[i, i] := M[i, i] + expr(b);
//    end_for;
//    M := numeric::matlinsolve(M, [coeff(g, i) $ i=0..N], Symbolic);
//    if M[1] = FAIL then // no solution
//      return([FAIL, FAIL]);
//    end_if;
//    p := poly([[M[1][i,1], i] $ i=0..N], [ts[1]]);
//    return([expr(p)/expr(h), [0$nops(w)]]);
//  end_if;

  // add c[0] for g -- shift M to the right
  M["cols"] := M["cols"] + 1;
  M["m"] := M["m"] + 1;
  M := table(map([op(M)],
    eq -> if type(op(eq, 1))= "_exprseq" then
        subsop(eq, [1, 2] = op(eq, [1, 2]) + 1);
      else
        eq
      end_if), 0);
  v := [g].v;
  
  // reduce a*d(q)+b*q=sum(c[i]*v[i]) to
  // d(p)+b'*p=sum(c[i]*z[i])
  [a, b, z, r, M2, N] :=
    intlib::algebraic::rde::parametricSPDE(a, b, v, ts, diffs, algs, N);
  M := intlib::algebraic::rde::combineHashMatrices(M, M2);

  // find a solution p of the polynomial problem
  [f, A] := intlib::algebraic::rde::parametricPolyRDE(b, z, N, ts, diffs, algs);

  // combine linear constraints, adding c[0]=1
  m := M["cols"];
  assert(A["cols"] = m + nops(f));
  A["m"] := M["m"];
  M := intlib::algebraic::rde::combineHashMatrices(M, A);
  M := matrix(M["rows"]+1, M["cols"],
    select([op(M)], eq -> testtype(op(eq, 1), "_exprseq")).
    [(M["rows"]+1, 1)=1]);
  // find special solution
  [A, dummy] := numeric::matlinsolve(M, [0 $ op(M, [0, 2, 2]) - 1, 1], Symbolic, "dont_use_normal");
  if A=FAIL then
    return([FAIL, FAIL]);
  end_if;
  p := _plus(A[i+m,1]*f[i] $ i=1..nops(f));
  
  // return solution to original problem
  q := _plus(p*a, A[i,1]*r[i]$i=1..nops(r));
  return([expr(q)/expr(h), [A[i, 1] $ i=2..nops(r)]]);
end_proc:
