// recognizing logarithmic derivatives, in-field

// given f in k(t), determine whether there is a nonzero u in k(t)
// such that f=d(u)/u. Return u if it exists, FAIL otherwise.

// in the case of hyperexponentials,
// this code relies on parametricLogarithmicDerivative and
// therefore is only a heuristic method.

// Essentially follows Bronstein, Symbolic Integration I, p. 176

intlib::algebraic::logarithmicDerivative :=
proc(fNum, fDen, ts, diffs, algs)
  local t, d, g, p, v, u, un_ln, a, b;
begin
  d := intlib::algebraic::diff(ts, diffs, algs);
  // f must be simple:
  if degree(gcd(fDen,d(fDen))) > 0 then
    return(FAIL);
  end_if;
  g := intlib::algebraic::residueReduce(fNum, fDen, ts, diffs, algs);
  if g[2] <> TRUE then
    return(FAIL);
  end_if;
  g := g[1];
  if iszero(g) then
    v := 1;
  else
    // g now should be a linear combination of logarithms,
    // d(g) = d(v)/v. Extract v:
    un_ln :=
    g -> case type(g)
          of "ln" do
            op(g);
            break;
          of "_mult" do
            if type(op(g, 1)) = "ln" and type(op(g, 2))=DOM_INT then
              op(g,[1,1])^op(g, 2);
            else
              return(FAIL);
            end_if;
            break;
          otherwise
            return(FAIL);
         end_case;
    case type(g)
    of "ln" do
    of "_mult" do
      v := un_ln(g);
      if v=FAIL then
        return(FAIL);
      end_if;
      break;
    of "_plus" do
      v := map([op(g)], un_ln);
      if contains(v, FAIL) > 0 then
        return(FAIL);
      end_if;
      v := _mult(op(v));
      break;
    otherwise
      return(FAIL);
    end_case;
  end_if;
  
  t := ts[-1];
  p := intlib::algebraic::normal(expr(fNum)/expr(fDen) - d(v)/v, [t]);
  if degree(p[2]) > 0 or degree(p[1]) >= max(1, degree(diffs[-1], [t])) then
    return(FAIL);
  end_if;
  
  p := multcoeffs(p[1], 1/lcoeff(p[2]));
  
  if iszero(p) then
    return(v);
  end_if;

  if nops(ts) = 1 then
    return(FAIL); // no constant is a logarithmic derivative
  end_if;
  
  case degree(diffs[1], [t])
  of 0 do // primitive
    p := intlib::algebraic::normal(expr(p), [ts[-2]]);
    u := intlib::algebraic::logarithmicDerivative(p[1], p[2], ts[1..-2], diffs[1..-2], algs[1..-2]);
    if u=FAIL then
      return(FAIL);
    end_if;
    return(u*v);
    break;
  of 1 do // hyperexponential
    p := intlib::algebraic::normal(expr(p), [ts[-2]]);
    u := intlib::algebraic::parametricLogarithmicDerivative(p[1], p[2], ts, diffs, algs);
    if u[1] <> 1 then
      return(FAIL);
    end_if;
    return(v*u[3]*t^u[2]);
    break;
  of 2 do // hypertangent
    a := coeff(p, 1);
    b := coeff(p, 0);
    // TODO: options
    b := intlib::Simplify(b/2*(t^2+1)/diffs[-1]);
    if is(b in Z_) then
      p := intlib::algebraic::normal(a, [ts[-2]]);
      u := intlib::algebraic::logarithmicDerivative(p[1], p[2], ts[1..-2], diffs[1..-2], algs[1..-2]);
      if u=FAIL then
        return(FAIL);
      end_if;
      return(u*v*(t^2+1)^b);
    end_if;
    return(FAIL);
    break;
  otherwise
    error("internal error: bad field tower");
  end_case;
  FAIL;
end:
