// recognizing logarithmic derivatives

// Given f in k(t), determine whether there is a nonzero n in Z_ and 
// a nonzero u in k(t) such that Du/u=n*f, and to compute them if they
// exist.

// Essentially follows Bronstein, Symbolic Integration I, p. 177

intlib::algebraic::radicalLogarithmicDerivative :=
proc(fNum, fDen, ts, diffs, algs)
  local g, v, good, m, alpha, p, dummy, dv, u, n, mn, e;
begin
  // is f simple?
  if degree(mapcoeffs(gcd(fDen, intlib::algebraic::diff(ts, diffs, algs)(fDen)), normal)) > 1
  then return([FAIL, FAIL]); end_if;

  [g, good] := intlib::algebraic::residueReduce(fNum, fDen, ts, diffs, algs);
  if not good then
    return([FAIL, FAIL]);
  end_if;
  // we prefer ln here; g is partly computed by rational integration and can contain arctan
  g := subs(g, hold(arctan) = (x -> I/2*(ln(1-I*x)-ln(1+I*x))), EvalChanges);
  
  if iszero(g) then
    if not iszero(degree(fNum)) or not iszero(degree(fDen)) then
      return([FAIL, FAIL]);
    end_if;
    v := 1;
    m := 1;
    p := multcoeffs(fNum, 1/expr(fDen));
  else
    // g is of the form _plus(alpha[i]*ln(v[i])). 
    // Get a common denominator of the alpha:
    if not testtype(g, "_plus") then
      if testtype(g, "_mult") then
        alpha := {select(g, _not@has, ts), 1};
      else
        alpha := {1};
      end_if;
    else
      alpha := map({op(g)}, select, _not@has, ts) union {1};
    end_if;
    if map(alpha, testtype, Type::Rational) <> {TRUE} then
      return([FAIL, FAIL]);
    end_if;
    m := denom(_plus(op(map(alpha, abs))));
  
    // next, go from _plus(alpha[i]*ln(v[i])) to
    // 1/m*ln(_mult(v[i]^(alpha[i]*m))). Actually, all we want
    // is the _mult(v[i]^(alpha[i]*m)) part.
    if testtype(g, "_plus") then
      v := _mult(op(map([op(g)], proc(t)
                                   local a,ga,dummy;
                                 begin
                                   if type(t)="ln" then op(t)^m 
                                   else 
                                     assert(type(t)="_mult");
                                     [ga,a,dummy] := split(t, testtype, "ln");
                                     assert(type(ga)="ln");
                                     op(ga)^(a*m);
                                   end_if;
                                 end_proc)));
    else
      if testtype(g, "ln") then
        v := op(g)^m;
      else
        assert(type(g)="_mult");
        [v, alpha, dummy] := split(g, testtype, "ln");
        assert(type(v)="ln");
        v := op(v)^(alpha*m);
      end_if;
    end_if;
  
    // p := f-D(g) = f-D(v)/(m*v)
    dv := intlib::algebraic::diff(ts, diffs, algs)(v);
    // v might not be a polynomial in t
    p := intlib::algebraic::normal(expr(fNum)/expr(fDen)-dv/(v*m), [ts[-1]]);
    if degree(p[2]) > 0 then
      // f is not a logarithmic derivative of a k(t) radical
      return([FAIL, FAIL]);
    end_if;

    p := multcoeffs(p[1], 1/expr(p[2]));
  end_if;
  
  if iszero(p) then return([v, m]); end_if;
  
  // The next step depends on the type of t.
  // Primitive:
  if not has(diffs[-1], ts[-1]) then
    if degree(p) > 0 then // f is not a logarithmic derivative of a k(t) radical
      return([FAIL, FAIL]);
    end_if;
    if nops(ts) > 1 then
      [u, n] := intlib::algebraic::radicalLogarithmicDerivative
        (op(intlib::algebraic::normal(expr(p), [ts[-2]])), ts[1..-2], diffs[1..-2], algs[1..-2]);
      if u=FAIL then
        return([FAIL, FAIL]);
      end_if;
      mn := lcm(abs(m), abs(n));
      v := v^(mn/m)*u^(mn/n);
      m := mn;
    end_if;
    return([v, m]);
  end_if;
  // Hyperexponential:
  if degree(diffs[-1], [ts[-1]]) = 1 then
    if degree(p) > 0 then // f is not a logarithmic derivative of a k(t) radical
      return([FAIL, FAIL]);
    end_if;
    [n, e, u] := intlib::algebraic::rde::parametricLogarithmicDerivative
       (op(intlib::algebraic::normal(expr(p), [ts[-2]])), ts, diffs, algs);
    if n=FAIL then // no solution found
      return([FAIL, FAIL]);
    end_if;
    mn := denom(1/abs(m)+1/abs(n));
    v := v^(mn/m)*u^(mn/n)*ts[-1]^(e*mn/n);
    return([v, m]);
  end_if;
  // Hypertangent:
  // TODO
  return([FAIL, FAIL]);
end:
