// integral basis for simple radical extension K(t)[y]/(y^n-a) of K(t)
//
// K(t) is given, as usual, by its list of non-constant generators,
// their derivatives, and information on algebraic extension.
// Return value is a list containing
// 1. the extended list of transcendentals
// 2. the extended list of derivatives thereof
// 3. the extended list with information on algebraic extensions
//   This list is amended with a list containing
//   the string "simpleradical",
//   n, a, the list of basis elements,
//   another list with the derivatives of the basis elements,
//   the list of what Bronstein calls D_i,
//   and what he refers to as H(=z^n)
//    (the latter two are used in the Hermite reduction)
//
// Follows Bronstein, "Symbolic Integration Tutorial", ISSAC '98, Sec. 2.2

intlib::algebraic::simpleRadicalIntegralBasis :=
proc(y, n, a, ts, diffs, algs)
  local t, aNum, aDen, aFact, H, D, F, z, w, i, j;
begin
  t := ts[-1];
  [aNum, aDen] := intlib::algebraic::normal(a, [t]);
  if aNum=FAIL then return(FAIL); end_if;
  aFact := coerce(polylib::sqrfree(aNum*aDen^(n-1)), DOM_LIST);
  H := [poly(1, [t]) $ n];
  H[1] := H[1]*aFact[1];
  F := poly(1, [t]);
  for i from 2 to nops(aFact) step 2 do
    j := (aFact[i+1]-1 mod n) + 1;
    H[j] := H[j] * aFact[i];
    F := F * aFact[i]^(aFact[i+1] div n);
  end_for;
  z := y*expr(aDen)/expr(F);
  H := map(H, expr);
  D := [1/_mult(H[j]^floor((i-1)*j/n) $ j=1..nops(H)) $ i = 1..n];
  w := [z^(i-1)/D[i] $ i=1..n]; // formula (11)
  diffs := diffs.[normal(y/(n*a)*intlib::algebraic::diff(ts, diffs, algs)(a), Expand=FALSE)];
  ts := ts.[y];
  [ts, diffs, 
    algs.[["simpleradical", n, a, w, map(w, intlib::algebraic::diff(ts, diffs, algs)), D, 
      _mult(H[i]^i $ i = 1..nops(H))]]];
end_proc:

