// Integration in k(y) where y^n in k, logarithmic part
//
// Given f in k(y) reduced, compute res = sum(c[i]*ln(u[i]), i=1..m)
// s.t. res'-f in k, c[i] in const(k), u[i] in k(t), or FAIL if no such F exists.
// 
// return value: [res, TRUE] if successful, [something, FALSE] else
// (for consistency with intlib::algebraic::integratePrimitive etc.)
// 
//  Follows Bronstein, Symbolic Integration Tutorial, ISSSAC '98, Section 2.5
//  and B.M. Trager, Integration of Algebraic Functions, 1984, Sections 5.2, 5.3, 6.1-6.3

intlib::algebraic::simpleRadicalLogPart :=
proc(fnum, fden, ts, diffs, algs)
  local num, den, G, z, y, n, a, w, F, h, r, i,
    d, integral, int2, good;
begin
  [z, y] := ts[-2..-1];
  [n, a, w] := algs[-1][2..4];
  assert(op(fden, 2) = [y]);
  if degree(fden) > 0 then
    [num, den] := intlib::algebraic::normalizeAlgebraicFraction(fnum, fden, ts, diffs, algs);
    G := _plus(op(zip(num, w, `*`)));
    [fnum, fden] := intlib::algebraic::normal(expr(G)/expr(den), [y]);
  end_if;
  assert(iszero(degree(fden)));

  // Split the work by linearity
  // to only have terms of the form a/b*y
  // see p.7 of Trager's paper for a proof 
  // that the sum is integrable if and only if each term is.
  integral := 0;
  if nterms(fnum) > 1 then
    for i from 1 to nterms(fnum) do
      [int2, good] := intlib::algebraic::simpleRadicalLogPart(
        nthmonomial(fnum, i), fden, ts, diffs, algs);
      integral := integral + int2;
      if good = FALSE then
        return([integral, FALSE]);
      end_if;
    end_for;
    return([integral, TRUE]);
  end_if;

  if iszero(degree(fnum)) then
    // nothing left to do for us. Smaller field tower is handled nicely by our caller.
    return([0, TRUE]);
  end_if;
  
  // for situations as in int(x/(sqrt(x+1)+sqrt(x-1)), x),
  // we can actually split the work to reductions over the hyperelliptic case
  // by leaving out intermediate field extensions not needed:
  [num, den] := intlib::algebraic::normal(expr(fnum)/expr(fden), [z, y]);
  while nops(ts) > 2 and
    iszero(degree(num, z)) and iszero(degree(den, z)) and
    not has(diffs[-1], [z]) and not has(algs[-1], [z]) do
    delete ts[-2], diffs[-2], algs[-2];
    z := ts[-2];
    [num, den] := intlib::algebraic::normal(expr(num)/expr(den), [z, y]);
  end_while;
  [num, den] := map([num, den], poly, [y]);
  
  // and if we removed something from the field tower, we can no longer assume
  // that our caller has found the rational part completely:
  [int2, r, h] := intlib::algebraic::Hermite(num, den, ts, diffs, algs);
  integral := integral + int2;
  assert(iszero(r[1]));

  [num, den] := intlib::algebraic::normal(expr(h[1])/expr(h[2]), [z, y]);

  F := poly(y^n-a, [z, y]);
  if F = FAIL then
    // TODO: transformation to algebraic over poly?
    // See p.10 of Bronstein for a way to do that.
    return([0, FALSE]);
  end_if;

  d := intlib::algebraic::diff(ts, diffs, algs);
  if n = 2 and degree(F, z) mod 2 = 1 and
    iszero(degree(gcd(d(a), a), z)) then
    [int2, good] := intlib::algebraic::simpleRadicalLogPartHyperelliptic(
      poly(num, [y]), poly(den, [y]), ts, diffs, algs);
    integral := integral + int2;
    return([integral, good]);
  end_if;

/*
//  warning("not hyperelliptic");
//  print("not hyperelliptic", n, z, degree(a, z), degree(F, z), gcd(d(a), a));  
  t := solvelib::getIdent(Any, indets(expr([args()])));
  R := poly(t*diff(den, z)-num, [y]);
  R := poly(polylib::resultant(R, poly(F, [y])), [t]);
  if R = FAIL then return([0, FALSE]); end_if;
  R := poly(polylib::primpart(R), [z]);
  if R = FAIL then return([0, FALSE]); end_if;
  R := poly(polylib::resultant(R, poly(den, [z])), [t]);
  if R = FAIL then return([0, FALSE]); end_if;
  
  alpha := solve(R, t) minus {0};
  if domtype(alpha) <> DOM_SET then
    // TODO
    return([0, FALSE]);
  end_if;
  alpha := [op(alpha)];
  // TODO: find basis for Q-vector space spanned by alpha.
  // TODO: Represent roots of R as polynomials in the generators of the splitting field
  // (p. 59 of Trager)
  q := alpha;
  r := [[0$nops(alpha)]$nops(alpha)];
  for i from 1 to nops(alpha) do
    r[i][i] := 1;
  end_for;

  // construct divisors
  denp := d(den);
  B := den/gcd(den, denp);
  divisors := table(map(alpha, r -> r = [num-r*denp, B]));
  
//  print(divisors);
  // TODO: construct divisors for basis elements to be constructed above
  // see pp.61 f. of Trager for that.
  
  // since fnum/fden is reduced, the denominator is square-free
  // and the residues are exactly fnum(p)/fden'(p) for the places
  // p where fden(p)=0.
  p := solve(expr(fden));
  if domtype(p) <> DOM_SET or indets(p) minus indets([args()]) <> {} then
    // TODO
    return([0, FALSE]);
  end_if;
  
  fdenp := intlib::algebraic::diff(ts, diffs)(fden);
  residues := subsop(table(), 0=[]);
  for pt in p do
    for yofz in map([exp(2*I*PI*i/n) $ i = 1..n], `*`, (a|pt)^(1/n)) do
      pty := pt.[y=yofz];
      res := (fnum | pty)/(fdenp | pty);
      residues[res] := residues[res].[pty];
    end_for;
  end_for;
  delete residues[0];
  
  // for simple radical extensions, the ramification index is always 1
  // TODO: Check if that is really true for nested radicals, too.
  
  delta := map(r,
    ri -> select(zip(ri, alpha, DOM_LIST), l -> not iszero(l[1])));
  delta := map(delta,
    proc(l)
      local r_ij, alpha_i, places;
    begin
      [r_ij, alpha_i] := l;
      places := residues[alpha_i];
    end_proc);
  
  
  print(residues);
  
*/
  return([integral, FALSE]);
end_proc:
