// Integration in k(x)(y) where f=y^2 in k[x],
// f squarefree, degree(f) odd
//
// Given g in k(y) reduced, compute res = sum(c[i]*ln(u[i]), i=1..m)
// s.t. res'-g in k, c[i] in const(k), u[i] in k(t), or FAIL if no such res exists.
// 
// return value: [res, TRUE] if successful, [something, FALSE] else
// (for consistency with intlib::algebraic::integratePrimitive etc.)
// 
// Follows Laurent Bertrand, Computing a Hyperelliptic Integral
// using Arithmetic in the Jacobian of the Curve, AAECC 6, 275-298 (1995)

intlib::algebraic::simpleRadicalLogPartHyperelliptic :=
proc(num, den, ts, diffs, algs)
  local d, f, y, x, w, P, Q, P1, integral, g, alpha, T, A, B, t, R,
    residues, residue_basis, residue_matrix, H, i, j, G, res, D, Di,
    nH, good, Div;
begin
  assert(algs[-1][1..2] = ["simpleradical", 2]);
  [x, y] := ts[-2..-1];
  f := poly(algs[-1][3], [x]);
  
  assert(f <> FAIL);
  assert(degree(f) mod 2 = 1);
  assert(iszero(degree(gcd(f, f', "UseAlgebraicExtension"))));

  w := algs[-1][4];
  assert(op(den, 2) = [y]);
  if degree(den) > 0 then
    [num, den] := intlib::algebraic::normalizeAlgebraicFraction(num, den, ts, diffs, algs);
    G := _plus(op(zip(num, w, `*`)));
    [num, den] := intlib::algebraic::normal(expr(G)/expr(den), [y]);
  end_if;
  // and ensure that all coefficients are polynomial in x
  [num, den] := intlib::algebraic::normal(expr(num)/expr(den), [x, y]);
  [num, den] := map([num, den], poly, [y]);

  assert(degree(num) < 2);
  assert(degree(den) = 0);
  
  P := coeff(num, 1);
  P1 := coeff(num, 0);
  Q := expr(den);

  // split off the part not depending on y.
  // See Trager, p. 74 for a proof that this never breaks integrability  
  integral := 0;
  if not iszero(P1) then
    integral := intlib::algebraic::rischInt(
      op(intlib::algebraic::normal(P1/Q, [x])), ts[1..-2], diffs[1..-2], algs[1..-2]);
  end_if;
  
  d := intlib::algebraic::diff(ts, diffs, algs);
  
  // Bertrand uses P/(Q*y)
  [P, Q] := intlib::algebraic::normal(expr(f)*P/Q, [x]);
  assert(iszero(degree(gcd(Q, Q', "UseAlgebraicExtension"))));
  assert(iszero(degree(gcd(Q, f, "UseAlgebraicExtension"))));

  g := (degree(f)-1)/2;
  alpha := degree(P)-degree(Q);
  if alpha >= 2*g then
    [T, P] := [divide(P, Q)];
    while degree(T) > 2*g do
      B := poly([[2*lcoeff(T)/((2*degree(T)-2*g+1)*lcoeff(f)), degree(T)-2*g]], [x]);
      // T/y - (B*f/y)' = D/y
      // <=> D = T - B*f'/2 - f*B'
      A := B*d(f)/2 + f*d(B);
      if lmonomial(A) <> lmonomial(T) then
        // Huh? We've seen this happen in int(exp(x - ln(4*exp(x) + 1))*(- 2*exp(x) - 1/2)^(3/2), x).
        return([integral, FALSE]);
      end_if;
      T := T - A;
      integral := integral + expr(B)*y;
    end_while;
    if degree(T) = 2*g then
      B := 2*lcoeff(T)/lcoeff(d(f));
      T := T - B*d(f)/2 - f*d(B);
      integral := integral + expr(B)*y;
    end_if;
    P := P + Q*T;
    alpha := degree(P)-degree(Q);
  end_if;
  if iszero(P) then
    return([integral, TRUE]);
  end_if;
  if alpha >= g then // not elementary
    // this may be a good place some day to branch to intlib::hyperint or something similar
    // also, we might consider splitting off an elementary part if we know ow to find one.
    return([integral, FALSE]);
  end_if;
  
  // compute the residues (sec. 2.4.1)
  t := solvelib::getIdent(Any, indets(expr([args()])));
  R := polylib::resultant(poly(t*y*d(Q)-P, [y]), poly(y^2-f, [y]));
  R := polylib::subresultant(poly(R, [x]), Q);
  residues := solve(expr(R[0]), t, IgnoreSpecialCases) minus {0};
  if domtype(residues) <> DOM_SET then
    // TODO: Can we do something with RootOf?
    return([integral, FALSE]);
  end_if;
  // The following should not lose any information, since R is even:
  res := residues;
  residues := select(residues, _not@generate::isNeg);
  assert(nops(residues) = nops(res)/2);
  
  // Q-basis of residues
  // TODO: This probably needs improvement.
  // rationalize may help, and it may also help to
  // find Q-linear dependencies by subtituting subexpressions
  // with random values and calling i::a::ratlinsolve on the
  // resulting (hopefully overdetermined) system.
  residues := [op(residues)];
  residue_basis := [];
  residue_matrix := [[0 $ nops(residues)] $ nops(residues)];
  for res from 1 to nops(residues) do
    A := FAIL;
    for i from 1 to nops(residue_basis) do
      B := normal(residues[res]/residue_basis[i], Expand=FALSE);
      if domtype(B) in {DOM_RAT, DOM_INT} then
        A := TRUE;
        residue_matrix[i][res] := B;
        break;
      end_if;
    end_for;
    if A = FAIL then
      // residues[res] not yet in Q(residue_basis).
      residue_basis := residue_basis.[residues[res]];
      residue_matrix[nops(residue_basis)][res] := 1;
    end_if;
  end_for;
  // TODO: We need a Z_ basis actually.
  // Go through residue_matrix and replace all DOM_RAT by DOM_INT,
  // updating residue_basis accordingly. Also, we prefer positive entries.
  
  // divisors (sec. 2.4.2)
  // Note that residues only contains half the residues,
  // since we discarded the negative ones above.
  Div := intlib::algebraic::Divisor(ts, diffs, algs);
  H := map(residues,
    proc(res)
      local A, B;
    begin
      A := gcd(Q, P^2-res^2*d(Q)^2*f,"UseAlgebraicExtension");
      // Bertrand claims that for the minimal i s.t.
      // R[i]|t=res does not vanish, A=R[i]|t=res.
      // Seems not to work for int(-(2*a*(x^3 + 1)^(1/2) - 3*x^4)/(2*x^2 + 2*x^5 + 2*a*x*(x^3 + 1)^(1/2)), x), though.

      // B := P/r/d(Q) mod A
      assert(iszero(degree(gcd(d(Q), A))));
      B := P/res * ((g, s, t) -> s/g)(gcdex(d(Q), A));
      B := mapcoeffs(divide(B, A, Rem), intlib::Simplify);

      assert(B <> FAIL);
      Div(A, B);
    end_proc);
  
  // divisors corrsponding to the basis (Sec. 2.4.3)
  D := [NIL $ nops(residue_basis)];
  for i from 1 to nops(residue_basis) do
    Di := 0;
    for j from 1 to nops(residues) do
      if not iszero(residue_matrix[i][j]) then
        // Di := Di + a[i,j]*(2*H[j] - div(A[j]))
        nH := H[j] + H[j];
        nH := Div::multDiv(nH, 1/expr(H[j][1]));
        nH := Div::intmult(nH, residue_matrix[i][j]);
        Di := Di + nH;
      end_if;
    end_for;
    assert(domtype(Di) = Div); // at least one non-zero entry
    D[i] := Div::reduce(Di);
  end_for;

  // find orders of divisors (Sec. 2.4.4)
  good := TRUE;
  zip(residue_basis, D,
    proc(ri, Di)
      local i, H;
    begin
      if good = FALSE then return(); end_if;
      i := Div::order(Di);
      if i = infinity then
        good := FALSE;
      else
        H := Div::reduce(i*Di);
        if Div::isPrincipal(H) then
          integral := integral + ri/i*ln(H[3]);
        else
          good := FALSE;
        end_if;
      end_if;
    end_proc):

  return([integral, good]);
end_proc:
