/* helper functions for expressing elliptic integrals by special functions */


// NOTE: handling of more cases should go into intlib::elliptic::ellrec below


/* intlib::elliptic(f, x, options)

   computes int(f, x) where  f, x, options are the same
   as in intlib::int, i.e.

   f --  arithmetical expression
         only expressions of the form
         rational(x, sqrt(poly(x)), sqrt(poly(x)), ... )
         can be expected to give useful results here, others will cause
         intlib::elliptic to return FAIL

   x -- identifier

   options -- table of options passed through from intlib::int

Returns: FAIL if no indefinite integral of f is found
         otherwise: the indefinite integral of f w.r.t. x
                    as an arithmetical expression
                    (just as intlib::int would return)

Possible extensions: 
      handle more special cases (i.e. which sqrt(poly(x)) can be handled)
      these cases are handled in intlib::elliptic::ellrec
*/

intlib::elliptic:=proc(f,x,options)
local powops,i,j,polys,result;
begin
  // get rid of the assumption x in R_
  save x;
  eval(hold(unprotect)(x));
  eval(hold(_delete)(x));

  powops:=select([prog::find(f,"_power",Type)], o->not testtype(op(op(f,o),2), DOM_INT));
  if nops(powops)=0 then
    return(FAIL);
  end_if;
  
  polys:=[];
 
  // replace all square roots by `#Y`(j)
  for i in powops do
    if not testtype(op(op(f,i),2), DOM_RAT) then
      return(FAIL);
    end_if;

    if not testtype(op(op(f,i),2)*2, DOM_INT) then
      return(FAIL);
    end_if;

    if not testtype(op(op(f,i),1), Type::PolyExpr([x], Type::IndepOf({x, `#Y`}))) then
      return(FAIL);
    end_if;

    j:=contains(polys, op(op(f,i),1));
    if j=0 then
      polys:=polys . [op(op(f,i),1)];
      j:=nops(polys);
    end_if;
    
    f:=subsop(f, i.[1]=`#Y`(j), i.[2]=op(op(f,i),2)*2, Unsimplified);
  end_for;

  f:=eval(f);

  // make sure that f is rational
  if not intlib::elliptic::testrat(f,x) then
    return(FAIL);
  end_if;

  // now recursively split f up into different normal forms of elliptic integrals
  result:=intlib::elliptic::ellrec(f,x,polys,[],options);

  // see if we actually did some useful work
  if type(result)="int" then
    return(FAIL);
  end_if;

  if _lazy_and(type(result)="_plus", map(op(result), o->type(o)="int")) then
    return(FAIL);
  end_if;

  return(result);
end_proc:


intlib::elliptic:=funcenv(intlib::elliptic):


//----------------------------------------------

/* intlib::elliptic::testrat(f,x)

   test whether f is a rational expression in x and `#Y`(j), j=1,2,3,...

    f -- arithmentical expression
    x -- identifier
  
   Returns:
    TRUE if f is a rational expression in x and `#Y`(j), j=1,2,3...
    FALSE otherwise
*/
intlib::elliptic::testrat:=proc(f,x)
begin
  case type(f)
    of "_plus" do
    of "_mult" do
      return(_lazy_and(map(op(f), intlib::elliptic::testrat, x)));
    of "_power" do
      if not testtype(op(f,2), DOM_INT) and has(op(f,1),{x,`#Y`}) then
        return(FALSE);
      end_if;
      return(intlib::elliptic::testrat(op(f,1),x));
    of "function" do
      return(bool(op(f,0)=`#Y`) or not has(f,{x,`#Y`}));
    otherwise
      return(bool(f=x) or not has(f,{x,`#Y`}));
  end_case;
end_proc:

//----------------------------------------------
/*  intlib::elliptic::ellrec(f,x,splitpolys,radpolys,options)

  recursively split f into normalized forms (i.e. R(x)/sqrt(poly(x))/sqrt(poly(x))/...)
  and compute indefinite integral
  
    f -- arithmetical expression
    x -- identifier
    splitpolys,radpolys -- lists of polynomials in x
    options -- table of options passed through from intlib::int

  Returns:
    arithmetical expression

  Description:
    Each `#Y`(j) in f corresponds to sqrt(splitpolys[j]). ellrec will recursively get rid of these
    `#Y`(j) in f, by splitting up f into f1+f2/sqrt(splitpolys[j]), such that f1 and f2 do not contain
    `#Y`(j). Then f1 and f2 are handled recursively -- splitpolys[j] is removed from splitpolys and added
    to radpolys for f2, but not for f1. For example:

      intlib::elliptic::ellrec(`#Y`(1), x, [1-x^2], [], table())
    
    means that the rational function sqrt(1-x^2) (substitute `#Y`(j) by sqrt(splitpolys[1])) is to be
    expressed in normal form f1+f2/sqrt(1-x^2) with f1,f2 rational in x (there are no more `#Y`(j) terms).
    Obviously, it follows that f1=0 and f2=1-x^2. Now ellrec is applied recursively to f1 and f2:

    intlib::elliptic::ellrec(0, x, [], [], table())
    intlib::elliptic::ellrec(1-x^2, x, [], [1-x^2], table())

    Now that splitpolys is empty, there is no more to split up and actual integration can take place.  

  Possible extensions:
    This is where check for different cases can be applied. Currently supported are integrands of the forms:
      R(x)/sqrt(a+b*x^2)
      R(x)/sqrt(1-x^2)/sqrt(a+b*x)
*/
intlib::elliptic::ellrec:=proc(f,x,splitpolys,radpolys,options)
local tmp,result,a,b,c,p,t;
begin
  if iszero(f) then
    return(0);  // int(0,x)=0
  end_if;

  // as long as splitpolys is not empty, split up f into a rational expression which does not depend on
  // sqrt(splitpolys[-1]) and one which does depend on sqrt(splitpolys[-1]).
  if nops(splitpolys)>0 then
    t := solvelib::getIdent(Any,indets({args()}, All));
    tmp:=intlib::elliptic::splitrat(subs(f,`#Y`(nops(splitpolys))=t), t, splitpolys[-1]);
    if has(tmp, t) then
      return(hold(int)(f/_mult(op(map(radpolys,sqrt))), x, intlib::printOptions(options)));
    end_if;
    
    return(
      intlib::elliptic::ellrec(tmp[1],x,splitpolys[1..-2],radpolys.[splitpolys[-1]],options) +
      intlib::elliptic::ellrec(tmp[2],x,splitpolys[1..-2],radpolys,options));
  end_if;

  // Now f is a rational function in x.
  // Objective: integrate f/sqrt(radpolys[1])/sqrt(radpolys[2])...

  // try partial fraction decomposition
  if degree(denom(f))>1 then
    result:=intlib::elliptic::partialfraction(numer(f), denom(f), x, radpolys, options);
    if result<>FAIL then
      return(result);
    end_if;
  end_if;

  if nops(radpolys)=0 then
    // pure rational function
    return(intlib::algebraic::rational(f,x,options));
  end_if;

  radpolys:=sort(radpolys, (p,q)->degree(p,[x])>degree(q,[x]));


  if iszero(1-x^2-radpolys[1]) then
    if nops(radpolys)=2 and degree(radpolys[2],x)=1 then
      a:=coeff(radpolys[2],x,1);
      b:=coeff(radpolys[2],x,0);
      
      // FIXME: case b<0 is wrong
      if iszero(a-b) then
        return(hold(int)(f/(1+x)/sqrt(1-x)/sqrt(b),x));
      elif iszero(a+b) then
        return(hold(int)(f/(1-x)/sqrt(1+x)/sqrt(b),x));
      end_if;
      
      return(intlib::elliptic::ell3c(f,x,a,b));
    end_if;
  end_if;

  if nops(radpolys)=1 and degree(radpolys[1],x)=2 and coeff(radpolys[1],x,1)=0 then
    // special case: circular integral
    a:=coeff(radpolys[1],x,2);
    b:=coeff(radpolys[1],x,0);

    if is(b>0) = TRUE then
      if is(a<0)=TRUE then
        return(intlib::circular(f,x,-b/a)/sqrt(-a));  // handle this specially for cosmetic reasons (keep things real if possible)
      elif is(a>0)=TRUE then
        return(intlib::hyperbolic(f,x,b/a)/sqrt(a));
      else
        return(subs(intlib::hyperbolic(subs(f, x=x/sqrt(a)),x,b), x=x*sqrt(a)) / sqrt(a));  // integration by substitution
      end_if;
    end_if;
  end_if;

  if nops(radpolys)=1 and degree(radpolys[1],x)=3 then
    // R/sqrt((x+a)*(x+b)*(x+c))
    p := radpolys[1]/lcoeff(radpolys[1], [x]);
    a := solve(p, x, MaxDegree=3);
    if domtype(a) = DOM_SET then
      case nops(a)
      of 1 do
      of 2 do
        break;
      of 3 do
        [a,b,c] := [op(a)];
        return(intlib::elliptic::cubic(f,
          x, p, -a, -b, -c, options) * sqrt(p)/sqrt(radpolys[1]));
      end_case;
    end_if;
  end_if;

  return(hold(int)(f/_mult(op(map(radpolys,sqrt))), x, intlib::printOptions(options)));
end_proc:


//----------------------------------------------
/* intlib::elliptic::partialfraction(n,d,x,radpolys,options)

  Try to integrate n/d/sqrt(radpolys[1])/sqrt(radpolys[2])/... by doing a partial fraction
  decomposition of n/d.

  n,d -- polynomial expressions in x
  x -- identifier
  radpolys -- list of polynomials in x
  options -- table of options passed through from intlib::int

  Returns:
    indefinite integral of n/d/sqrt(radpolys[1])/... or FAIL
*/
intlib::elliptic::partialfraction:=proc(n,d,x,radpolys,options)
local q,r,tmp,tmp2,result;
begin
  // cannot handle multiple roots yet
  if degree(gcd(d, diff(d,x)))>0 then
    return(FAIL);
  end_if;

  [q,r]:=[divide(n,d,[x])];

  tmp:=genident(expr2text(x));
  result:=intlib::elliptic::ellrec(1/(x-tmp), x, [], radpolys, options);
  if result=FAIL or type(result)="int" then
    return(FAIL);
  end_if;
 
  tmp2:=solve(d=0, x, if options[IgnoreAnalyticConstraints]=TRUE then IgnoreAnalyticConstraints end_if);
  if has(tmp2, RootOf) then
    tmp2:=genident(expr2text(x));
    result:=sum(result*subs(r/diff(d,x), x=tmp), tmp=RootOf(subs(d,x=tmp2),tmp2));
  else
    result:=sum(result*subs(r/diff(d,x), x=tmp), tmp in tmp2);
  end_if;

  if not iszero(q) then
    result:=result + intlib::elliptic::ellrec(q, x, [], radpolys, options);
  end_if;

  return(result);
end_proc;


//----------------------------------------------
/* intlib::elliptic::splitrat(r,X,X2)
  
  Helper function for intlib::elliptic::ellrec:
  split a rational function R(x,X) with X^2=X2 into R1(x)/X+R2(x).

  r -- arithmetical expression
  X,X2 -- polynomials

  Returns:
    List of arithmetical expressions: [R1, R2]
*/
  
intlib::elliptic::splitrat:=proc(r,X,X2)
local n,d,red;
begin
  // reduction polynomial
  red:=poly(X^2-X2, [X]);
  
  r:=normal(r);
  n:=divide(poly(numer(r), [X]), red, Rem);
  d:=divide(poly(denom(r), [X]), red, Rem);

  n:=divide(n*poly(coeff(d,1)*X-coeff(d,0), [X]), red, Rem);
  d:=coeff(d,1)^2*X2-coeff(d,0)^2;

  return([normal(coeff(n,1)*X2/d), normal(coeff(n,0)/d)]);
end_proc:


//----------------------------------------------
/* intlib::elliptic::ell3(c,z,m)

  Computes
    int(sum(c[k]*z^(k-1), k=1..nops(c)) / sqrt(z) / sqrt(1-z) / sqrt(1-m*z), z)

  c -- list of arithmetical expressions indep. of z
  z -- identifier
  m -- arithmetical expression indep. of z
*/
intlib::elliptic::ell3:=proc(c,z,m)
local k,tmp;
begin
  k:=nops(c);
  // padding
  if k=1 then
    c:=c . [0];
  end_if;

  tmp:=0;

  // reduce case k>2 to k=2
  // G&H 243.4a
  while k>2 do
    tmp:=tmp + c[k]*2/(2*k-3)/m*z^(k-3);
    c[k-1]:=c[k-1] + c[k]*(2*k-4)/(2*k-3)*(m+1)/m;
    c[k-2]:=c[k-2] - c[k]*(2*k-5)/(2*k-3)/m;
    k:=k-1;
  end_while;

  tmp:=expand(tmp) * sqrt(z) * sqrt(1-z) * sqrt(1-m*z);
  z:=arcsin(sqrt(z));
  return(tmp + 2*(c[1]+c[2]/m)*ellipticF(z, m) - 2*c[2]/m*ellipticE(z, m));
end_proc;


//----------------------------------------------
/* intlib::elliptic::ell3pi(c,z,m,n,k)

  Computes
    int(sum(c[j]*z^(j-1), j=1..nops(c)) / sqrt(z) / sqrt(1-z) / sqrt(1-m*z) / (1-n*z)^k, z)

  c -- list of arithmetical expressions indep. of z
  z -- identifier
  m,n -- arithmetical expressions indep. of z
  k -- positive integer
*/
intlib::elliptic::ell3pi:=proc(c,z,m,n,k)
local i,j,tmp,result;
begin
  // TODO: special cases
  if iszero(n-1) or iszero(n-m) then
    return(FAIL);
  end_if;

  // padding
  while nops(c)<k+4 do
    c:=c.[0];
  end_while;
  
  // first switch from base {_X_^j} to base {(1-n*_X_)^j}
  i:=nops(c);
  while i>1 do
    tmp:=c[i] / (-n)^(i-1);
    
    c[i]:=tmp;
    for j from 1 to i-1 do
      c[j]:=c[j] - tmp*binomial(i-1,j-1)*(-n)^(j-1);
    end_for;
    
    i:=i-1;
  end_while;

  // reduce case k>1 to k=1
  // G&H 241.9a
  result:=0;
  
  while k>1 do
    tmp:=c[1]/2/(k-1)/(1-n)/(m-n);

    result:=result + tmp/(1-n*z)^(k-1);
    c[2]:=c[2] + tmp*(2*k-3)*(n^2-2*m*n-2*n+3*m);
    c[3]:=c[3] - tmp*(2*k-4)*(-m*n-n+3*m);
    c[4]:=c[4] + tmp*(2*k-5)*m;
    
    c:=c[2..nops(c)];
    k:=k-1;
  end_while;

  // now switch back to base {_X_^j}
  for i from k+1 to nops(c) do
    tmp:=c[i];

    c[i]:=tmp*(-n)^(i-k-1);
    for j from k+1 to i-1 do
      c[j]:=c[j] + tmp*binomial(i-k-1,j-k-1)*(-n)^(j-k-1);
    end_for;
  end_for;

  return(
    result*2*n^2*sqrt(z)*sqrt(1-z)*sqrt(1-m*z) + 
    c[1]*2*ellipticPi(n,arcsin(sqrt(z)),m) + 
    intlib::elliptic::ell3(c[2..nops(c)],z,m));
end_proc;


//----------------------------------------------
/* intlib::elliptic::ell3a(p,x,a,b)

  Computes
    int(p/sqrt(a*x+b)/sqrt(1-x^2), x), abs(a)<b

  p -- polynomial expression in x
  x -- identifier
  a,b -- arithmetical expressions indep. of x
*/
intlib::elliptic::ell3a:=
proc(p,x,a,b)
  local j: DOM_INT;
begin
  p:=poly(subs(p, x=2*x-1), [x]);
  p:=[coeff(p,j) $ j=0..degree(p)];
  
  return(intlib::elliptic::ell3(p, (x+1)/2, 2*a/(a-b)) / sqrt(b-a));
end_proc:


//----------------------------------------------
/* intlib::elliptic::ell3b(p,x,a,b,c,d,k)

  Computes
    int(p/sqrt(a*x+b)/(c*x+d)^k/sqrt(1-x^2), x), abs(a)<b

  p -- polynomial expression in x
  x -- identifier
  a,b,c,d -- arithmetical expressions indep. of x
  k -- positive integer
*/
intlib::elliptic::ell3b:=
proc(p,x,a,b,c,d,k)
  local j: DOM_INT;
begin
  p:=poly(subs(p, x=2*x-1), [x]);
  p:=[coeff(p,j) $ j=0..degree(p)];
  
  return(intlib::elliptic::ell3pi(p, (x+1)/2, 2*a/(a-b), 2*c/(c-d), k) / (d-c)^k / sqrt(b-a));
end_proc:


//----------------------------------------------
/* intlib::elliptic::ell3c(r,x,a,b)

  Computes
    int(r/sqrt(a*x+b)/sqrt(1-x^2), x), abs(a)<b

  r -- rational expression in x
  x -- identifier
  a,b -- arithmetical expressions indep. of x
*/    
intlib::elliptic::ell3c:=
proc(r,x,a,b)
  local result,p,s,t,k;
begin
  // the most likely reason for ell3c to fail is that partfrac
  // cannot decompose denom(r) into linear factors
  r:=partfrac(r,x,Full);
  
  if type(r)="_plus" then
    p:=select(r, testtype, Type::PolyExpr([x], Type::IndepOf(x)));
    r:=select([op(r)], not testtype, Type::PolyExpr([x], Type::IndepOf(x)));
  elif testtype(r, Type::PolyExpr([x], Type::IndepOf(x))) then
    p:=r;
    r:=[];
  else
    p:=0;
    r:=[r];
  end_if;

  // all of the following could be done a lot more efficient

  result:=intlib::elliptic::ell3a(p,x,a,b);

  for s in r do
    t:=denom(s);
    if type(t)="_power" then
      k:=op(t,2);
      t:=op(t,1);

      assert(k>1);
    else
      k:=1;
    end_if;

    assert(has(t,x));

    if degree(t,x)>1 then
      return(FAIL);
    end_if;

    result:=result + intlib::elliptic::ell3b(numer(s),x,a,b,coeff(t,x,1),coeff(t,x,0),k);
  end_for;

  return(result);
end_proc:
//----------------------------------------------

// int(R(x)/sqrt((x+a)*(x+b)*(x+c)), x)
// f = R(x), y=(x+a)*(x+b)*(x+c)
intlib::elliptic::cubic :=
proc(f, x, y, a, b, c, options)
  local num, den, d, e, n, A, B, g, res;
begin
  num := numer(f);
  den := denom(f);
  
  // normal forms of integrals
  case [degree(num), degree(den)]
  of [0, 0] do
    // int(1/sqrt((x+a)*(x+b)*(x+c)), x)
    return(-2*num/den*ellipticF(arcsin((-(c + x)/(a - c))^(1/2)), (a - c)/(b - c))
      *((a + x)/(a - c))^(1/2)*((b + x)/(b - c))^(1/2)
      *(-(c + x)/(a - c))^(1/2)*(a - c)
      /(x^3 + (a + b + c)*x^2 + (a*b + a*c + b*c)*x + a*b*c)^(1/2));
  of [1, 0] do
    // int(x/sqrt((x+a)*(x+b)*(x+c)), x)
    e := coeff(num, [x], 1)/den;
    d := coeff(num, [x], 0)/den;
    if not iszero(d) then
      d := intlib::elliptic::cubic(d, x, y, a, b, c, options);
    end_if;
    return(d + 2*e*(c-a)*((x+c)/(c-a))^(1/2)*((x+b)/(b-c))^(1/2)*((x+a)/(a-c))^(1/2)
        *((-c+b)*ellipticE(arcsin(((x+c)/(c-a))^(1/2)),(-c+a)/(-c+b))
          -b*ellipticF(arcsin(((x+c)/(c-a))^(1/2)),(-c+a)/(-c+b)))
        /(x^3 + (a + b + c)*x^2 + (a*b + a*c + b*c)*x + a*b*c)^(1/2));
  of [0, 1] do
    // int(1/((x+d)*sqrt((x+a)*(x+b)*(x+c))), x)
    e := coeff(den, [x], 1);
    d := coeff(den, [x], 0)/e;
    if c=d then
      [a,b,c] := [b,c,a];
    end_if;
    return(num/e*(2*((a + x)/(a - c))^(1/2)*((b + x)/(b - c))^(1/2)
      *(-(c + x)/(a - c))^(1/2)*(a - c)
      *ellipticPi(-(a - c)/(c - d), arcsin((-(c + x)/(a - c))^(1/2)), (a - c)/(b - c)))
      /((c - d)*(x^3 + (a + b + c)*x^2 + (a*b + a*c + b*c)*x + a*b*c)^(1/2)));
  end_case;
  
  // recursion formula, cf. GH 243.4-7
  A := [coeff(y, [x], 3), coeff(y, [x], 2)/3, coeff(y, [x], 1)/3, coeff(y, [x], 0)];
  if degree(den) = 0 then
    res := 0;
    num := num/den;
    for g in poly2list(num, [x]) do
      [d, n] := g;
      if n < 2 then
        res := res + d*intlib::elliptic::cubic(x^n, x, y, a, b, c, options);
      elif n = 2 then
        // GH 243.6a
        res := res + d/A[1]*(
          2/3*sqrt(y) 
          -2*A[2]*intlib::elliptic::cubic(
            x, x, y, a, b, c, options)
          -A[3]*intlib::elliptic::cubic(
            1, x, y, a, b, c, options)
        );
      elif n = 3 then
        // GH 243.6b
        res := res + d/(5*A[1]^2)*(
          2*(A[1]*x-4*A[2])*sqrt(y)
          +3*(8*A[2]^2-3*A[1]*A[3])*intlib::elliptic::cubic(
            x, x, y, a, b, c, options)
          +2*(6*A[2]*A[3]-A[1]*A[4])*intlib::elliptic::cubic(
            1, x, y, a, b, c, options)
        );
      else
        res := res + d/((2*n-1)*A[1])*(
          // GH 243.4a
          2*x^(n-2)*sqrt(y)
          -(2*n-2)*3*A[2]*intlib::elliptic::cubic(
            x^(n-1), x, y, a, b, c, options)
          -(2*n-3)*3*A[3]*intlib::elliptic::cubic(
            x^(n-2), x, y, a, b, c, options)
          -(2*n-4)*A[4]*intlib::elliptic::cubic(
            x^(n-3), x, y, a, b, c, options)
        );
      end_if;
    end_for;
    return(res);
  end_if;
  
  if degree(num) = 0 then
    assert(degree(den, [x])>1); // treated above
    if type(den) = "_power" and degree(op(den, 1), [x]) = 1 then
      n := op(den, 2);
      g := op(den, 1);
      res := 0;
      d := lcoeff(g)^n/num;
      if iszero(coeff(g, [x], 0)) then
        if iszero(A[4]) then
          if not iszero(A[3]) then
            // GH 243.4c
            return((
              -2*sqrt(y)
              -(2*n-2)*3*A[2]*intlib::elliptic::cubic(
                x^(1-n), x, y, a, b, c, options)
              -(2*n-3)*A[1]*intlib::elliptic::cubic(
                x^(2-n), x, y, a, b, c, options)
            )/(d*(2*n-1)*3*A[3]));
          else
            return(FAIL);
          end_if;
        else
          // GH 243.4b
          return((
            (-2*sqrt(y)/x^(n-1)
            -(2*n-3)*3*A[3]*intlib::elliptic::cubic(
              x^(1-n), x, y, a, b, c, options)
            -(2*n-4)*3*A[2]*intlib::elliptic::cubic(
              x^(2-n), x, y, a, b, c, options)
            -(2*n-5)*A[1]*intlib::elliptic::cubic(
              x^(3-n), x, y, a, b, c, options)
            ))/(d*(2*n-2)*A[4]));
        end_if;
      else // not x^(-n), but (x-e)^(-n)
        e := -coeff(g, [x], 0)/lcoeff(g);
        B := [A[1], e*A[1]+A[2], e^2*A[1]+2*e*A[2]+A[3],
          e^3*A[1]+3*e^2*A[2]+3*e*A[3]+A[4]];
        if iszero(B[4]) then
          // GH 243.5b
          // implies e in {a,b,c}
          return((
            -2*sqrt(y)/g^n
            -3*(2*n-2)*B[2]*intlib::elliptic::cubic(
              g^(1-n), x, y, a, b, c, options)
            -(2*n-3)*B[1]*intlib::elliptic::cubic(
              g^(2-n), x, y, a, b, c, options)
          )/(d*3*(2*n-1)*B[3]));
        else
          // GH 243.5a
          return((
            -2*sqrt(y)/g^(n-1)
            -3*(2*n-3)*B[3]*intlib::elliptic::cubic(
              g^(1-n), x, y, a, b, c, options)
            -3*(2*n-4)*B[2]*intlib::elliptic::cubic(
              g^(2-n), x, y, a, b, c, options)
            -(2*n-5)*B[1]*intlib::elliptic::cubic(
              g^(3-n), x, y, a, b, c, options)
          )/(2*d*(n-1)*B[4]));
        end_if;
      end_if;
      return(res);
    end_if;
  end_if;
  
  // nothing of the above? try partial fraction decompositin of f
  g := partfrac(f, x, Full);
  if hastype(g, "sum") then
    // TODO: The above should work here, too, right?
    return(FAIL);
  end_if;
  if type(g) = "_plus" then
    g := map([op(g)], intlib::elliptic::cubic, x, y, a, b, c, options);
    if not has(g, FAIL) then
      return(_plus(op(g)));
    end_if;
  end_if;
  
  /*
  // special case of partial fraction, cf. GH 243.2
  t := solvelib::getIdent(Any, indets([args()]));
  n := [0, 0, 0];
  g := coerce(polylib::sqrfree(den), DOM_LIST);
  for i from 2 to nops(g) step 2 do
    for j from 1 to 3 do
      if testeq(g[i]|x=-([a,b,c][j]), 0, Goal=TRUE)=TRUE then
        assert(iszero(n[j]));
        n[j] := g[i+1];
        g[i] := divide(g[i], x+([a,b,c][j]), Quo); // exact
      end_if;
    end_for;
  end_for;
  g1 := _mult(g[i] $ i=2..nops(g) step 2);
  g2 := divide(den, g1, Quo); // exact
  num := num/g[1];
  g := coerce(factor(_mult(g[i]^g[i+1] $ i=2..nops(g) step 2), Full), DOM_LIST);
  // degree(h) <= max(degree(num)-1, degree(den))-card(linear factors)-1
  r := (nops(g)-1)/2;
  hdeg := max(degree(num)-1, degree(den))-r-1;
  if hdeg < 0 then
    h := 0;
  else
    h := poly([[t[i], i] $ i=0..hdeg], [x]);
  end_if;
  A := t[-1];
  B := t[-2];
  D := solvelib::getIdent(Any, indets([args(), t]));
  ansatz :=
  expr(h')*g1*y+3/2*expr(h)*g1*(x^2+2/3*(a+b+c)*x+(a*b+a*c+b*c)/3)
  -expr(h)*g1*y*(n[1]/(x+a)+n[2]/(x+b)+n[3]/(x+c)+
    _plus((g[i+1]-1)/g[i] $ i=2..nops(g) step 2))
  +den*(A+B*x+_plus(D[i]/g[2*i] $ i=1..r))
  -f;
  ansatz := numer(expr(ansatz));
  ansatz := [coeff(poly(ansatz, [x]))];
  ansatz := intlib::algebraic::linsolve(ansatz,
    [t[-2], t[-1], t[i] $ i=0..hdeg, D[i] $ i=1..r]);
  if ansatz <> FAIL then
    [A, B, h] := subs([A, B, h], ansatz);
    D := subs([D[i] $ i=1..r], ansatz);
    res := expr(h)/g2*sqrt(y);
    if not iszero(A) then
      res := res + A*intlib::elliptic::cubic(1, x, y, a, b, c, options);
    end_if;
    if not iszero(B) then
      res := res + B*intlib::elliptic::cubic(x, x, y, a, b, c, options);
    end_if;
    for i from 1 to r do
      if not iszero(D[i]) then
        res := res + D[i]*intlib::elliptic::cubic(1/g[2*i], x, y, a, b, c, options);
      end_if;
    end_for;
    return(res);
  end_if;
  */
  
  FAIL;
end_proc:

// because intlib::elliptic::cubic calls itself recursively
// in the form f(n) = a*f(n-1)+b*f(n-2)+c*f(n-3) all the time:
intlib::elliptic::cubic := prog::remember(intlib::elliptic::cubic):
// NB: prog::remember is less memory-hungry than option remember.

