// handle hyperbolic integrals as degenerate cases of elliptic integrals

/* intlib::hyperbolic(f,x,a)

  Computes
    int(f/sqrt(a+x^2),x) with f rational, a>0

  f -- rational expression in x
  x -- identifier
  a -- arithmetical expression indep of x
*/
intlib::hyperbolic:=
proc(f,x,a)
  local p,q,s,t,k,b,c,fac,result,tmp;
begin
  q:=partfrac(f, x, Full);
  if type(q)="_plus" then
    p:=select(q, testtype, Type::PolyExpr([x], Type::IndepOf(x)));
    q:=select([op(q)], not testtype, Type::PolyExpr([x], Type::IndepOf(x)));
  elif testtype(q, Type::PolyExpr([x], Type::IndepOf(x))) then
    p:=q;
    q:=[];
  else
    p:=0;
    q:=[q];
  end_if;

  result:=0;

  for s in q do
    t:=denom(s);
    if type(t)="_mult" then
      fac:=select(t, not has, x);
      t:=select(t, has, x);
    else
      fac:=1;
    end_if;
    
    if type(t)="_power" then
      k:=op(t,2);
      t:=op(t,1);
    else
      k:=1;
    end_if;

    if degree(t,x)<>1 then
      return(FAIL);
    end_if;
    
    c:=coeff(t,x,0);
    b:=coeff(t,x,1);

    tmp:=intlib::hyperbolic::rat(numer(s)/fac, x, a, -c/b, k);
    result:=result + tmp[1]/b^k;
    p:=p + tmp[2]/b^k;
  end_for;
  
  return(result + intlib::hyperbolic::pol(p,x,a));
end_proc:


intlib::hyperbolic:=funcenv(intlib::hyperbolic):


/* intlib::hyperbolic::pol(f,x,a)

  Computes
    int(f/sqrt(a+x^2),x) with f polynomial, a>0

  f -- polynomial expression in x
  x -- identifier
  a -- arithmetical expression indep. of x
*/
intlib::hyperbolic::pol:=proc(f,x,a)
local p,j,result,tmp;
begin
  p:=poly(f,[x]);
  p:=[coeff(p,All)];

  j:=nops(p);
  if j=1 then
    p:=p . [0];
  end_if;

  result:=0;

  while j>2 do
    tmp:=p[j]/(j-1);
    result:=result + tmp*x^(j-2);
    p[j-2]:=p[j-2] - tmp*(j-2)*a;
    
    j:=j-1;
  end_while;

  result:=result+p[2];

  return(result*sqrt(a+x^2) + p[1]*arcsinh(x/sqrt(a)));
end_proc:

/* intlib::hyperbolic::rat(f,x,a,b,k)

  Computes
    int(f/(x-b)^k/sqrt(a+x^2) with f polynomial, a>0, k>0

  f -- polynomial expression in x
  x -- identifier
  a,b -- arithmetical expressions indep. of x
  k -- positive integer

  Returns:
    a list ret containing two arithmetical expressions such that
    ret[1]+int(ret[2]/sqrt(a+x^2),x) (ret[2] a polynomial) is the integral
*/
intlib::hyperbolic::rat:=
proc(f,x,a,b,k)
  local p,result,tmp, j: DOM_INT;
begin
  // switch from base {x^j} to base {(x-b)^j}
  p:=poly(subs(f, x=x+b), [x]);
  p:=[coeff(p,All)];

  result:=0;

  // reduce case k>1 to k=1
  // G&H 234.3a
  if iszero(a+b^2) then
    // this can only happen for complex b
    while k>0 do
      if nops(p)<2 then
        p:=p . [0];
      end_if;

      tmp:=p[1]/(2*k-1)/b;
      result:=result - tmp/(x-b)^k;
      p[2]:=p[2] - tmp*(k-1);
      
      p:=p[2..-1];
      k:=k-1;
    end_while;

    return([result*sqrt(a+x^2), _plus(p[j]*(x-b)^(j-1) $ j=1..nops(p))]);
  else
    while k>1 do
      while nops(p)<3 do
        p:=p . [0];
      end_while;

      tmp:=p[1]/(k-1)/(a+b^2);
      result:=result - tmp/(x-b)^(k-1);
      p[2]:=p[2] - tmp*(2*k-3)*b;
      p[3]:=p[3] - tmp*(k-2);
  
      p:=p[2..-1];
      k:=k-1;
    end_while;

    if iszero(b) then
      result:=result*sqrt(a+x^2) - p[1]*arcsinh(sqrt(a/x^2))/sqrt(a);
    else
      result:=result*sqrt(a+x^2) - p[1]*ln((b*x+a+sqrt((a+b^2)*(a+x^2)))/(x-b))/sqrt(a+b^2);
    end_if;
 
    return([result, _plus(p[j]*(x-b)^(j-2) $ j=2..nops(p))]);
  end_if;
end_proc:


